/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.nio;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.SocketChannel;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Predicate;
import org.elasticsearch.common.concurrent.CompletableContext;
import org.elasticsearch.nio.ChannelContext;
import org.elasticsearch.nio.FlushOperation;
import org.elasticsearch.nio.InboundChannelBuffer;
import org.elasticsearch.nio.NioSelector;
import org.elasticsearch.nio.NioSocketChannel;
import org.elasticsearch.nio.ReadWriteHandler;
import org.elasticsearch.nio.WriteOperation;
import org.elasticsearch.nio.utils.ExceptionsHelper;

public abstract class SocketChannelContext
extends ChannelContext<SocketChannel> {
    protected static final Predicate<NioSocketChannel> ALWAYS_ALLOW_CHANNEL = c -> true;
    protected final NioSocketChannel channel;
    protected final InboundChannelBuffer channelBuffer;
    protected final AtomicBoolean isClosing = new AtomicBoolean(false);
    private final ReadWriteHandler readWriteHandler;
    private final Predicate<NioSocketChannel> allowChannelPredicate;
    private final NioSelector selector;
    private final CompletableContext<Void> connectContext = new CompletableContext();
    private final LinkedList<FlushOperation> pendingFlushes = new LinkedList();
    private boolean closeNow;
    private Exception connectException;
    private final int WRITE_LIMIT = 65536;

    protected SocketChannelContext(NioSocketChannel channel, NioSelector selector, Consumer<Exception> exceptionHandler, ReadWriteHandler readWriteHandler, InboundChannelBuffer channelBuffer, Predicate<NioSocketChannel> allowChannelPredicate) {
        super(channel.getRawChannel(), exceptionHandler);
        this.selector = selector;
        this.channel = channel;
        this.readWriteHandler = readWriteHandler;
        this.channelBuffer = channelBuffer;
        this.allowChannelPredicate = allowChannelPredicate;
    }

    @Override
    public NioSelector getSelector() {
        return this.selector;
    }

    @Override
    public NioSocketChannel getChannel() {
        return this.channel;
    }

    public void addConnectListener(BiConsumer<Void, Exception> listener) {
        this.connectContext.addListener(listener);
    }

    public boolean isConnectComplete() {
        return this.connectContext.isDone() && !this.connectContext.isCompletedExceptionally();
    }

    public boolean connect() throws IOException {
        if (this.isConnectComplete()) {
            return true;
        }
        if (this.connectContext.isCompletedExceptionally()) {
            Exception exception = this.connectException;
            if (exception == null) {
                throw new AssertionError((Object)"Should have received connection exception");
            }
            if (exception instanceof IOException) {
                throw (IOException)exception;
            }
            throw (RuntimeException)exception;
        }
        boolean isConnected = ((SocketChannel)this.rawChannel).isConnected();
        if (!isConnected) {
            try {
                isConnected = ((SocketChannel)this.rawChannel).finishConnect();
            }
            catch (IOException | RuntimeException e) {
                this.connectException = e;
                this.connectContext.completeExceptionally(e);
                throw e;
            }
        }
        if (isConnected) {
            this.connectContext.complete(null);
        }
        return isConnected;
    }

    public void sendMessage(Object message, BiConsumer<Void, Exception> listener) {
        if (this.isClosing.get()) {
            listener.accept(null, new ClosedChannelException());
            return;
        }
        WriteOperation writeOperation = this.readWriteHandler.createWriteOperation(this, message, listener);
        NioSelector selector = this.getSelector();
        if (!selector.isOnCurrentThread()) {
            selector.queueWrite(writeOperation);
            return;
        }
        selector.writeToChannel(writeOperation);
    }

    public void queueWriteOperation(WriteOperation writeOperation) {
        this.getSelector().assertOnSelectorThread();
        this.pendingFlushes.addAll(this.readWriteHandler.writeToBytes(writeOperation));
    }

    public abstract int read() throws IOException;

    public abstract void flushChannel() throws IOException;

    protected void currentFlushOperationFailed(IOException e) {
        FlushOperation flushOperation = this.pendingFlushes.pollFirst();
        this.getSelector().executeFailedListener(flushOperation.getListener(), e);
    }

    protected void currentFlushOperationComplete() {
        FlushOperation flushOperation = this.pendingFlushes.pollFirst();
        this.getSelector().executeListener(flushOperation.getListener(), null);
    }

    protected FlushOperation getPendingFlush() {
        return this.pendingFlushes.peekFirst();
    }

    @Override
    protected void register() throws IOException {
        super.register();
        if (!this.allowChannelPredicate.test(this.channel)) {
            this.closeNow = true;
        }
    }

    @Override
    public void closeFromSelector() throws IOException {
        this.getSelector().assertOnSelectorThread();
        if (this.isOpen()) {
            FlushOperation flushOperation;
            ArrayList<IOException> closingExceptions = new ArrayList<IOException>(3);
            try {
                super.closeFromSelector();
            }
            catch (IOException e) {
                closingExceptions.add(e);
            }
            this.isClosing.set(true);
            this.pendingFlushes.addAll(this.readWriteHandler.pollFlushOperations());
            while ((flushOperation = this.pendingFlushes.pollFirst()) != null) {
                this.selector.executeFailedListener(flushOperation.getListener(), new ClosedChannelException());
            }
            try {
                this.readWriteHandler.close();
            }
            catch (IOException e) {
                closingExceptions.add(e);
            }
            this.channelBuffer.close();
            if (!closingExceptions.isEmpty()) {
                ExceptionsHelper.rethrowAndSuppress(closingExceptions);
            }
        }
    }

    protected void handleReadBytes() throws IOException {
        int bytesConsumed = Integer.MAX_VALUE;
        while (bytesConsumed > 0 && this.channelBuffer.getIndex() > 0L) {
            bytesConsumed = this.readWriteHandler.consumeReads(this.channelBuffer);
            this.channelBuffer.release(bytesConsumed);
        }
        this.pendingFlushes.addAll(this.readWriteHandler.pollFlushOperations());
    }

    public boolean readyForFlush() {
        this.getSelector().assertOnSelectorThread();
        return !this.pendingFlushes.isEmpty();
    }

    public abstract boolean selectorShouldClose();

    protected boolean closeNow() {
        return this.closeNow;
    }

    protected void setCloseNow() {
        this.closeNow = true;
    }

    protected int readFromChannel(ByteBuffer buffer) throws IOException {
        int bytesRead;
        ByteBuffer ioBuffer = this.getSelector().getIoBuffer();
        ioBuffer.limit(Math.min(buffer.remaining(), ioBuffer.limit()));
        try {
            bytesRead = ((SocketChannel)this.rawChannel).read(ioBuffer);
        }
        catch (IOException e) {
            this.closeNow = true;
            throw e;
        }
        if (bytesRead < 0) {
            this.closeNow = true;
            return 0;
        }
        ioBuffer.flip();
        buffer.put(ioBuffer);
        return bytesRead;
    }

    protected int readFromChannel(InboundChannelBuffer channelBuffer) throws IOException {
        int bytesRead;
        ByteBuffer ioBuffer = this.getSelector().getIoBuffer();
        try {
            bytesRead = ((SocketChannel)this.rawChannel).read(ioBuffer);
        }
        catch (IOException e) {
            this.closeNow = true;
            throw e;
        }
        if (bytesRead < 0) {
            this.closeNow = true;
            return 0;
        }
        ioBuffer.flip();
        channelBuffer.ensureCapacity(channelBuffer.getIndex() + (long)ioBuffer.remaining());
        ByteBuffer[] buffers = channelBuffer.sliceBuffersFrom(channelBuffer.getIndex());
        int j = 0;
        while (j < buffers.length && ioBuffer.remaining() > 0) {
            ByteBuffer buffer = buffers[j++];
            this.copyBytes(ioBuffer, buffer);
        }
        channelBuffer.incrementIndex(bytesRead);
        return bytesRead;
    }

    protected int flushToChannel(ByteBuffer buffer) throws IOException {
        int bytesWritten;
        int initialPosition = buffer.position();
        ByteBuffer ioBuffer = this.getSelector().getIoBuffer();
        ioBuffer.limit(Math.min(65536, ioBuffer.limit()));
        this.copyBytes(buffer, ioBuffer);
        ioBuffer.flip();
        try {
            bytesWritten = ((SocketChannel)this.rawChannel).write(ioBuffer);
        }
        catch (IOException e) {
            this.closeNow = true;
            buffer.position(initialPosition);
            throw e;
        }
        buffer.position(initialPosition + bytesWritten);
        return bytesWritten;
    }

    protected int flushToChannel(FlushOperation flushOperation) throws IOException {
        ByteBuffer ioBuffer = this.getSelector().getIoBuffer();
        boolean continueFlush = !flushOperation.isFullyFlushed();
        int totalBytesFlushed = 0;
        while (continueFlush) {
            int bytesFlushed;
            ioBuffer.clear();
            ioBuffer.limit(Math.min(65536, ioBuffer.limit()));
            int j = 0;
            ByteBuffer[] buffers = flushOperation.getBuffersToWrite();
            while (j < buffers.length && ioBuffer.remaining() > 0) {
                ByteBuffer buffer = buffers[j++];
                this.copyBytes(buffer, ioBuffer);
            }
            ioBuffer.flip();
            try {
                bytesFlushed = ((SocketChannel)this.rawChannel).write(ioBuffer);
            }
            catch (IOException e) {
                this.closeNow = true;
                throw e;
            }
            flushOperation.incrementIndex(bytesFlushed);
            totalBytesFlushed += bytesFlushed;
            continueFlush = !ioBuffer.hasRemaining() && !flushOperation.isFullyFlushed();
        }
        return totalBytesFlushed;
    }

    private void copyBytes(ByteBuffer from, ByteBuffer to) {
        int nBytesToCopy = Math.min(to.remaining(), from.remaining());
        int initialLimit = from.limit();
        from.limit(from.position() + nBytesToCopy);
        to.put(from);
        from.limit(initialLimit);
    }
}

