/*
 * Decompiled with CFR 0.152.
 */
package org.embulk.input.jdbc;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.JsonNode;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.TreeMap;
import org.embulk.input.jdbc.JdbcColumn;
import org.embulk.input.jdbc.JdbcLiteral;
import org.embulk.input.jdbc.JdbcSchema;
import org.embulk.input.jdbc.getter.ColumnGetter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JdbcInputConnection
implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(JdbcInputConnection.class);
    protected final Connection connection;
    protected final String schemaName;
    protected final DatabaseMetaData databaseMetaData;
    protected String identifierQuoteString;

    public JdbcInputConnection(Connection connection, String schemaName) throws SQLException {
        this.connection = connection;
        this.schemaName = schemaName;
        this.databaseMetaData = connection.getMetaData();
        this.identifierQuoteString = this.databaseMetaData.getIdentifierQuoteString();
        if (schemaName != null) {
            this.setSearchPath(schemaName);
        }
        connection.setAutoCommit(false);
    }

    protected void setSearchPath(String schema) throws SQLException {
        String sql = "SET search_path TO " + this.quoteIdentifierString(schema);
        this.executeUpdate(sql);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public JdbcSchema getSchemaOfQuery(String query) throws SQLException {
        try (PreparedStatement stmt = this.connection.prepareStatement(query);){
            JdbcSchema jdbcSchema = this.getSchemaOfResultMetadata(stmt.getMetaData());
            return jdbcSchema;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<String> getPrimaryKeys(String tableName) throws SQLException {
        ArrayList<String> primaryKeys = new ArrayList<String>();
        try (ResultSet rs = this.databaseMetaData.getPrimaryKeys(null, this.schemaName, tableName);){
            while (rs.next()) {
                primaryKeys.add(rs.getString("COLUMN_NAME"));
            }
        }
        return Collections.unmodifiableList(primaryKeys);
    }

    protected JdbcSchema getSchemaOfResultMetadata(ResultSetMetaData metadata) throws SQLException {
        ArrayList<JdbcColumn> columns = new ArrayList<JdbcColumn>();
        for (int i = 0; i < metadata.getColumnCount(); ++i) {
            int index = i + 1;
            String name = metadata.getColumnLabel(index);
            String typeName = metadata.getColumnTypeName(index);
            int sqlType = metadata.getColumnType(index);
            int scale = metadata.getScale(index);
            int precision = metadata.getPrecision(index);
            columns.add(new JdbcColumn(name, typeName, sqlType, precision, scale));
        }
        return new JdbcSchema(Collections.unmodifiableList(columns));
    }

    public BatchSelect newSelectCursor(PreparedQuery preparedQuery, List<ColumnGetter> getters, int fetchRows, int queryTimeout) throws SQLException {
        return this.newBatchSelect(preparedQuery, getters, fetchRows, queryTimeout);
    }

    protected BatchSelect newBatchSelect(PreparedQuery preparedQuery, List<ColumnGetter> getters, int fetchRows, int queryTimeout) throws SQLException {
        String query = preparedQuery.getQuery();
        List<JdbcLiteral> params = preparedQuery.getParameters();
        PreparedStatement stmt = this.connection.prepareStatement(query);
        stmt.setFetchSize(fetchRows);
        stmt.setQueryTimeout(queryTimeout);
        logger.info("SQL: " + query);
        if (!params.isEmpty()) {
            logger.info("Parameters: {}", params);
            this.prepareParameters(stmt, getters, params);
        }
        return new SingleSelect(stmt);
    }

    protected void prepareParameters(PreparedStatement stmt, List<ColumnGetter> getters, List<JdbcLiteral> parameters) throws SQLException {
        for (int i = 0; i < parameters.size(); ++i) {
            JdbcLiteral literal = parameters.get(i);
            ColumnGetter getter = getters.get(literal.getColumnIndex());
            int index = i + 1;
            getter.decodeFromJsonTo(stmt, index, literal.getValue());
        }
    }

    @Override
    public void close() throws SQLException {
        this.connection.close();
    }

    protected void executeUpdate(String sql) throws SQLException {
        logger.info("SQL: " + sql);
        try (Statement stmt = this.connection.createStatement();){
            stmt.executeUpdate(sql);
        }
    }

    protected String quoteIdentifierString(String str) {
        return this.identifierQuoteString + str + this.identifierQuoteString;
    }

    protected String buildTableName(String tableName) {
        return this.quoteIdentifierString(tableName);
    }

    public String buildSelectQuery(String tableName, Optional<String> selectExpression, Optional<String> whereCondition, Optional<String> orderByExpression) throws SQLException {
        StringBuilder sb = new StringBuilder();
        sb.append("SELECT ");
        sb.append(selectExpression.orElse("*"));
        sb.append(" FROM ").append(this.buildTableName(tableName));
        if (whereCondition.isPresent()) {
            sb.append(" WHERE ").append(whereCondition.get());
        }
        if (orderByExpression.isPresent()) {
            sb.append(" ORDER BY ").append(orderByExpression.get());
        }
        return sb.toString();
    }

    public PreparedQuery rebuildIncrementalQuery(String tableName, Optional<String> selectExpression, Optional<String> whereCondition, JdbcSchema querySchema, List<String> incrementalColumns, List<JsonNode> incrementalValues) throws SQLException {
        Optional<String> newWhereCondition;
        List<JdbcLiteral> parameters = Collections.emptyList();
        if (incrementalValues != null) {
            StringBuilder sb = new StringBuilder();
            if (whereCondition.isPresent()) {
                sb.append("(");
                sb.append(whereCondition.get());
                sb.append(") AND ");
            }
            sb.append("(");
            parameters = this.buildIncrementalConditionTo(sb, querySchema, incrementalColumns, incrementalValues);
            sb.append(")");
            newWhereCondition = Optional.of(sb.toString());
        } else {
            newWhereCondition = whereCondition;
        }
        StringBuilder sb = new StringBuilder();
        this.buildIncrementalOrderTo(sb, querySchema, incrementalColumns);
        Optional<String> newOrderByExpression = Optional.of(sb.toString());
        String newQuery = this.buildSelectQuery(tableName, selectExpression, newWhereCondition, newOrderByExpression);
        return new PreparedQuery(newQuery, parameters);
    }

    public PreparedQuery wrapIncrementalQuery(String rawQuery, JdbcSchema querySchema, List<String> incrementalColumns, List<JsonNode> incrementalValues, boolean useRawQuery) throws SQLException {
        StringBuilder sb = new StringBuilder();
        List<JdbcLiteral> parameters = Collections.emptyList();
        if (useRawQuery) {
            parameters = this.replacePlaceholder(sb, rawQuery, querySchema, incrementalColumns, incrementalValues);
        } else {
            sb.append("SELECT * FROM (");
            sb.append(this.truncateStatementDelimiter(rawQuery));
            sb.append(") embulk_incremental_");
            if (incrementalValues != null) {
                sb.append(" WHERE ");
                parameters = this.buildIncrementalConditionTo(sb, querySchema, incrementalColumns, incrementalValues);
            }
            sb.append(" ORDER BY ");
            this.buildIncrementalOrderTo(sb, querySchema, incrementalColumns);
        }
        return new PreparedQuery(sb.toString(), parameters);
    }

    private List<JdbcLiteral> buildIncrementalConditionTo(StringBuilder sb, JdbcSchema querySchema, List<String> incrementalColumns, List<JsonNode> incrementalValues) throws SQLException {
        int n;
        ArrayList parameters = new ArrayList();
        ArrayList<String> leftColumnNames = new ArrayList<String>();
        ArrayList<JdbcLiteral> rightLiterals = new ArrayList<JdbcLiteral>();
        for (n = 0; n < incrementalColumns.size(); ++n) {
            int columnIndex = this.findIncrementalColumnIndex(querySchema, incrementalColumns.get(n));
            JsonNode value = incrementalValues.get(n);
            leftColumnNames.add(querySchema.getColumnName(columnIndex));
            rightLiterals.add(new JdbcLiteral(columnIndex, value));
        }
        for (n = 0; n < leftColumnNames.size(); ++n) {
            if (n > 0) {
                sb.append(" OR ");
            }
            sb.append("(");
            for (int i = 0; i < n; ++i) {
                sb.append(this.quoteIdentifierString((String)leftColumnNames.get(i)));
                sb.append(" = ?");
                parameters.add(rightLiterals.get(i));
                sb.append(" AND ");
            }
            sb.append(this.quoteIdentifierString((String)leftColumnNames.get(n)));
            sb.append(" > ?");
            parameters.add(rightLiterals.get(n));
            sb.append(")");
        }
        return Collections.unmodifiableList(parameters);
    }

    private int findIncrementalColumnIndex(JdbcSchema schema, String incrementalColumn) throws SQLException {
        Optional<Integer> index = schema.findColumn(incrementalColumn);
        return index.get();
    }

    protected TreeMap<String, Integer> createColumnNameSortedMap() {
        return new TreeMap<String, Integer>(new Comparator<String>(){

            @Override
            public int compare(String val1, String val2) {
                int c = val2.length() - val1.length();
                if (c != 0) {
                    return c;
                }
                return val1.compareTo(val2);
            }
        });
    }

    private List<JdbcLiteral> replacePlaceholder(StringBuilder sb, String rawQuery, JdbcSchema querySchema, List<String> incrementalColumns, List<JsonNode> incrementalValues) throws SQLException {
        int columnIndex;
        TreeMap<String, Integer> columnNames = this.createColumnNameSortedMap();
        ArrayList<JdbcLiteral> parameters = new ArrayList<JdbcLiteral>();
        for (String string : incrementalColumns) {
            columnIndex = this.findIncrementalColumnIndex(querySchema, string);
            columnNames.put(string, columnIndex);
        }
        for (Map.Entry entry : this.generateColumnPositionList(rawQuery, columnNames).entrySet()) {
            columnIndex = (Integer)entry.getValue();
            JsonNode value = incrementalValues.get(columnIndex);
            parameters.add(new JdbcLiteral(columnIndex, value));
        }
        for (Map.Entry entry : columnNames.entrySet()) {
            String columnName = (String)entry.getKey();
            while (rawQuery.contains(":" + columnName)) {
                rawQuery = rawQuery.replaceFirst(":" + columnName, "?");
            }
        }
        sb.append(rawQuery);
        return Collections.unmodifiableList(parameters);
    }

    private TreeMap<Integer, Integer> generateColumnPositionList(String rawQuery, TreeMap<String, Integer> columnNames) {
        TreeMap<Integer, Integer> columnPositionList = new TreeMap<Integer, Integer>();
        for (Map.Entry<String, Integer> column : columnNames.entrySet()) {
            int index;
            int lastIndex = 0;
            while ((index = rawQuery.indexOf(":" + column.getKey(), lastIndex)) != -1) {
                if (!columnPositionList.containsKey(index)) {
                    columnPositionList.put(index, column.getValue());
                }
                lastIndex = index + 2;
            }
        }
        return columnPositionList;
    }

    private void buildIncrementalOrderTo(StringBuilder sb, JdbcSchema querySchema, List<String> incrementalColumns) throws SQLException {
        boolean first = true;
        for (String incrementalColumn : incrementalColumns) {
            if (first) {
                first = false;
            } else {
                sb.append(", ");
            }
            int columnIndex = this.findIncrementalColumnIndex(querySchema, incrementalColumn);
            sb.append(this.quoteIdentifierString(querySchema.getColumnName(columnIndex)));
        }
    }

    protected String truncateStatementDelimiter(String rawQuery) throws SQLException {
        return rawQuery.replaceAll(";\\s*$", "");
    }

    public boolean tableExists(String tableName) throws SQLException {
        try (ResultSet rs = this.connection.getMetaData().getTables(null, this.schemaName, tableName, null);){
            boolean bl = rs.next();
            return bl;
        }
    }

    private Set<String> getColumnNames(String tableName) throws SQLException {
        HashSet<String> columnNames = new HashSet<String>();
        try (ResultSet rs = this.connection.getMetaData().getColumns(null, this.schemaName, tableName, null);){
            while (rs.next()) {
                columnNames.add(rs.getString("COLUMN_NAME"));
            }
            Set<String> set = Collections.unmodifiableSet(columnNames);
            return set;
        }
    }

    public void showDriverVersion() throws SQLException {
        DatabaseMetaData meta = this.connection.getMetaData();
        logger.info(String.format(Locale.ENGLISH, "Using JDBC Driver %s", meta.getDriverVersion()));
    }

    public void commit() throws SQLException {
        this.connection.commit();
    }

    public class SingleSelect
    implements BatchSelect {
        private final PreparedStatement fetchStatement;
        private boolean fetched = false;

        public SingleSelect(PreparedStatement fetchStatement) {
            this.fetchStatement = fetchStatement;
        }

        @Override
        public ResultSet fetch() throws SQLException {
            if (this.fetched) {
                return null;
            }
            long startTime = System.currentTimeMillis();
            ResultSet rs = this.fetchStatement.executeQuery();
            double seconds = (double)(System.currentTimeMillis() - startTime) / 1000.0;
            logger.info(String.format("> %.2f seconds", seconds));
            this.fetched = true;
            return rs;
        }

        @Override
        public void close() throws SQLException {
        }
    }

    public static interface BatchSelect
    extends AutoCloseable {
        public ResultSet fetch() throws SQLException;

        @Override
        public void close() throws SQLException;
    }

    public static class PreparedQuery {
        private final String query;
        private final List<JdbcLiteral> parameters;

        @JsonCreator
        public PreparedQuery(@JsonProperty(value="query") String query, @JsonProperty(value="parameters") List<JdbcLiteral> parameters) {
            this.query = query;
            this.parameters = parameters;
        }

        @JsonProperty(value="query")
        public String getQuery() {
            return this.query;
        }

        @JsonProperty(value="parameters")
        public List<JdbcLiteral> getParameters() {
            return this.parameters;
        }
    }
}

