/*
 * Decompiled with CFR 0.152.
 */
package org.embulk.output.jdbc;

import java.net.UnknownHostException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import org.embulk.output.jdbc.JdbcColumn;
import org.embulk.output.jdbc.JdbcSchema;
import org.embulk.output.jdbc.MergeConfig;
import org.embulk.output.jdbc.TableIdentifier;
import org.embulk.output.jdbc.TransactionIsolation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JdbcOutputConnection
implements AutoCloseable {
    protected static final Logger logger = LoggerFactory.getLogger(JdbcOutputConnection.class);
    protected final Connection connection;
    protected final String schemaName;
    protected final DatabaseMetaData databaseMetaData;
    protected String identifierQuoteString;
    private static final String[] STANDARD_SIZE_TYPE_NAMES = new String[]{"CHAR", "VARCHAR", "CHAR VARYING", "CHARACTER VARYING", "LONGVARCHAR", "NCHAR", "NVARCHAR", "NCHAR VARYING", "NATIONAL CHAR VARYING", "NATIONAL CHARACTER VARYING", "BINARY", "VARBINARY", "BINARY VARYING", "LONGVARBINARY", "BIT", "VARBIT", "BIT VARYING", "FLOAT"};
    private static final String[] STANDARD_SIZE_AND_SCALE_TYPE_NAMES = new String[]{"DECIMAL", "NUMERIC"};

    public JdbcOutputConnection(Connection connection, String schemaName) throws SQLException {
        this.connection = connection;
        this.schemaName = schemaName;
        this.databaseMetaData = connection.getMetaData();
        this.identifierQuoteString = this.databaseMetaData.getIdentifierQuoteString();
        if (schemaName != null) {
            this.setSearchPath(schemaName);
        }
    }

    public void initialize(boolean autoCommit, Optional<TransactionIsolation> transactionIsolation) throws SQLException {
        this.connection.setAutoCommit(autoCommit);
        if (transactionIsolation.isPresent()) {
            this.connection.setTransactionIsolation(transactionIsolation.get().toInt());
        }
        try {
            TransactionIsolation currentTransactionIsolation = TransactionIsolation.fromInt(this.connection.getTransactionIsolation());
            logger.info("TransactionIsolation={}", (Object)currentTransactionIsolation.toString());
        }
        catch (IllegalArgumentException e) {
            logger.info("TransactionIsolation=unknown");
        }
    }

    @Override
    public void close() throws SQLException {
        if (!this.connection.isClosed()) {
            this.connection.close();
        }
    }

    public String getSchemaName() {
        return this.schemaName;
    }

    public DatabaseMetaData getMetaData() throws SQLException {
        return this.databaseMetaData;
    }

    public Charset getTableNameCharset() throws SQLException {
        return StandardCharsets.UTF_8;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void setSearchPath(String schema) throws SQLException {
        try (Statement stmt = this.connection.createStatement();){
            String sql = "SET search_path TO " + this.quoteIdentifierString(schema);
            this.executeUpdate(stmt, sql);
            this.commitIfNecessary(this.connection);
        }
    }

    public boolean tableExists(TableIdentifier table) throws SQLException {
        try (ResultSet rs = this.connection.getMetaData().getTables(table.getDatabase(), table.getSchemaName(), table.getTableName(), null);){
            boolean bl = rs.next();
            return bl;
        }
    }

    public boolean tableExists(String tableName) throws SQLException {
        return this.tableExists(new TableIdentifier(null, this.schemaName, tableName));
    }

    protected boolean supportsTableIfExistsClause() {
        return true;
    }

    public void dropTableIfExists(TableIdentifier table) throws SQLException {
        try (Statement stmt = this.connection.createStatement();){
            this.dropTableIfExists(stmt, table);
            this.commitIfNecessary(this.connection);
        }
    }

    protected void dropTableIfExists(Statement stmt, TableIdentifier table) throws SQLException {
        if (this.supportsTableIfExistsClause()) {
            String sql = String.format("DROP TABLE IF EXISTS %s", this.quoteTableIdentifier(table));
            this.executeUpdate(stmt, sql);
        } else if (this.tableExists(table)) {
            this.dropTable(stmt, table);
        }
    }

    public void dropTable(TableIdentifier table) throws SQLException {
        try (Statement stmt = this.connection.createStatement();){
            this.dropTable(stmt, table);
            this.commitIfNecessary(this.connection);
        }
    }

    protected void dropTable(Statement stmt, TableIdentifier table) throws SQLException {
        String sql = String.format("DROP TABLE %s", this.quoteTableIdentifier(table));
        this.executeUpdate(stmt, sql);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public void createTableIfNotExists(TableIdentifier table, JdbcSchema schema, Optional<String> tableConstraint, Optional<String> tableOption) throws SQLException {
        if (this.supportsTableIfExistsClause()) {
            try (Statement stmt = this.connection.createStatement();){
                String sql = this.buildCreateTableIfNotExistsSql(table, schema, tableConstraint, tableOption);
                this.executeUpdate(stmt, sql);
                this.commitIfNecessary(this.connection);
                return;
            }
        }
        if (this.tableExists(table)) return;
        try {
            this.createTable(table, schema, tableConstraint, tableOption);
            return;
        }
        catch (SQLException e) {
            if (this.tableExists(table)) return;
            throw e;
        }
    }

    protected String buildCreateTableIfNotExistsSql(TableIdentifier table, JdbcSchema schema, Optional<String> tableConstraint, Optional<String> tableOption) {
        StringBuilder sb = new StringBuilder();
        sb.append("CREATE TABLE IF NOT EXISTS ");
        this.quoteTableIdentifier(sb, table);
        sb.append(this.buildCreateTableSchemaSql(schema, tableConstraint));
        if (tableOption.isPresent()) {
            sb.append(" ");
            sb.append(tableOption.get());
        }
        return sb.toString();
    }

    public void createTable(TableIdentifier table, JdbcSchema schema, Optional<String> tableConstraint, Optional<String> tableOption) throws SQLException {
        try (Statement stmt = this.connection.createStatement();){
            String sql = this.buildCreateTableSql(table, schema, tableConstraint, tableOption);
            this.executeUpdate(stmt, sql);
            this.commitIfNecessary(this.connection);
        }
    }

    protected String buildCreateTableSql(TableIdentifier table, JdbcSchema schema, Optional<String> tableConstraint, Optional<String> tableOption) {
        StringBuilder sb = new StringBuilder();
        sb.append("CREATE TABLE ");
        this.quoteTableIdentifier(sb, table);
        sb.append(this.buildCreateTableSchemaSql(schema, tableConstraint));
        if (tableOption.isPresent()) {
            sb.append(" ");
            sb.append(tableOption.get());
        }
        return sb.toString();
    }

    protected String buildCreateTableSchemaSql(JdbcSchema schema, Optional<String> tableConstraint) {
        StringBuilder sb = new StringBuilder();
        sb.append(" (");
        for (int i = 0; i < schema.getCount(); ++i) {
            if (i != 0) {
                sb.append(", ");
            }
            this.quoteIdentifierString(sb, schema.getColumnName(i));
            sb.append(" ");
            String typeName = this.getCreateTableTypeName(schema.getColumn(i));
            sb.append(typeName);
        }
        if (tableConstraint.isPresent()) {
            sb.append(", ");
            sb.append(tableConstraint.get());
        }
        sb.append(")");
        return sb.toString();
    }

    protected String buildRenameTableSql(TableIdentifier fromTable, TableIdentifier toTable) {
        StringBuilder sb = new StringBuilder();
        sb.append("ALTER TABLE ");
        this.quoteTableIdentifier(sb, fromTable);
        sb.append(" RENAME TO ");
        this.quoteTableIdentifier(sb, toTable);
        return sb.toString();
    }

    protected String getCreateTableTypeName(JdbcColumn c) {
        if (c.getDeclaredType().isPresent()) {
            return c.getDeclaredType().get();
        }
        return this.buildColumnTypeName(c);
    }

    protected String buildColumnTypeName(JdbcColumn c) {
        String simpleTypeName = c.getSimpleTypeName();
        switch (this.getColumnDeclareType(simpleTypeName, c)) {
            case SIZE: {
                return String.format("%s(%d)", simpleTypeName, c.getSizeTypeParameter());
            }
            case SIZE_AND_SCALE: {
                if (c.getScaleTypeParameter() < 0) {
                    return String.format("%s(%d,0)", simpleTypeName, c.getSizeTypeParameter());
                }
                return String.format("%s(%d,%d)", simpleTypeName, c.getSizeTypeParameter(), c.getScaleTypeParameter());
            }
            case SIZE_AND_OPTIONAL_SCALE: {
                if (c.getScaleTypeParameter() < 0) {
                    return String.format("%s(%d)", simpleTypeName, c.getSizeTypeParameter());
                }
                return String.format("%s(%d,%d)", simpleTypeName, c.getSizeTypeParameter(), c.getScaleTypeParameter());
            }
        }
        return simpleTypeName;
    }

    protected ColumnDeclareType getColumnDeclareType(String convertedTypeName, JdbcColumn col) {
        for (String x : STANDARD_SIZE_TYPE_NAMES) {
            if (!x.equals(convertedTypeName)) continue;
            return ColumnDeclareType.SIZE;
        }
        for (String x : STANDARD_SIZE_AND_SCALE_TYPE_NAMES) {
            if (!x.equals(convertedTypeName)) continue;
            return ColumnDeclareType.SIZE_AND_SCALE;
        }
        return ColumnDeclareType.SIMPLE;
    }

    public PreparedStatement prepareBatchInsertStatement(TableIdentifier toTable, JdbcSchema toTableSchema, Optional<MergeConfig> mergeConfig) throws SQLException {
        String sql = mergeConfig.isPresent() ? this.buildPreparedMergeSql(toTable, toTableSchema, mergeConfig.get()) : this.buildPreparedInsertSql(toTable, toTableSchema);
        logger.info("Prepared SQL: {}", (Object)sql);
        return this.connection.prepareStatement(sql);
    }

    protected String buildPreparedInsertSql(TableIdentifier toTable, JdbcSchema toTableSchema) throws SQLException {
        int i;
        StringBuilder sb = new StringBuilder();
        sb.append("INSERT INTO ");
        this.quoteTableIdentifier(sb, toTable);
        sb.append(" (");
        for (i = 0; i < toTableSchema.getCount(); ++i) {
            if (i != 0) {
                sb.append(", ");
            }
            this.quoteIdentifierString(sb, toTableSchema.getColumnName(i));
        }
        sb.append(") VALUES (");
        for (i = 0; i < toTableSchema.getCount(); ++i) {
            if (i != 0) {
                sb.append(", ");
            }
            sb.append("?");
        }
        sb.append(")");
        return sb.toString();
    }

    protected String buildPreparedMergeSql(TableIdentifier toTable, JdbcSchema toTableSchema, MergeConfig mergeConfig) throws SQLException {
        throw new UnsupportedOperationException("not implemented");
    }

    @Deprecated
    protected void executeSql(String sql) throws SQLException {
        this.executeUpdateInNewStatement(sql);
    }

    protected void executeUpdateInNewStatement(String sql) throws SQLException {
        try (Statement stmt = this.connection.createStatement();){
            this.executeUpdate(stmt, sql);
            this.commitIfNecessary(this.connection);
        }
    }

    protected void executeInNewStatement(String sql) throws SQLException {
        try (Statement stmt = this.connection.createStatement();){
            this.execute(stmt, sql);
            this.commitIfNecessary(this.connection);
        }
    }

    protected void collectInsert(List<TableIdentifier> fromTables, JdbcSchema schema, TableIdentifier toTable, boolean truncateDestinationFirst, Optional<String> preSql, Optional<String> postSql) throws SQLException {
        if (fromTables.isEmpty()) {
            return;
        }
        try (Statement stmt = this.connection.createStatement();){
            String sql;
            if (truncateDestinationFirst) {
                sql = this.buildTruncateSql(toTable);
                this.executeUpdate(stmt, sql);
            }
            if (preSql.isPresent()) {
                this.execute(stmt, preSql.get());
            }
            sql = this.buildCollectInsertSql(fromTables, schema, toTable);
            this.executeUpdate(stmt, sql);
            if (postSql.isPresent()) {
                this.execute(stmt, postSql.get());
            }
            this.commitIfNecessary(this.connection);
        }
    }

    protected String buildTruncateSql(TableIdentifier table) {
        StringBuilder sb = new StringBuilder();
        sb.append("DELETE FROM ");
        this.quoteTableIdentifier(sb, table);
        return sb.toString();
    }

    protected String buildCollectInsertSql(List<TableIdentifier> fromTables, JdbcSchema schema, TableIdentifier toTable) {
        int i;
        StringBuilder sb = new StringBuilder();
        sb.append("INSERT INTO ");
        this.quoteTableIdentifier(sb, toTable);
        sb.append(" (");
        for (i = 0; i < schema.getCount(); ++i) {
            if (i != 0) {
                sb.append(", ");
            }
            this.quoteIdentifierString(sb, schema.getColumnName(i));
        }
        sb.append(") ");
        for (i = 0; i < fromTables.size(); ++i) {
            if (i != 0) {
                sb.append(" UNION ALL ");
            }
            sb.append("SELECT ");
            for (int j = 0; j < schema.getCount(); ++j) {
                if (j != 0) {
                    sb.append(", ");
                }
                this.quoteIdentifierString(sb, schema.getColumnName(j));
            }
            sb.append(" FROM ");
            this.quoteTableIdentifier(sb, fromTables.get(i));
        }
        return sb.toString();
    }

    protected void collectMerge(List<TableIdentifier> fromTables, JdbcSchema schema, TableIdentifier toTable, MergeConfig mergeConfig, Optional<String> preSql, Optional<String> postSql) throws SQLException {
        if (fromTables.isEmpty()) {
            return;
        }
        try (Statement stmt = this.connection.createStatement();){
            if (preSql.isPresent()) {
                this.execute(stmt, preSql.get());
            }
            String sql = this.buildCollectMergeSql(fromTables, schema, toTable, mergeConfig);
            this.executeUpdate(stmt, sql);
            if (postSql.isPresent()) {
                this.execute(stmt, postSql.get());
            }
            this.commitIfNecessary(this.connection);
        }
    }

    protected String buildCollectMergeSql(List<TableIdentifier> fromTables, JdbcSchema schema, TableIdentifier toTable, MergeConfig mergeConfig) throws SQLException {
        throw new UnsupportedOperationException("not implemented");
    }

    public void replaceTable(TableIdentifier fromTable, JdbcSchema schema, TableIdentifier toTable, Optional<String> postSql) throws SQLException {
        try (Statement stmt = this.connection.createStatement();){
            this.dropTableIfExists(stmt, toTable);
            this.executeUpdate(stmt, this.buildRenameTableSql(fromTable, toTable));
            if (postSql.isPresent()) {
                this.execute(stmt, postSql.get());
            }
            this.commitIfNecessary(this.connection);
        }
    }

    protected String quoteTableIdentifier(TableIdentifier table) {
        StringBuilder sb = new StringBuilder();
        if (table.getDatabase() != null) {
            sb.append(this.quoteIdentifierString(table.getDatabase(), this.identifierQuoteString));
            sb.append(".");
        }
        if (table.getSchemaName() != null) {
            sb.append(this.quoteIdentifierString(table.getSchemaName(), this.identifierQuoteString));
            sb.append(".");
        }
        sb.append(this.quoteIdentifierString(table.getTableName(), this.identifierQuoteString));
        return sb.toString();
    }

    protected void quoteTableIdentifier(StringBuilder sb, TableIdentifier table) {
        sb.append(this.quoteTableIdentifier(table));
    }

    protected void quoteIdentifierString(StringBuilder sb, String str) {
        sb.append(this.quoteIdentifierString(str));
    }

    protected String quoteIdentifierString(String str) {
        return this.quoteIdentifierString(str, this.identifierQuoteString);
    }

    protected String quoteIdentifierString(String str, String quoteString) {
        return quoteString + str + quoteString;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean isValidConnection(int timeout) throws SQLException {
        try (Statement stmt = this.connection.createStatement();){
            stmt.executeQuery("SELECT 1").close();
            boolean bl = true;
            return bl;
        }
    }

    protected String[] getDeterministicSqlStates() {
        return new String[0];
    }

    protected int[] getDeterministicErrorCodes() {
        return new int[0];
    }

    protected Class[] getDeterministicRootCauses() {
        return new Class[]{UnknownHostException.class};
    }

    public boolean isRetryableException(SQLException exception) {
        String sqlState = exception.getSQLState();
        for (String deterministic : this.getDeterministicSqlStates()) {
            if (!sqlState.equals(deterministic)) continue;
            return false;
        }
        int errorCode = exception.getErrorCode();
        for (int deterministic : this.getDeterministicErrorCodes()) {
            if (errorCode != deterministic) continue;
            return false;
        }
        Throwable rootCause = this.getRootCause(exception);
        for (Class deterministic : this.getDeterministicRootCauses()) {
            if (!deterministic.equals(rootCause.getClass())) continue;
            return false;
        }
        return true;
    }

    private Throwable getRootCause(Throwable e) {
        while (e.getCause() != null) {
            e = e.getCause();
        }
        return e;
    }

    protected int executeUpdate(Statement stmt, String sql) throws SQLException {
        logger.info("SQL: " + sql);
        long startTime = System.currentTimeMillis();
        int count = stmt.executeUpdate(sql);
        double seconds = (double)(System.currentTimeMillis() - startTime) / 1000.0;
        if (count == 0) {
            logger.info(String.format("> %.2f seconds", seconds));
        } else {
            logger.info(String.format("> %.2f seconds (%,d rows)", seconds, count));
        }
        return count;
    }

    protected boolean execute(Statement stmt, String sql) throws SQLException {
        logger.info("SQL: " + sql);
        long startTime = System.currentTimeMillis();
        boolean result = stmt.execute(sql);
        double seconds = (double)(System.currentTimeMillis() - startTime) / 1000.0;
        if (result) {
            logger.info(String.format("> succeed %.2f seconds", seconds));
        } else {
            logger.info(String.format("> failed %.2f seconds", seconds));
        }
        return result;
    }

    protected void commitIfNecessary(Connection con) throws SQLException {
        if (!con.getAutoCommit()) {
            con.commit();
        }
    }

    protected SQLException safeRollback(Connection con, SQLException cause) {
        try {
            if (!con.getAutoCommit()) {
                con.rollback();
            }
            return cause;
        }
        catch (SQLException ex) {
            if (cause != null) {
                cause.addSuppressed(ex);
                return cause;
            }
            return ex;
        }
    }

    public void showDriverVersion() throws SQLException {
        DatabaseMetaData meta = this.connection.getMetaData();
        logger.info(String.format(Locale.ENGLISH, "Using JDBC Driver %s", meta.getDriverVersion()));
    }

    public static enum ColumnDeclareType {
        SIMPLE,
        SIZE,
        SIZE_AND_SCALE,
        SIZE_AND_OPTIONAL_SCALE;

    }
}

