/*
 * Decompiled with CFR 0.152.
 */
package org.apache.metamodel.excel;

import java.io.File;
import java.io.InputStream;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import org.apache.metamodel.MetaModelException;
import org.apache.metamodel.data.DataSet;
import org.apache.metamodel.data.DataSetHeader;
import org.apache.metamodel.data.DefaultRow;
import org.apache.metamodel.data.EmptyDataSet;
import org.apache.metamodel.data.Style;
import org.apache.metamodel.data.StyleBuilder;
import org.apache.metamodel.excel.DefaultSpreadsheetReaderDelegate;
import org.apache.metamodel.excel.ExcelConfiguration;
import org.apache.metamodel.excel.ExcelDataContext;
import org.apache.metamodel.excel.XlsDataSet;
import org.apache.metamodel.excel.ZeroBasedRowIterator;
import org.apache.metamodel.query.SelectItem;
import org.apache.metamodel.schema.ColumnType;
import org.apache.metamodel.schema.Table;
import org.apache.metamodel.util.DateUtils;
import org.apache.metamodel.util.FileHelper;
import org.apache.metamodel.util.FileResource;
import org.apache.metamodel.util.InMemoryResource;
import org.apache.metamodel.util.Resource;
import org.apache.poi.hssf.usermodel.HSSFFont;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.hssf.util.HSSFColor;
import org.apache.poi.ss.usermodel.BuiltinFormats;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.Color;
import org.apache.poi.ss.usermodel.DataFormatter;
import org.apache.poi.ss.usermodel.DateUtil;
import org.apache.poi.ss.usermodel.FillPatternType;
import org.apache.poi.ss.usermodel.Font;
import org.apache.poi.ss.usermodel.FontUnderline;
import org.apache.poi.ss.usermodel.FormulaError;
import org.apache.poi.ss.usermodel.FormulaEvaluator;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.apache.poi.xssf.usermodel.XSSFCell;
import org.apache.poi.xssf.usermodel.XSSFColor;
import org.apache.poi.xssf.usermodel.XSSFFont;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.XMLReader;

final class ExcelUtils {
    private static final Logger logger = LoggerFactory.getLogger(ExcelUtils.class);

    private ExcelUtils() {
    }

    public static XMLReader createXmlReader() {
        try {
            SAXParserFactory saxFactory = SAXParserFactory.newInstance();
            SAXParser saxParser = saxFactory.newSAXParser();
            XMLReader sheetParser = saxParser.getXMLReader();
            return sheetParser;
        }
        catch (Exception e) {
            throw new MetaModelException(e);
        }
    }

    public static Workbook readWorkbook(Resource resource, boolean allowFileOptimization) {
        if (!resource.isExists()) {
            if (ExcelUtils.isXlsxFile(resource)) {
                return new SXSSFWorkbook(1000);
            }
            return new HSSFWorkbook();
        }
        if (allowFileOptimization && resource instanceof FileResource) {
            File file = ((FileResource)resource).getFile();
            try {
                return WorkbookFactory.create((File)file, null, (boolean)true);
            }
            catch (Exception e) {
                logger.error("Could not open workbook", (Throwable)e);
                throw new IllegalStateException("Could not open workbook", e);
            }
        }
        return (Workbook)resource.read(inputStream -> {
            try {
                return WorkbookFactory.create((InputStream)inputStream);
            }
            catch (Exception e) {
                logger.error("Could not open workbook", (Throwable)e);
                throw new IllegalStateException("Could not open workbook", e);
            }
        });
    }

    public static boolean isXlsxFile(Resource resource) {
        if (resource == null) {
            return false;
        }
        return resource.getName().toLowerCase().endsWith(".xlsx");
    }

    public static Workbook readWorkbookForUpdate(ExcelDataContext dataContext) {
        Resource resource = dataContext.getResource();
        return ExcelUtils.readWorkbook(resource, false);
    }

    public static void writeAndCloseWorkbook(ExcelDataContext dataContext, Workbook wb) {
        Resource realResource = dataContext.getResource();
        InMemoryResource tempResource = new InMemoryResource(realResource.getQualifiedPath());
        tempResource.write(out -> wb.write(out));
        FileHelper.safeClose((Object[])new Object[]{wb});
        FileHelper.copy((Resource)tempResource, (Resource)realResource);
    }

    public static String getCellValue(Workbook wb, Cell cell) {
        String result;
        if (cell == null) {
            return null;
        }
        switch (cell.getCellType()) {
            case BLANK: 
            case _NONE: {
                result = null;
                break;
            }
            case BOOLEAN: {
                result = Boolean.toString(cell.getBooleanCellValue());
                break;
            }
            case ERROR: {
                result = ExcelUtils.getErrorResult(cell);
                break;
            }
            case FORMULA: {
                result = ExcelUtils.getFormulaCellValue(wb, cell);
                break;
            }
            case NUMERIC: {
                if (DateUtil.isCellDateFormatted((Cell)cell)) {
                    Date date = cell.getDateCellValue();
                    if (date == null) {
                        result = null;
                        break;
                    }
                    result = DateUtils.createDateFormat().format(date);
                    break;
                }
                result = ExcelUtils.getNumericCellValueAsString(cell.getCellStyle(), cell.getNumericCellValue());
                break;
            }
            case STRING: {
                result = cell.getRichStringCellValue().getString();
                break;
            }
            default: {
                throw new IllegalStateException("Unknown cell type: " + cell.getCellType());
            }
        }
        logger.debug("cell ({},{}) resolved to value: {}", new Object[]{cell.getRowIndex(), cell.getColumnIndex(), result});
        return result;
    }

    private static Object getCellValueAsObject(Workbook workbook, Cell cell) {
        Object result;
        if (cell == null) {
            return null;
        }
        switch (cell.getCellType()) {
            case BLANK: 
            case _NONE: {
                result = null;
                break;
            }
            case BOOLEAN: {
                result = cell.getBooleanCellValue();
                break;
            }
            case ERROR: {
                result = ExcelUtils.getErrorResult(cell);
                break;
            }
            case FORMULA: {
                result = ExcelUtils.getFormulaCellValueAsObject(workbook, cell);
                break;
            }
            case NUMERIC: {
                if (DateUtil.isCellDateFormatted((Cell)cell)) {
                    result = cell.getDateCellValue();
                    break;
                }
                result = ExcelUtils.getDoubleAsNumber(cell.getNumericCellValue());
                break;
            }
            case STRING: {
                result = cell.getRichStringCellValue().getString();
                break;
            }
            default: {
                throw new IllegalStateException("Unknown cell type: " + cell.getCellType());
            }
        }
        logger.debug("cell ({},{}) resolved to value: {}", new Object[]{cell.getRowIndex(), cell.getColumnIndex(), result});
        return result;
    }

    private static String getErrorResult(Cell cell) {
        try {
            return FormulaError.forInt((byte)cell.getErrorCellValue()).getString();
        }
        catch (RuntimeException e) {
            logger.debug("Getting error code for ({},{}) failed!: {}", new Object[]{cell.getRowIndex(), cell.getColumnIndex(), e.getMessage()});
            if (cell instanceof XSSFCell) {
                return ((XSSFCell)cell).getErrorCellString();
            }
            logger.error("Couldn't handle unexpected error scenario in cell: ({},{})", (Object)cell.getRowIndex(), (Object)cell.getColumnIndex());
            throw e;
        }
    }

    private static Object evaluateCell(Workbook workbook, Cell cell, ColumnType expectedColumnType) {
        Object value = ExcelUtils.getCellValueAsObject(workbook, cell);
        if (value == null || value.getClass().equals(expectedColumnType.getJavaEquivalentClass()) || value.getClass().equals(Integer.class) && expectedColumnType.getJavaEquivalentClass().equals(Double.class)) {
            return value;
        }
        if (logger.isWarnEnabled()) {
            logger.warn("Cell ({},{}) has the value '{}' of data type '{}', which doesn't match the detected column's data type '{}'. This cell gets value NULL in the DataSet.", new Object[]{cell.getRowIndex(), cell.getColumnIndex(), value, value.getClass().getSimpleName(), expectedColumnType});
        }
        return null;
    }

    private static String getFormulaCellValue(Workbook workbook, Cell cell) {
        try {
            double numericCellValue = cell.getNumericCellValue();
            return ExcelUtils.getNumericCellValueAsString(cell.getCellStyle(), numericCellValue);
        }
        catch (Exception e) {
            Cell evaluatedCell;
            if (logger.isInfoEnabled()) {
                logger.info("Failed to fetch cached/precalculated formula value of cell: " + cell, (Throwable)e);
            }
            if ((evaluatedCell = ExcelUtils.getEvaluatedCell(workbook, cell)) != null) {
                return ExcelUtils.getCellValue(workbook, evaluatedCell);
            }
            return cell.getCellFormula();
        }
    }

    private static Object getFormulaCellValueAsObject(Workbook workbook, Cell cell) {
        try {
            return ExcelUtils.getDoubleAsNumber(cell.getNumericCellValue());
        }
        catch (Exception e) {
            Cell evaluatedCell;
            if (logger.isInfoEnabled()) {
                logger.info("Failed to fetch cached/precalculated formula value of cell: " + cell, (Throwable)e);
            }
            if ((evaluatedCell = ExcelUtils.getEvaluatedCell(workbook, cell)) != null) {
                return ExcelUtils.getCellValueAsObject(workbook, evaluatedCell);
            }
            return cell.getCellFormula();
        }
    }

    private static Cell getEvaluatedCell(Workbook workbook, Cell cell) {
        try {
            if (logger.isInfoEnabled()) {
                logger.info("cell ({},{}) is a formula. Attempting to evaluate: {}", new Object[]{cell.getRowIndex(), cell.getColumnIndex(), cell.getCellFormula()});
            }
            FormulaEvaluator evaluator = workbook.getCreationHelper().createFormulaEvaluator();
            return evaluator.evaluateInCell(cell);
        }
        catch (RuntimeException e) {
            logger.warn("Exception occurred while evaluating formula at position ({},{}): {}", new Object[]{cell.getRowIndex(), cell.getColumnIndex(), e.getMessage()});
            return null;
        }
    }

    private static Number getDoubleAsNumber(double value) {
        Double doubleValue = value;
        if (doubleValue % 1.0 == 0.0 && doubleValue <= 2.147483647E9) {
            return doubleValue.intValue();
        }
        return doubleValue;
    }

    public static Style getCellStyle(Workbook workbook, Cell cell) {
        XSSFColor color;
        if (cell == null) {
            return Style.NO_STYLE;
        }
        CellStyle cellStyle = cell.getCellStyle();
        int fontIndex = cellStyle.getFontIndex();
        Font font = workbook.getFontAt(fontIndex);
        StyleBuilder styleBuilder = new StyleBuilder();
        if (font.getBold()) {
            styleBuilder.bold();
        }
        if (font.getItalic()) {
            styleBuilder.italic();
        }
        if (font.getUnderline() != FontUnderline.NONE.getByteValue()) {
            styleBuilder.underline();
        }
        Font stdFont = workbook.getFontAt(0);
        short fontSize = font.getFontHeightInPoints();
        if (stdFont.getFontHeightInPoints() != fontSize) {
            styleBuilder.fontSize((int)fontSize, Style.SizeUnit.PT);
        }
        short colorIndex = font.getColor();
        if (font instanceof HSSFFont) {
            HSSFWorkbook wb;
            if (colorIndex != Short.MAX_VALUE && (color = (wb = (HSSFWorkbook)workbook).getCustomPalette().getColor(colorIndex)) != null) {
                short[] triplet = color.getTriplet();
                styleBuilder.foreground(triplet);
            }
        } else if (font instanceof XSSFFont) {
            String argbHex;
            XSSFFont xssfFont = (XSSFFont)font;
            color = xssfFont.getXSSFColor();
            if (color != null && (argbHex = color.getARGBHex()) != null) {
                styleBuilder.foreground(argbHex.substring(2));
            }
        } else {
            throw new IllegalStateException("Unexpected font type: " + (font == null ? "null" : font.getClass()) + ")");
        }
        if (cellStyle.getFillPattern() == FillPatternType.SOLID_FOREGROUND) {
            Color color2 = cellStyle.getFillForegroundColorColor();
            if (color2 instanceof HSSFColor) {
                short[] triplet = ((HSSFColor)color2).getTriplet();
                if (triplet != null) {
                    styleBuilder.background(triplet);
                }
            } else if (color2 instanceof XSSFColor) {
                String argb = ((XSSFColor)color2).getARGBHex();
                if (argb != null) {
                    styleBuilder.background(argb.substring(2));
                }
            } else {
                throw new IllegalStateException("Unexpected color type: " + (color2 == null ? "null" : color2.getClass()) + ")");
            }
        }
        switch (cellStyle.getAlignment()) {
            case LEFT: {
                styleBuilder.leftAligned();
                break;
            }
            case RIGHT: {
                styleBuilder.rightAligned();
                break;
            }
            case CENTER: {
                styleBuilder.centerAligned();
                break;
            }
            case JUSTIFY: {
                styleBuilder.justifyAligned();
                break;
            }
        }
        return styleBuilder.create();
    }

    public static Iterator<Row> getRowIterator(Sheet sheet, ExcelConfiguration configuration, boolean jumpToDataRows) {
        int columnNameLineNumber;
        Iterator iterator = configuration.isSkipEmptyLines() ? sheet.rowIterator() : new ZeroBasedRowIterator(sheet);
        if (jumpToDataRows && (columnNameLineNumber = configuration.getColumnNameLineNumber()) != 0) {
            if (iterator.hasNext()) {
                iterator.next();
            }
            for (int i = 1; i < columnNameLineNumber && iterator.hasNext(); ++i) {
                iterator.next();
            }
        }
        return iterator;
    }

    public static DefaultRow createRow(Workbook workbook, Row row, DataSetHeader header) {
        int size = header.size();
        Object[] values = new Object[size];
        Style[] styles = new Style[size];
        if (row != null) {
            for (int i = 0; i < size; ++i) {
                int columnNumber = header.getSelectItem(i).getColumn().getColumnNumber();
                ColumnType columnType = header.getSelectItem(i).getColumn().getType();
                Cell cell = row.getCell(columnNumber);
                Object value = columnType.equals(DefaultSpreadsheetReaderDelegate.DEFAULT_COLUMN_TYPE) || columnType.equals(DefaultSpreadsheetReaderDelegate.LEGACY_COLUMN_TYPE) ? ExcelUtils.getCellValue(workbook, cell) : ExcelUtils.evaluateCell(workbook, cell, columnType);
                Style style = ExcelUtils.getCellStyle(workbook, cell);
                values[i] = value;
                styles[i] = style;
            }
        }
        return new DefaultRow(header, values, styles);
    }

    public static DataSet getDataSet(Workbook workbook, Sheet sheet, Table table, ExcelConfiguration configuration) {
        List<SelectItem> selectItems = table.getColumns().stream().map(SelectItem::new).collect(Collectors.toList());
        Iterator<Row> rowIterator = ExcelUtils.getRowIterator(sheet, configuration, true);
        if (!rowIterator.hasNext()) {
            FileHelper.safeClose((Object[])new Object[]{workbook});
            return new EmptyDataSet(selectItems);
        }
        XlsDataSet dataSet = new XlsDataSet(selectItems, workbook, rowIterator);
        return dataSet;
    }

    private static String getNumericCellValueAsString(CellStyle cellStyle, double cellValue) {
        short formatIndex = cellStyle.getDataFormat();
        String formatString = cellStyle.getDataFormatString();
        if (formatString == null) {
            formatString = BuiltinFormats.getBuiltinFormat((int)formatIndex);
        }
        DataFormatter formatter = new DataFormatter();
        return formatter.formatRawCellContents(cellValue, (int)formatIndex, formatString);
    }
}

