package org.epics.pvaccess.client.pvms;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.SocketException;
import java.nio.ByteBuffer;
import java.util.Timer;
import java.util.TimerTask;

import org.epics.pvaccess.PVAConstants;
import org.epics.pvaccess.impl.remote.SerializationHelper;
import org.epics.pvdata.misc.SerializeHelper;
import org.epics.pvdata.pv.Field;
import org.epics.pvdata.pv.PVField;
import org.epics.pvdata.pv.SerializableControl;

class PVMSPublisher extends PVMSCodec implements SerializableControl
{
	
	private final DatagramSocket socket;
	private final DatagramPacket packet;
	
	private final ByteBuffer buffer = ByteBuffer.allocate(PVAConstants.MAX_UDP_PACKET);
	
	private final TimerTask timerTask;
	private boolean destroyed = false;
	
	/**
	 * Outgoing (codes generated by this party) introspection registry that
	 * always serializes with FULL_WITH_ID_TYPE_CODE.
	 */
	private final OutgoingMulticastIntrospectionRegistry outgoingIR = new OutgoingMulticastIntrospectionRegistry();
	
	public PVMSPublisher(InetAddress sendAddress, int port) throws SocketException
	{
		this(sendAddress, port, null, 0);
	}
	
	public PVMSPublisher(InetAddress sendAddress, int port, Timer timer, final int keepAlivePeriod) throws SocketException
	{
		socket = new DatagramSocket();
		packet = new DatagramPacket(buffer.array(), 0, sendAddress, port);
		
		// optional feature; can not be used or manager by external code
		// by calling sendKeepAliveControlMessage() periodically 
		if (timer != null && keepAlivePeriod > 0)
		{
			timerTask = new TimerTask() {
				@Override
				public void run() {
					try {
						synchronized (this)
						{
							if (!destroyed)
								sendKeepAliveControlMessage(keepAlivePeriod);
						}
					} catch (IOException e) {
						// noop
					}
				}
			};

			timer.schedule(timerTask, 0, keepAlivePeriod * 1000);
		}
		else
			timerTask = null;
	}
	
	@Override
	public void flushSerializeBuffer() {
		// TODO Auto-generated method stub
	}

	@Override
	public void ensureBuffer(int size) {
		/*
		if (sendBuffer.remaining() >= size)
			return;
		
		// too large for buffer...
		if (maxSendPayloadSize < size)
			throw new IllegalArgumentException("requested for buffer size " + size + ", but only " + maxSendPayloadSize + " available.");
		
		while (sendBuffer.remaining() < size)
			flush(false);
			*/
	}

	/*
	public void flush(boolean lastMessageCompleted) {
		// automatic end
		endMessage(!lastMessageCompleted);
		
		sendBuffer.flip();
		
		try {
			send(sendBuffer);
		} catch (IOException e) {
			try {
				if (isOpen())
					close();
			} catch (IOException iex) {
				// noop, best-effort close
			}
			throw new ConnectionClosedException("Failed to send buffer.", e);
		}
		
		sendBuffer.clear();

		lastMessageStartPosition = -1;

		// start with last header
		if (!lastMessageCompleted && lastSegmentedMessageType != 0)
			startMessage(lastSegmentedMessageCommand, 0);
	}
			*/

	
	@Override
	public void alignBuffer(int alignment) {
		// TODO Auto-generated method stub
	}

	@Override
	public void cachedSerialize(Field field, ByteBuffer buffer) {
		outgoingIR.serialize(field, buffer, this);
	}
		
	
	
	protected synchronized void sendShutdownControlMessage() throws IOException
	{
		buffer.clear();
		
		pmsShutdownControlMessage(buffer);
		
		packet.setLength(buffer.position());
		
		socket.send(packet);
	}
	
	protected synchronized void sendKeepAliveControlMessage(int expirationTimeSec) throws IOException
	{
		buffer.clear();
		
		pmsKeepAliveControlMessage(buffer, expirationTimeSec);
		
		packet.setLength(buffer.position());
		
		socket.send(packet);
	}

	private void pmsDataMessage(ByteBuffer buffer, String topicId, String[] tags, PVField data)
	{
		int messageSeqNo = incrementMessageSeqNum();
		
		udtDataHeader(buffer, messageSeqNo, PacketPosition.SOLO);
		
		// string topicID
		SerializeHelper.serializeString(topicId, buffer, this);
		
		// string[] tags
		if (tags.length == 0)
			SerializeHelper.writeSize(0, buffer, this);
		else
		{
			SerializeHelper.writeSize(tags.length, buffer, this);
			for (String tag : tags)
				SerializeHelper.serializeString(tag, buffer, this); 
		}
		
		// Field + PVData
		if (data == null)
			SerializationHelper.serializeNullField(buffer, this);
		else
		{
			this.cachedSerialize(data.getField(), buffer);
			data.serialize(buffer, this);
		}
	}

	protected synchronized void sendData(String topicId, String[] tags, PVField data) throws IOException
	{
		buffer.clear();
		
		pmsDataMessage(buffer, topicId, tags, data);
		
		packet.setLength(buffer.position());
		
		socket.send(packet);
	}

	// TODO avoid autounboxing !!!
	// TODO cleanup
	// private final Map<String, Long> subscriptions = new HashMap<String, Long>();
	
	public void publishData(String topicId, String[] tags, PVField data) throws IOException
	{
	/*
		synchronized (subscriptions) {
			Long expirationTime = subscriptions.get(topicId);
			if (expirationTime == null || expirationTime.longValue() < System.currentTimeMillis())
				return;
		}
	 */
		sendData(topicId, tags, data);
	}
	/*
	// TODO this would requires publisher to listen (another socket, another thread)
	@Override
	protected void pmsSubscribeControlMessage(ByteBuffer buffer,
			int expirationTimeSec, String topicID) {
		
		System.out.println("subscribed to " + topicID + " for " + expirationTimeSec);
		long untilWhen = System.currentTimeMillis() + 1000 * expirationTimeSec;
		synchronized (subscriptions) {
			subscriptions.put(topicID, untilWhen);
		}
	}
	*/
	public synchronized void destroy()
	{
		if (destroyed)
			return;
		destroyed = true;
		
		if (timerTask != null)
			timerTask.cancel();
		
		try {
			sendShutdownControlMessage();
		} catch (IOException e) {
			// noop
		}

		// TODO do I need to wait a bit for shutdown message to be sent
		socket.close();
	}
}