/*
 * Decompiled with CFR 0.152.
 */
package org.evosuite.continuous.job;

import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import org.evosuite.Properties;
import org.evosuite.continuous.CtgConfiguration;
import org.evosuite.continuous.job.JobDefinition;
import org.evosuite.continuous.job.JobHandler;
import org.evosuite.continuous.persistency.StorageManager;
import org.evosuite.utils.LoggingUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JobExecutor {
    private static Logger logger = LoggerFactory.getLogger(JobExecutor.class);
    private volatile boolean executing;
    private long startTimeInMs;
    private volatile CountDownLatch latch;
    private BlockingQueue<JobDefinition> jobQueue;
    private Map<String, JobDefinition> finishedJobs;
    protected final CtgConfiguration configuration;
    private String projectClassPath;
    private StorageManager storage;

    public JobExecutor(StorageManager storage, String projectClassPath, CtgConfiguration conf) throws IllegalArgumentException {
        this.storage = storage;
        if (!storage.isStorageOk()) {
            throw new IllegalArgumentException("Storage is not initialized");
        }
        this.configuration = conf;
        this.projectClassPath = projectClassPath;
    }

    protected long getRemainingTimeInMs() {
        long elapsed = System.currentTimeMillis() - this.startTimeInMs;
        long budgetInMs = this.configuration.timeInMinutes * 60 * 1000;
        long remaining = budgetInMs - elapsed;
        return remaining;
    }

    public synchronized void executeJobs(final List<JobDefinition> jobs, final int cores) throws IllegalStateException {
        if (this.executing) {
            throw new IllegalStateException("Already executing jobs");
        }
        logger.info("Going to execute " + jobs.size() + " jobs");
        this.initExecution(jobs);
        Thread mainThread = new Thread(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void run() {
                JobHandler[] handlers;
                for (JobHandler handler : handlers = JobHandler.getPool(cores, JobExecutor.this)) {
                    handler.start();
                }
                long longestJob = -1L;
                try {
                    LoggingUtils.getEvoLogger().info("Going to execute " + jobs.size() + " jobs");
                    long minutes = JobExecutor.this.configuration.timeInMinutes;
                    LocalDateTime endBy = LocalDateTime.now().plus(minutes, ChronoUnit.MINUTES);
                    LoggingUtils.getEvoLogger().info("Estimated completion time: " + minutes + " minutes, by " + endBy);
                    longestJob = JobExecutor.this.execute(jobs);
                }
                catch (Exception e) {
                    logger.error("Error while trying to execute the " + jobs.size() + " jobs: " + e.getMessage(), e);
                }
                finally {
                    if (!this.isInterrupted() && longestJob > 0L) {
                        try {
                            JobExecutor.this.latch.await(longestJob * 2L + 60000L, TimeUnit.MILLISECONDS);
                        }
                        catch (InterruptedException e) {
                            this.interrupt();
                        }
                    }
                    long stillNotFinished = JobExecutor.this.latch.getCount();
                    int i = 0;
                    while ((long)i < stillNotFinished) {
                        JobExecutor.this.latch.countDown();
                        ++i;
                    }
                    for (JobHandler handler : handlers) {
                        handler.stopExecution();
                    }
                    JobExecutor.this.executing = false;
                }
            }
        };
        mainThread.start();
    }

    protected void initExecution(List<JobDefinition> jobs) {
        this.executing = true;
        this.startTimeInMs = System.currentTimeMillis();
        this.latch = new CountDownLatch(jobs.size());
        this.jobQueue = new ArrayBlockingQueue<JobDefinition>(1);
        this.finishedJobs = new ConcurrentHashMap<String, JobDefinition>();
    }

    protected long execute(List<JobDefinition> jobs) {
        long remaining;
        long longestJob = -1L;
        LinkedList<JobDefinition> toExecute = new LinkedList<JobDefinition>();
        toExecute.addAll(jobs);
        LinkedList<JobDefinition> postponed = new LinkedList<JobDefinition>();
        while (!(toExecute.isEmpty() && postponed.isEmpty() || (remaining = this.getRemainingTimeInMs()) <= 0L)) {
            JobDefinition chosenJob = null;
            if (!postponed.isEmpty()) {
                Iterator iterator = postponed.iterator();
                while (iterator.hasNext()) {
                    JobDefinition job = (JobDefinition)iterator.next();
                    if (!job.areDependenciesSatisfied(jobs, this.finishedJobs.keySet())) continue;
                    chosenJob = job;
                    iterator.remove();
                    break;
                }
            }
            if (chosenJob == null && toExecute.isEmpty()) {
                assert (!postponed.isEmpty());
                chosenJob = (JobDefinition)postponed.remove(0);
            }
            if (chosenJob == null) {
                assert (!toExecute.isEmpty());
                while (!toExecute.isEmpty()) {
                    JobDefinition job = (JobDefinition)toExecute.poll();
                    if (job.areDependenciesSatisfied(jobs, this.finishedJobs.keySet())) {
                        chosenJob = job;
                        break;
                    }
                    postponed.add(job);
                }
                if (chosenJob == null) {
                    assert (!postponed.isEmpty() && toExecute.isEmpty());
                    continue;
                }
            }
            assert (chosenJob != null);
            longestJob = Math.max(longestJob, (long)(chosenJob.seconds * 1000));
            try {
                this.jobQueue.offer(chosenJob, remaining, TimeUnit.MILLISECONDS);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                break;
            }
        }
        return longestJob;
    }

    public JobDefinition pollJob() throws InterruptedException {
        return this.jobQueue.take();
    }

    public void doneWithJob(JobDefinition job) {
        this.finishedJobs.put(job.cut, job);
        this.latch.countDown();
        LoggingUtils.getEvoLogger().info("Completed job. Left: " + this.latch.getCount());
    }

    public void waitForJobs() {
        try {
            if (Properties.CTG_DEBUG_PORT != null) {
                this.latch.await();
            } else {
                boolean elapsed;
                boolean bl = elapsed = !this.latch.await(this.configuration.timeInMinutes + 1, TimeUnit.MINUTES);
                if (elapsed) {
                    logger.error("The jobs did not finish in time");
                }
            }
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    public String getProjectClassPath() {
        return this.projectClassPath;
    }

    public StorageManager getStorage() {
        return this.storage;
    }
}

