/*
 * Decompiled with CFR 0.152.
 */
package org.firebirdsql.gds.ng.wire.auth;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.ServiceConfigurationError;
import java.util.ServiceLoader;
import java.util.Set;
import java.util.regex.Pattern;
import org.firebirdsql.gds.ClumpletReader;
import org.firebirdsql.gds.ConnectionParameterBuffer;
import org.firebirdsql.gds.ParameterTagMapping;
import org.firebirdsql.gds.ng.FbExceptionBuilder;
import org.firebirdsql.gds.ng.IAttachProperties;
import org.firebirdsql.gds.ng.wire.auth.AuthenticationPlugin;
import org.firebirdsql.gds.ng.wire.auth.AuthenticationPluginSpi;
import org.firebirdsql.logging.Logger;
import org.firebirdsql.logging.LoggerFactory;

public final class ClientAuthBlock {
    private static final Logger log = LoggerFactory.getLogger(ClientAuthBlock.class);
    private static final Pattern AUTH_PLUGIN_LIST_SPLIT = Pattern.compile("[ \t,;]+");
    private static final String DEFAULT_AUTH_PLUGINS = "Srp256,Srp";
    private static final Map<String, AuthenticationPluginSpi> PLUGIN_MAPPING = ClientAuthBlock.getAvailableAuthenticationPlugins();
    private final IAttachProperties<?> attachProperties;
    private List<AuthenticationPluginSpi> pluginProviders;
    private final Set<String> serverPlugins = new LinkedHashSet<String>();
    private AuthenticationPlugin currentPlugin;
    private boolean authComplete;
    private boolean firstTime = true;

    public ClientAuthBlock(IAttachProperties<?> attachProperties) throws SQLException {
        this.attachProperties = attachProperties;
        this.resetClient(null);
    }

    public String getLogin() {
        return this.attachProperties.getUser();
    }

    public String getNormalizedLogin() {
        return ClientAuthBlock.normalizeLogin(this.getLogin());
    }

    public String getPassword() {
        return this.attachProperties.getPassword();
    }

    public boolean isAuthComplete() {
        return this.authComplete;
    }

    public void setAuthComplete(boolean authComplete) {
        this.authComplete = authComplete;
    }

    public String getCurrentPluginName() {
        return this.currentPlugin != null ? this.currentPlugin.getName() : null;
    }

    public String getPluginNames() {
        return ClientAuthBlock.getPluginNames(this.pluginProviders);
    }

    public byte[] getClientData() {
        return this.currentPlugin != null ? this.currentPlugin.getClientData() : null;
    }

    public void setFirstTime(boolean firstTime) {
        this.firstTime = firstTime;
    }

    public boolean isFirstTime() {
        return this.firstTime;
    }

    public void authenticateStep0() throws SQLException {
        Iterator<AuthenticationPluginSpi> providerIterator = this.pluginProviders.iterator();
        while (providerIterator.hasNext()) {
            AuthenticationPluginSpi provider = providerIterator.next();
            AuthenticationPlugin plugin = provider.createPlugin();
            log.debug("Trying authentication plugin " + plugin);
            try {
                switch (plugin.authenticate(this)) {
                    case AUTH_SUCCESS: 
                    case AUTH_MORE_DATA: {
                        this.currentPlugin = plugin;
                        return;
                    }
                    case AUTH_CONTINUE: {
                        providerIterator.remove();
                    }
                }
            }
            catch (SQLException ex) {
                throw new FbExceptionBuilder().exception(335544472).cause(ex).toFlatSQLException();
            }
        }
    }

    public void resetClient(byte[] serverInfo) throws SQLException {
        if (serverInfo != null) {
            if (this.currentPlugin != null && this.currentPlugin.hasServerData()) {
                return;
            }
            ClumpletReader serverList = new ClumpletReader(ClumpletReader.Kind.UnTagged, serverInfo);
            if (serverList.find(2)) {
                String serverPluginNames = serverList.getString(StandardCharsets.US_ASCII);
                this.serverPlugins.clear();
                this.serverPlugins.addAll(ClientAuthBlock.splitPluginList(serverPluginNames));
            }
        }
        this.firstTime = true;
        this.currentPlugin = null;
        this.pluginProviders = this.getSupportedPluginProviders();
        if (!this.serverPlugins.isEmpty()) {
            LinkedList<AuthenticationPluginSpi> mergedProviderList = new LinkedList<AuthenticationPluginSpi>();
            for (AuthenticationPluginSpi clientProvider : this.pluginProviders) {
                if (!this.serverPlugins.contains(clientProvider.getPluginName())) continue;
                mergedProviderList.add(clientProvider);
            }
            if (mergedProviderList.isEmpty()) {
                throw new FbExceptionBuilder().exception(335544472).exception(335544382).messageParameter("No matching plugins on server").toFlatSQLException();
            }
            this.pluginProviders = mergedProviderList;
        }
    }

    public void setServerData(byte[] serverData) {
        if (this.currentPlugin == null) {
            log.debug("Received server data without current plugin");
        } else {
            this.currentPlugin.setServerData(serverData);
        }
    }

    private static String getPluginNames(List<AuthenticationPluginSpi> pluginProviders) {
        if (pluginProviders.size() == 0) {
            return null;
        }
        StringBuilder names = new StringBuilder();
        for (int idx = 0; idx < pluginProviders.size(); ++idx) {
            if (idx > 0) {
                names.append(',');
            }
            names.append(pluginProviders.get(idx).getPluginName());
        }
        return names.toString();
    }

    public void writePluginDataTo(OutputStream userId) throws IOException {
        byte[] specificDataBytes;
        String pluginList;
        String pluginName;
        String user = this.getLogin();
        if (user != null) {
            byte[] loginBytes = user.getBytes(StandardCharsets.UTF_8);
            userId.write(9);
            int loginLength = Math.min(loginBytes.length, 255);
            userId.write(loginLength);
            userId.write(loginBytes, 0, loginLength);
        }
        if ((pluginName = this.getCurrentPluginName()) != null) {
            userId.write(8);
            byte[] pluginNameBytes = pluginName.getBytes(StandardCharsets.UTF_8);
            userId.write(pluginNameBytes.length);
            userId.write(pluginNameBytes, 0, pluginNameBytes.length);
        }
        if ((pluginList = this.getPluginNames()) != null) {
            userId.write(10);
            byte[] pluginListBytes = pluginList.getBytes(StandardCharsets.UTF_8);
            userId.write(pluginListBytes.length);
            userId.write(pluginListBytes, 0, pluginListBytes.length);
        }
        if (this.currentPlugin != null && (specificDataBytes = this.currentPlugin.getClientData()) != null) {
            this.addMultiPartConnectParameter(userId, 7, specificDataBytes);
        }
    }

    private void addMultiPartConnectParameter(OutputStream userId, int paramType, byte[] specificDataBytes) throws IOException {
        int remaining = specificDataBytes.length;
        int position = 0;
        int step = 0;
        while (remaining > 0) {
            userId.write(paramType);
            int toWrite = Math.min(remaining, 254);
            userId.write(toWrite + 1);
            userId.write(step++);
            userId.write(specificDataBytes, position, toWrite);
            remaining -= toWrite;
            position += toWrite;
        }
    }

    public boolean switchPlugin(String pluginName) {
        if (this.hasPlugin() && Objects.equals(this.getCurrentPluginName(), pluginName)) {
            return false;
        }
        Iterator<AuthenticationPluginSpi> iterator = this.pluginProviders.iterator();
        while (iterator.hasNext()) {
            AuthenticationPluginSpi pluginProvider = iterator.next();
            if (pluginProvider.getPluginName().equals(pluginName)) {
                this.currentPlugin = pluginProvider.createPlugin();
                return true;
            }
            iterator.remove();
        }
        return false;
    }

    public boolean hasPlugin() {
        return this.currentPlugin != null;
    }

    public AuthenticationPlugin.AuthStatus authenticate() throws SQLException {
        return this.currentPlugin.authenticate(this);
    }

    public void authFillParametersBlock(ConnectionParameterBuffer pb) throws SQLException {
        Iterator<AuthenticationPluginSpi> providerIterator = this.pluginProviders.iterator();
        while (providerIterator.hasNext()) {
            AuthenticationPluginSpi provider = providerIterator.next();
            AuthenticationPlugin plugin = this.hasPlugin() && provider.getPluginName().equals(this.getCurrentPluginName()) ? this.currentPlugin : provider.createPlugin();
            log.debug("Trying authentication plugin " + plugin);
            try {
                switch (plugin.authenticate(this)) {
                    case AUTH_SUCCESS: 
                    case AUTH_MORE_DATA: {
                        log.debug("Trying authentication plugin " + plugin + " is OK");
                        this.currentPlugin = plugin;
                        this.cleanParameterBuffer(pb);
                        this.extractDataToParameterBuffer(pb);
                        return;
                    }
                    case AUTH_CONTINUE: {
                        providerIterator.remove();
                    }
                }
            }
            catch (SQLException ex) {
                throw new FbExceptionBuilder().exception(335544472).cause(ex).toFlatSQLException();
            }
            log.debug(String.format("try next plugin, %s skipped", plugin));
        }
    }

    public boolean supportsEncryption() throws SQLException {
        if (this.currentPlugin == null) {
            throw new SQLException("No authentication plugin available");
        }
        return this.currentPlugin.generatesSessionKey();
    }

    public byte[] getSessionKey() throws SQLException {
        if (this.currentPlugin == null) {
            throw new SQLException("No authentication plugin available");
        }
        return this.currentPlugin.getSessionKey();
    }

    static String normalizeLogin(String login) {
        if (login == null || login.isEmpty()) {
            return login;
        }
        if (login.length() > 2 && login.charAt(0) == '\"' && login.charAt(login.length() - 1) == '\"') {
            return ClientAuthBlock.normalizeQuotedLogin(login);
        }
        return login.toUpperCase(Locale.ROOT);
    }

    private static String normalizeQuotedLogin(String login) {
        StringBuilder sb = new StringBuilder(login.length() - 2);
        sb.append(login, 1, login.length() - 1);
        for (int idx = 0; idx < sb.length(); ++idx) {
            if (sb.charAt(idx) != '\"') continue;
            sb.deleteCharAt(idx);
            if (idx < sb.length() && sb.charAt(idx) == '\"') {
                ++idx;
                continue;
            }
            sb.setLength(idx);
            return sb.toString();
        }
        return sb.toString();
    }

    private void extractDataToParameterBuffer(ConnectionParameterBuffer pb) {
        byte[] clientData = this.getClientData();
        if (clientData == null || clientData.length == 0) {
            return;
        }
        String pluginName = this.getCurrentPluginName();
        ParameterTagMapping tagMapping = pb.getTagMapping();
        if (this.firstTime) {
            if (pluginName != null) {
                pb.addArgument(tagMapping.getAuthPluginNameTag(), pluginName);
            }
            pb.addArgument(tagMapping.getAuthPluginListTag(), this.getPluginNames());
            this.firstTime = false;
            log.debug("first time - added plugName & pluginList");
        }
        pb.addArgument(tagMapping.getSpecificAuthDataTag(), clientData);
        log.debug(String.format("Added %d bytes of spec data with tag isc_dpb_specific_auth_data", clientData.length));
    }

    private void cleanParameterBuffer(ConnectionParameterBuffer pb) {
        ParameterTagMapping tagMapping = pb.getTagMapping();
        pb.removeArgument(tagMapping.getPasswordTag());
        pb.removeArgument(tagMapping.getEncryptedPasswordTag());
        pb.removeArgument(tagMapping.getTrustedAuthTag());
    }

    private List<AuthenticationPluginSpi> getSupportedPluginProviders() throws SQLException {
        List<String> requestedPluginNames = this.getRequestedPluginNames();
        ArrayList<AuthenticationPluginSpi> pluginProviders = new ArrayList<AuthenticationPluginSpi>(requestedPluginNames.size());
        for (String pluginName : requestedPluginNames) {
            AuthenticationPluginSpi pluginSpi = PLUGIN_MAPPING.get(pluginName);
            if (pluginSpi != null) {
                pluginProviders.add(pluginSpi);
                continue;
            }
            log.warn("No authentication plugin available with name " + pluginName);
        }
        if (pluginProviders.isEmpty()) {
            throw new FbExceptionBuilder().exception(337248287).messageParameter(requestedPluginNames.toString()).toFlatSQLException();
        }
        return pluginProviders;
    }

    private List<String> getRequestedPluginNames() {
        String pluginListString = this.attachProperties.getAuthPlugins();
        if (pluginListString == null || pluginListString.isEmpty()) {
            pluginListString = DEFAULT_AUTH_PLUGINS;
        }
        return ClientAuthBlock.splitPluginList(pluginListString);
    }

    private static List<String> splitPluginList(String pluginList) {
        return Arrays.asList(AUTH_PLUGIN_LIST_SPLIT.split(pluginList));
    }

    private static Map<String, AuthenticationPluginSpi> getAvailableAuthenticationPlugins() {
        HashMap<String, AuthenticationPluginSpi> pluginMapping = new HashMap<String, AuthenticationPluginSpi>();
        for (AuthenticationPluginSpi pluginSpi : ClientAuthBlock.getAvailableAuthenticationPluginSpis()) {
            String pluginName = pluginSpi.getPluginName();
            if (pluginMapping.containsKey(pluginName)) {
                log.warn("Authentication plugin provider for " + pluginName + " already registered. Skipping " + pluginSpi.getClass().getName());
                continue;
            }
            pluginMapping.put(pluginName, pluginSpi);
        }
        return Collections.unmodifiableMap(pluginMapping);
    }

    private static List<AuthenticationPluginSpi> getAvailableAuthenticationPluginSpis() {
        try {
            ServiceLoader<AuthenticationPluginSpi> pluginLoader = ServiceLoader.load(AuthenticationPluginSpi.class, ClientAuthBlock.class.getClassLoader());
            ArrayList<AuthenticationPluginSpi> pluginList = new ArrayList<AuthenticationPluginSpi>();
            Iterator<AuthenticationPluginSpi> pluginIterator = pluginLoader.iterator();
            while (pluginIterator.hasNext()) {
                try {
                    AuthenticationPluginSpi plugin = pluginIterator.next();
                    pluginList.add(plugin);
                }
                catch (Exception | ServiceConfigurationError e) {
                    log.warn("Can't register plugin, see debug level for more information (skipping): " + e);
                    log.debug("Failed to load plugin with exception", e);
                }
            }
            if (!pluginList.isEmpty()) {
                return pluginList;
            }
            log.warn("No authentication plugins loaded through service loader, falling back to default list");
        }
        catch (Exception e) {
            String message = "Unable to load authentication plugins through ServiceLoader, using fallback list";
            log.warn(message + ": " + e + "; see debug level for stacktrace");
            log.debug(message, e);
        }
        return ClientAuthBlock.loadFallbackPluginProviders(ClientAuthBlock.class.getClassLoader());
    }

    private static List<AuthenticationPluginSpi> loadFallbackPluginProviders(ClassLoader classLoader) {
        ArrayList<AuthenticationPluginSpi> fallbackPluginProviders = new ArrayList<AuthenticationPluginSpi>(6);
        for (String providerName : new String[]{"org.firebirdsql.gds.ng.wire.auth.legacy.LegacyAuthenticationPluginSpi", "org.firebirdsql.gds.ng.wire.auth.srp.SrpAuthenticationPluginSpi", "org.firebirdsql.gds.ng.wire.auth.srp.Srp224AuthenticationPluginSpi", "org.firebirdsql.gds.ng.wire.auth.srp.Srp256AuthenticationPluginSpi", "org.firebirdsql.gds.ng.wire.auth.srp.Srp384AuthenticationPluginSpi", "org.firebirdsql.gds.ng.wire.auth.srp.Srp512AuthenticationPluginSpi"}) {
            try {
                Class<?> clazz = Class.forName(providerName, true, classLoader);
                AuthenticationPluginSpi provider = (AuthenticationPluginSpi)clazz.getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
                fallbackPluginProviders.add(provider);
            }
            catch (ReflectiveOperationException e) {
                log.warn("Could not load plugin provider (see debug level for details) " + providerName + ", reason: " + e);
                log.debug("Failed to load plugin provider " + providerName + " with exception", e);
            }
        }
        return fallbackPluginProviders;
    }
}

