/*
 * Decompiled with CFR 0.152.
 */
package org.firebirdsql.jna.embedded.classpath;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Comparator;
import java.util.stream.Stream;
import org.firebirdsql.jna.embedded.classpath.ClasspathFirebirdEmbeddedLibrary;
import org.firebirdsql.jna.embedded.classpath.ClasspathFirebirdEmbeddedResource;
import org.firebirdsql.jna.embedded.spi.FirebirdEmbeddedLoadingException;
import org.firebirdsql.jna.embedded.spi.FirebirdEmbeddedProvider;
import org.firebirdsql.logging.Logger;
import org.firebirdsql.logging.LoggerFactory;

final class ClasspathFirebirdEmbeddedLoader {
    private static final Logger log = LoggerFactory.getLogger(ClasspathFirebirdEmbeddedLoader.class);
    private static final int LOAD_BUFFER_SIZE = 8192;
    private static final String TEMP_FOLDER_PREFIX = "firebird-embedded";
    private static final String DELETION_MARKER_SUFFIX = ".jaybird_x";
    private final FirebirdEmbeddedProvider firebirdEmbeddedProvider;
    private final ClasspathFirebirdEmbeddedResource classpathFirebirdEmbeddedResource;
    private final Path targetDirectory;
    private final Path libraryEntryPoint;

    ClasspathFirebirdEmbeddedLoader(FirebirdEmbeddedProvider firebirdEmbeddedProvider, ClasspathFirebirdEmbeddedResource classpathFirebirdEmbeddedResource) throws FirebirdEmbeddedLoadingException {
        this.firebirdEmbeddedProvider = firebirdEmbeddedProvider;
        this.classpathFirebirdEmbeddedResource = classpathFirebirdEmbeddedResource;
        ClasspathFirebirdEmbeddedLoader.cleanupOldTemporaryFiles();
        try {
            this.targetDirectory = Files.createTempDirectory(TEMP_FOLDER_PREFIX, new FileAttribute[0]);
            this.targetDirectory.toFile().deleteOnExit();
            this.libraryEntryPoint = this.getValidatedLibraryEntryPoint();
        }
        catch (IOException e) {
            throw new FirebirdEmbeddedLoadingException(this.getProviderName() + ": Could not create temporary folder for Firebird Embedded: " + String.valueOf(e), e);
        }
    }

    Path getTargetDirectory() {
        return this.targetDirectory;
    }

    Path getLibraryEntryPoint() {
        return this.libraryEntryPoint;
    }

    void install() throws FirebirdEmbeddedLoadingException {
        try {
            log.infof("Extracting Firebird Embedded %s to %s", (Object)this.firebirdEmbeddedProvider.getVersion(), (Object)this.targetDirectory);
            for (String resourceName : this.classpathFirebirdEmbeddedResource.getResourceList()) {
                this.copyResourceToTargetDirectory(resourceName);
            }
        }
        catch (IOException e) {
            throw new FirebirdEmbeddedLoadingException(this.getProviderName() + ": Could not extract Firebird Embedded to local file system: " + String.valueOf(e), e);
        }
        finally {
            try (Stream<Path> tempFiles = Files.walk(this.targetDirectory, new FileVisitOption[0]);){
                tempFiles.map(Path::toFile).forEach(File::deleteOnExit);
            }
            catch (IOException e) {
                log.warnfe("Firebird Embedded files in %s could not be marked for deletion", this.targetDirectory, e);
            }
        }
    }

    private void copyResourceToTargetDirectory(String resourceName) throws FirebirdEmbeddedLoadingException, IOException {
        Path targetFile = this.safeResolve(resourceName);
        Path fileParent = targetFile.getParent();
        if (!Files.isDirectory(fileParent, new LinkOption[0])) {
            Files.createDirectories(fileParent, new FileAttribute[0]);
        }
        log.debugf("Saving %s to %s", (Object)resourceName, (Object)targetFile);
        try (InputStream inputStream = this.firebirdEmbeddedProvider.getClass().getResourceAsStream(resourceName);){
            if (inputStream == null) {
                throw new FirebirdEmbeddedLoadingException(this.getProviderName() + ": File " + resourceName + " doesn't exist");
            }
            try (OutputStream outputStream = Files.newOutputStream(targetFile, new OpenOption[0]);){
                int len;
                byte[] buffer = new byte[8192];
                while ((len = inputStream.read(buffer)) != -1) {
                    outputStream.write(buffer, 0, len);
                }
            }
        }
    }

    private Path getValidatedLibraryEntryPoint() throws FirebirdEmbeddedLoadingException {
        return this.safeResolve(this.classpathFirebirdEmbeddedResource.getLibraryEntryPoint());
    }

    private Path safeResolve(String relativePath) throws FirebirdEmbeddedLoadingException {
        Path targetFile = this.targetDirectory.resolve(relativePath).toAbsolutePath();
        if (targetFile.startsWith(this.targetDirectory)) {
            return targetFile;
        }
        throw new FirebirdEmbeddedLoadingException(this.getProviderName() + ": File " + relativePath + " escapes the target directory");
    }

    private String getProviderName() {
        return this.firebirdEmbeddedProvider.getClass().getName();
    }

    static void dispose(ClasspathFirebirdEmbeddedLibrary classpathFirebirdEmbeddedLibrary) {
        boolean allDeleted;
        Path pathToDelete = classpathFirebirdEmbeddedLibrary.getRootPath();
        try {
            allDeleted = ClasspathFirebirdEmbeddedLoader.deletePath(pathToDelete);
        }
        catch (IOException e) {
            log.errorfe("Error deleting Firebird Embedded temporary files in %s", pathToDelete, e);
            allDeleted = false;
        }
        if (!allDeleted) {
            log.infof("Could not fully delete %s, creating deletion marker for cleanup on next run", (Object)pathToDelete);
            String deletionMarkerName = pathToDelete.getFileName().toString() + DELETION_MARKER_SUFFIX;
            Path deletionMarkerPath = pathToDelete.resolveSibling(deletionMarkerName);
            if (!Files.exists(deletionMarkerPath, new LinkOption[0])) {
                try {
                    Files.createFile(deletionMarkerPath, new FileAttribute[0]);
                }
                catch (IOException e) {
                    log.errorfe("Could not create deletion marker for %s manual cleanup will be necessary", pathToDelete, e);
                }
            }
        }
    }

    private static boolean deletePath(Path pathToDelete) throws IOException {
        log.infof("Attempting to delete Firebird Embedded temporary files in %s", (Object)pathToDelete);
        if (!Files.exists(pathToDelete, new LinkOption[0])) {
            return true;
        }
        try (Stream<Path> filesToDelete = Files.walk(pathToDelete, new FileVisitOption[0]);){
            boolean bl = filesToDelete.sorted(Comparator.reverseOrder()).map(Path::toFile).map(File::delete).reduce(true, (a, b) -> a != false && b != false);
            return bl;
        }
    }

    private static void cleanupOldTemporaryFiles() {
        try {
            Path fileToLocateTempFolder = Files.createTempFile(TEMP_FOLDER_PREFIX, DELETION_MARKER_SUFFIX, new FileAttribute[0]);
            Files.delete(fileToLocateTempFolder);
            Path tempFolder = fileToLocateTempFolder.getParent();
            try (Stream<Path> tempFiles = Files.list(tempFolder);){
                tempFiles.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).filter(ClasspathFirebirdEmbeddedLoader::isJaybirdDeletionMarker).forEach(ClasspathFirebirdEmbeddedLoader::handleDeletionMarker);
            }
        }
        catch (IOException e) {
            log.error("Could not cleanup old Firebird Embedded temporary files", e);
        }
    }

    private static boolean isJaybirdDeletionMarker(Path path) {
        String fileName = path.getFileName().toString();
        return fileName.endsWith(DELETION_MARKER_SUFFIX) && fileName.startsWith(TEMP_FOLDER_PREFIX);
    }

    private static void handleDeletionMarker(Path deletionMarkerPath) {
        log.debugf("Handling deletion marker %s", (Object)deletionMarkerPath);
        String deletionMarkerName = deletionMarkerPath.getFileName().toString();
        String tempFolderName = deletionMarkerName.substring(0, deletionMarkerName.lastIndexOf(DELETION_MARKER_SUFFIX));
        Path tempFolder = deletionMarkerPath.resolveSibling(tempFolderName);
        try {
            if (ClasspathFirebirdEmbeddedLoader.deletePath(tempFolder)) {
                if (!deletionMarkerPath.toFile().delete()) {
                    log.debugf("Could not delete deletion marker %s", (Object)deletionMarkerPath);
                }
            } else {
                log.debugf("Could not fully delete %s", (Object)tempFolder);
            }
        }
        catch (IOException e) {
            log.error("Error deleting old Firebird Embedded temporary folder", e);
        }
    }
}

