/* Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.flowable.cmmn.spring;

import org.flowable.cmmn.engine.CmmnEngine;
import org.flowable.cmmn.engine.CmmnEngineConfiguration;
import org.flowable.common.engine.impl.cfg.SpringBeanFactoryProxyMap;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

/**
 * @author Dave Syer
 * @author Tijs Rademakers
 * @author Joram Barrez
 * @author Josh Long
 */
public class CmmnEngineFactoryBean implements FactoryBean<CmmnEngine>, DisposableBean, ApplicationContextAware {

    protected CmmnEngineConfiguration cmmnEngineConfiguration;

    protected ApplicationContext applicationContext;
    protected CmmnEngine cmmnEngine;

    @Override
    public void destroy() throws Exception {
        if (cmmnEngine != null) {
            cmmnEngine.close();
        }
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    @Override
    public CmmnEngine getObject() throws Exception {
        configureExternallyManagedTransactions();
        
        if (cmmnEngineConfiguration.getBeans() == null) {
            cmmnEngineConfiguration.setBeans(new SpringBeanFactoryProxyMap(applicationContext));
        }

        this.cmmnEngine = cmmnEngineConfiguration.buildCmmnEngine();
        return this.cmmnEngine;
    }
    
    protected void configureExternallyManagedTransactions() {
        if (cmmnEngineConfiguration instanceof SpringCmmnEngineConfiguration) { // remark: any config can be injected, so we cannot have SpringConfiguration as member
            SpringCmmnEngineConfiguration engineConfiguration = (SpringCmmnEngineConfiguration) cmmnEngineConfiguration;
            if (engineConfiguration.getTransactionManager() != null) {
                cmmnEngineConfiguration.setTransactionsExternallyManaged(true);
            }
        }
    }

    @Override
    public Class<CmmnEngine> getObjectType() {
        return CmmnEngine.class;
    }

    @Override
    public boolean isSingleton() {
        return true;
    }

    public CmmnEngineConfiguration getCmmnEngineConfiguration() {
        return cmmnEngineConfiguration;
    }

    public void setCmmnEngineConfiguration(CmmnEngineConfiguration cmmnEngineConfiguration) {
        this.cmmnEngineConfiguration = cmmnEngineConfiguration;
    }
}
