/*
 * Decompiled with CFR 0.152.
 */
package org.fluentd.logger.sender;

import java.io.BufferedOutputStream;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.net.SocketAddress;
import java.nio.ByteBuffer;
import java.util.LinkedList;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.fluentd.logger.sender.Event;
import org.fluentd.logger.sender.EventTemplate;
import org.fluentd.logger.sender.Sender;
import org.msgpack.MessagePack;
import org.msgpack.template.Template;

public class RawSocketSender
implements Sender {
    private static final Logger LOG = Logger.getLogger(RawSocketSender.class.getName());
    private MessagePack msgpack = new MessagePack();
    private SocketAddress server;
    private Socket socket;
    private String name;
    private int timeout;
    private BufferedOutputStream out;
    private ByteBuffer pendings;
    private ExponentialDelayReconnector reconnector;

    public RawSocketSender() {
        this("localhost", 24224);
    }

    public RawSocketSender(String host, int port) {
        this(host, port, 3000, 0x800000);
    }

    public RawSocketSender(String host, int port, int timeout, int bufferCapacity) {
        this.msgpack.register(Event.class, (Template)EventTemplate.INSTANCE);
        this.pendings = ByteBuffer.allocate(bufferCapacity);
        this.server = new InetSocketAddress(host, port);
        this.name = String.format("%s{host=%s,port=%d,timeout=%d,bufCap=%d}", this.getClass().getName(), host, port, timeout, bufferCapacity);
        this.reconnector = new ExponentialDelayReconnector();
        this.open();
    }

    private void open() {
        try {
            this.connect();
        }
        catch (IOException e) {
            LOG.severe("Failed to connect fluentd: " + this.server.toString());
            LOG.severe("Connection will be retried");
            e.printStackTrace();
            this.close();
        }
    }

    private void connect() throws IOException {
        try {
            this.socket = new Socket();
            this.socket.connect(this.server);
            this.socket.setSoTimeout(this.timeout);
            this.out = new BufferedOutputStream(this.socket.getOutputStream());
            this.reconnector.clearErrorHistory();
        }
        catch (IOException e) {
            this.reconnector.addErrorHistory(System.currentTimeMillis());
            throw e;
        }
    }

    private void reconnect() throws IOException {
        if (this.socket == null) {
            this.connect();
        } else if (this.socket.isClosed() || !this.socket.isConnected()) {
            this.close();
            this.connect();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() {
        if (this.out != null) {
            try {
                this.out.close();
            }
            catch (IOException e) {
            }
            finally {
                this.out = null;
            }
        }
        if (this.socket != null) {
            try {
                this.socket.close();
            }
            catch (IOException iOException) {
            }
            finally {
                this.socket = null;
            }
        }
    }

    @Override
    public boolean emit(String tag, Map<String, Object> data) {
        return this.emit(tag, System.currentTimeMillis() / 1000L, data);
    }

    @Override
    public boolean emit(String tag, long timestamp, Map<String, Object> data) {
        return this.emit(new Event(tag, timestamp, data));
    }

    protected boolean emit(Event event) {
        if (LOG.isLoggable(Level.FINE)) {
            LOG.fine(String.format("Created %s", event));
        }
        byte[] bytes = null;
        try {
            bytes = this.msgpack.write((Object)event);
        }
        catch (IOException e) {
            LOG.severe("Cannot serialize event: " + event);
            e.printStackTrace();
            return false;
        }
        return this.send(bytes);
    }

    private synchronized boolean send(byte[] bytes) {
        if (this.pendings.position() + bytes.length > this.pendings.capacity()) {
            LOG.severe("Cannot send logs to " + this.server.toString());
            return false;
        }
        this.pendings.put(bytes);
        try {
            if (!this.reconnector.enableReconnection(System.currentTimeMillis())) {
                return true;
            }
            this.flush();
        }
        catch (IOException e) {
            this.close();
        }
        return true;
    }

    @Override
    public synchronized void flush() throws IOException {
        this.reconnect();
        this.out.write(this.getBuffer());
        this.out.flush();
        this.clearBuffer();
    }

    @Override
    public byte[] getBuffer() {
        int len = this.pendings.position();
        this.pendings.position(0);
        byte[] ret = new byte[len];
        this.pendings.get(ret, 0, len);
        return ret;
    }

    void clearBuffer() {
        this.pendings.clear();
    }

    public String toString() {
        return this.name;
    }

    private static class ExponentialDelayReconnector {
        private double wait = 0.5;
        private double waitIncrRate = 1.5;
        private double waitMax = 60.0;
        private int waitMaxCount = this.getWaitMaxCount();
        private LinkedList<Long> errorHistory = new LinkedList();

        private int getWaitMaxCount() {
            double r = this.waitMax / this.wait;
            for (int j = 1; j <= 100; ++j) {
                if (r < this.waitIncrRate) {
                    return j + 1;
                }
                r /= this.waitIncrRate;
            }
            return 100;
        }

        public void addErrorHistory(long timestamp) {
            this.errorHistory.addLast(timestamp);
            if (this.errorHistory.size() > this.waitMaxCount) {
                this.errorHistory.removeFirst();
            }
        }

        public void clearErrorHistory() {
            this.errorHistory.clear();
        }

        public boolean enableReconnection(long timestamp) {
            int size = this.errorHistory.size();
            if (size == 0) {
                return true;
            }
            double suppressSec = size < this.waitMaxCount ? this.wait * Math.pow(this.waitIncrRate, size - 1) : this.waitMax;
            return !((double)(timestamp - this.errorHistory.getLast()) < suppressSec);
        }
    }
}

