package com.apple.foundationdb;

/**
 * A set of options that can be set on a {@link Transaction}.
 */
public class TransactionOptions extends OptionsSet {
	public TransactionOptions( OptionConsumer consumer ) { super(consumer); }

	/**
	 * The transaction, if not self-conflicting, may be committed a second time after commit succeeds, in the event of a fault.
	 */
	public void setCausalWriteRisky() { setOption(10); }

	/**
	 * The read version will be committed, and usually will be the latest committed, but might not be the latest committed in the event of a fault or partition.
	 */
	public void setCausalReadRisky() { setOption(20); }

	/**
	 * The next write performed on this transaction will not generate a write conflict range. As a result, other transactions which read the key(s) being modified by the next write will not conflict with this transaction. Care needs to be taken when using this option on a transaction that is shared between multiple threads. When setting this option, write conflict ranges will be disabled on the next write operation, regardless of what thread it is on.
	 */
	public void setNextWriteNoWriteConflictRange() { setOption(30); }

	/**
	 * Reads performed by a transaction will not see any prior mutations that occured in that transaction, instead seeing the value which was in the database at the transaction's read version. This option may provide a small performance benefit for the client, but also disables a number of client-side optimizations which are beneficial for transactions which tend to read and write the same keys within a single transaction.
	 */
	public void setReadYourWritesDisable() { setOption(51); }

	/**
	 * Deprecated.
	 */
	@Deprecated
	public void setReadAheadDisable() { setOption(52); }

	/**
	 * Deprecated.
	 */
	@Deprecated
	public void setDurabilityDevNullIsWebScale() { setOption(130); }

	/**
	 * Specifies that this transaction should be treated as highest priority and that lower priority transactions should block behind this one. Use is discouraged outside of low-level tools.
	 */
	public void setPrioritySystemImmediate() { setOption(200); }

	/**
	 * Specifies that this transaction should be treated as low priority and that default priority transactions should be processed first. Useful for doing batch work simultaneously with latency-sensitive work.
	 */
	public void setPriorityBatch() { setOption(201); }

	/**
	 * This is a write-only transaction which sets the initial configuration. This option is designed for use by database system tools only.
	 */
	public void setInitializeNewDatabase() { setOption(300); }

	/**
	 * Allows this transaction to read and modify system keys (those that start with the byte 0xFF).
	 */
	public void setAccessSystemKeys() { setOption(301); }

	/**
	 * Allows this transaction to read system keys (those that start with the byte 0xFF).
	 */
	public void setReadSystemKeys() { setOption(302); }

	/**
	 * Enables tracing for this transaction and logs results to the client trace logs. Client trace logging must be enabled to get log output.
	 * 
	 * @param value String identifier to be used in the logs when tracing this transaction. The identifier must not exceed 100 characters.
	 */
	public void setTransactionLoggingEnable(String value) { setOption(402, value); }

	/**
	 * Set a timeout in milliseconds which, when elapsed, will cause the transaction automatically to be cancelled. Valid parameter values are {@code [0, INT_MAX]}. If set to 0, will disable all timeouts. All pending and any future uses of the transaction will throw an exception. The transaction can be used again after it is reset. Like all transaction options, a timeout must be reset after a call to onError. This behavior allows the user to make the timeout dynamic.
	 * 
	 * @param value value in milliseconds of timeout
	 */
	public void setTimeout(long value) { setOption(500, value); }

	/**
	 * Set a maximum number of retries after which additional calls to onError will throw the most recently seen error code. Valid parameter values are {@code [-1, INT_MAX]}. If set to -1, will disable the retry limit. Like all transaction options, the retry limit must be reset after a call to onError. This behavior allows the user to make the retry limit dynamic.
	 * 
	 * @param value number of times to retry
	 */
	public void setRetryLimit(long value) { setOption(501, value); }

	/**
	 * Set the maximum amount of backoff delay incurred in the call to onError if the error is retryable. Defaults to 1000 ms. Valid parameter values are {@code [0, INT_MAX]}. Like all transaction options, the maximum retry delay must be reset after a call to onError. If the maximum retry delay is less than the current retry delay of the transaction, then the current retry delay will be clamped to the maximum retry delay.
	 * 
	 * @param value value in milliseconds of maximum delay
	 */
	public void setMaxRetryDelay(long value) { setOption(502, value); }

	/**
	 * Snapshot read operations will see the results of writes done in the same transaction.
	 */
	public void setSnapshotRywEnable() { setOption(600); }

	/**
	 * Snapshot read operations will not see the results of writes done in the same transaction.
	 */
	public void setSnapshotRywDisable() { setOption(601); }

	/**
	 * The transaction can read and write to locked databases, and is resposible for checking that it took the lock.
	 */
	public void setLockAware() { setOption(700); }

	/**
	 * By default, operations that are performed on a transaction while it is being committed will not only fail themselves, but they will attempt to fail other in-flight operations (such as the commit) as well. This behavior is intended to help developers discover situations where operations could be unintentionally executed after the transaction has been reset. Setting this option removes that protection, causing only the offending operation to fail.
	 */
	public void setUsedDuringCommitProtectionDisable() { setOption(701); }

	/**
	 * The transaction can read from locked databases.
	 */
	public void setReadLockAware() { setOption(702); }

	public void setCausalReadDisable() { setOption(21); }

	public void setDurabilityDatacenter() { setOption(110); }

	public void setDurabilityRisky() { setOption(120); }

	public void setDebugRetryLogging(String value) { setOption(401, value); }
}
