/*
 * Decompiled with CFR 0.152.
 */
package com.apple.foundationdb.async.rtree;

import com.apple.foundationdb.Database;
import com.apple.foundationdb.ReadTransaction;
import com.apple.foundationdb.Transaction;
import com.apple.foundationdb.TransactionContext;
import com.apple.foundationdb.annotation.API;
import com.apple.foundationdb.annotation.SpotBugsSuppressWarnings;
import com.apple.foundationdb.async.AsyncIterator;
import com.apple.foundationdb.async.AsyncUtil;
import com.apple.foundationdb.async.rtree.ByNodeStorageAdapter;
import com.apple.foundationdb.async.rtree.BySlotStorageAdapter;
import com.apple.foundationdb.async.rtree.ChildSlot;
import com.apple.foundationdb.async.rtree.IntermediateNode;
import com.apple.foundationdb.async.rtree.ItemSlot;
import com.apple.foundationdb.async.rtree.LeafNode;
import com.apple.foundationdb.async.rtree.Node;
import com.apple.foundationdb.async.rtree.NodeHelpers;
import com.apple.foundationdb.async.rtree.NodeKind;
import com.apple.foundationdb.async.rtree.NodeSlot;
import com.apple.foundationdb.async.rtree.OnReadListener;
import com.apple.foundationdb.async.rtree.OnWriteListener;
import com.apple.foundationdb.async.rtree.StorageAdapter;
import com.apple.foundationdb.subspace.Subspace;
import com.apple.foundationdb.tuple.Tuple;
import com.apple.foundationdb.tuple.TupleHelpers;
import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.errorprone.annotations.CanIgnoreReturnValue;
import java.math.BigInteger;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Deque;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.BiPredicate;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@API(value=API.Status.EXPERIMENTAL)
public class RTree {
    private static final Logger logger = LoggerFactory.getLogger(RTree.class);
    static final byte[] rootId = new byte[16];
    public static final int MAX_CONCURRENT_READS = 16;
    public static final boolean DEFAULT_USE_NODE_SLOT_INDEX = false;
    public static final int DEFAULT_MIN_M = 16;
    public static final int DEFAULT_MAX_M = 32;
    public static final int DEFAULT_S = 2;
    @Nonnull
    public static final Storage DEFAULT_STORAGE = Storage.BY_NODE;
    public static final boolean DEFAULT_STORE_HILBERT_VALUES = true;
    @Nonnull
    public static final Config DEFAULT_CONFIG = new Config();
    @Nonnull
    private final StorageAdapter storageAdapter;
    @Nonnull
    private final Executor executor;
    @Nonnull
    private final Config config;
    @Nonnull
    private final Function<Point, BigInteger> hilbertValueFunction;
    @Nonnull
    private final Supplier<byte[]> nodeIdSupplier;
    @Nonnull
    private final OnWriteListener onWriteListener;
    @Nonnull
    private final OnReadListener onReadListener;

    public static ConfigBuilder newConfigBuilder() {
        return new ConfigBuilder();
    }

    public RTree(@Nonnull Subspace subspace, @Nonnull Subspace secondarySubspace, @Nonnull Executor executor, @Nonnull Function<Point, BigInteger> hilbertValueFunction) {
        this(subspace, secondarySubspace, executor, DEFAULT_CONFIG, hilbertValueFunction, NodeHelpers::newRandomNodeId, OnWriteListener.NOOP, OnReadListener.NOOP);
    }

    public RTree(@Nonnull Subspace subspace, @Nonnull Subspace nodeSlotIndexSubspace, @Nonnull Executor executor, @Nonnull Config config, @Nonnull Function<Point, BigInteger> hilbertValueFunction, @Nonnull Supplier<byte[]> nodeIdSupplier, @Nonnull OnWriteListener onWriteListener, @Nonnull OnReadListener onReadListener) {
        this.storageAdapter = config.getStorage().newStorageAdapter(config, subspace, nodeSlotIndexSubspace, hilbertValueFunction, onWriteListener, onReadListener);
        this.executor = executor;
        this.config = config;
        this.hilbertValueFunction = hilbertValueFunction;
        this.nodeIdSupplier = nodeIdSupplier;
        this.onWriteListener = onWriteListener;
        this.onReadListener = onReadListener;
    }

    @Nonnull
    StorageAdapter getStorageAdapter() {
        return this.storageAdapter;
    }

    @Nonnull
    public Executor getExecutor() {
        return this.executor;
    }

    @Nonnull
    public Config getConfig() {
        return this.config;
    }

    @Nonnull
    public OnWriteListener getOnWriteListener() {
        return this.onWriteListener;
    }

    @Nonnull
    public OnReadListener getOnReadListener() {
        return this.onReadListener;
    }

    @Nonnull
    public AsyncIterator<ItemSlot> scan(@Nonnull ReadTransaction readTransaction, @Nonnull Predicate<Rectangle> mbrPredicate, @Nonnull BiPredicate<Tuple, Tuple> suffixKeyPredicate) {
        return this.scan(readTransaction, null, null, mbrPredicate, suffixKeyPredicate);
    }

    @Nonnull
    public AsyncIterator<ItemSlot> scan(@Nonnull ReadTransaction readTransaction, @Nullable BigInteger lastHilbertValue, @Nullable Tuple lastKey, @Nonnull Predicate<Rectangle> mbrPredicate, @Nonnull BiPredicate<Tuple, Tuple> suffixKeyPredicate) {
        Preconditions.checkArgument(lastHilbertValue == null && lastKey == null || lastHilbertValue != null && lastKey != null);
        LeafIterator leafIterator = new LeafIterator(readTransaction, rootId, lastHilbertValue, lastKey, mbrPredicate, suffixKeyPredicate);
        return new ItemSlotIterator(leafIterator);
    }

    @Nonnull
    private CompletableFuture<TraversalState> fetchLeftmostPathToLeaf(@Nonnull ReadTransaction readTransaction, @Nonnull byte[] nodeId, @Nullable BigInteger lastHilbertValue, @Nullable Tuple lastKey, @Nonnull Predicate<Rectangle> mbrPredicate, @Nonnull BiPredicate<Tuple, Tuple> suffixPredicate) {
        AtomicReference<byte[]> currentId = new AtomicReference<byte[]>(nodeId);
        ArrayList toBeProcessed = Lists.newArrayList();
        AtomicReference<Object> leafNode = new AtomicReference<Object>(null);
        return AsyncUtil.whileTrue(() -> this.onReadListener.onAsyncRead(this.storageAdapter.fetchNode(readTransaction, (byte[])currentId.get())).thenApply(node -> {
            if (node == null) {
                if (Arrays.equals((byte[])currentId.get(), rootId)) {
                    Verify.verify(leafNode.get() == null);
                    return false;
                }
                throw new IllegalStateException("unable to fetch node for scan");
            }
            if (node.getKind() == NodeKind.INTERMEDIATE) {
                Iterable childSlots = ((IntermediateNode)node).getSlots();
                ArrayDeque<ChildSlot> toBeProcessedThisLevel = new ArrayDeque<ChildSlot>();
                Iterator iterator = childSlots.iterator();
                while (iterator.hasNext()) {
                    int hilbertValueAndKeyCompare;
                    ChildSlot childSlot = (ChildSlot)iterator.next();
                    if (lastHilbertValue != null && lastKey != null && (hilbertValueAndKeyCompare = childSlot.compareLargestHilbertValueAndKey(lastHilbertValue, lastKey)) < 0) continue;
                    if (!mbrPredicate.test(childSlot.getMbr())) {
                        this.onReadListener.onChildNodeDiscard(childSlot);
                        continue;
                    }
                    if (childSlot.suffixPredicateCanBeApplied() && !suffixPredicate.test(childSlot.getSmallestKeySuffix(), childSlot.getLargestKeySuffix())) {
                        this.onReadListener.onChildNodeDiscard(childSlot);
                        continue;
                    }
                    toBeProcessedThisLevel.addLast(childSlot);
                    iterator.forEachRemaining(toBeProcessedThisLevel::addLast);
                }
                toBeProcessed.add(toBeProcessedThisLevel);
                ChildSlot nextChildSlot = RTree.resolveNextIdForFetch(toBeProcessed, mbrPredicate, suffixPredicate, this.onReadListener);
                if (nextChildSlot == null) {
                    return false;
                }
                currentId.set(Objects.requireNonNull(nextChildSlot.getChildId()));
                return true;
            }
            leafNode.set(((LeafNode)node));
            return false;
        }), this.executor).thenApply(vignore -> leafNode.get() == null ? TraversalState.end() : TraversalState.of(toBeProcessed, (LeafNode)leafNode.get()));
    }

    @Nonnull
    private CompletableFuture<TraversalState> fetchNextPathToLeaf(@Nonnull ReadTransaction readTransaction, @Nonnull TraversalState traversalState, @Nullable BigInteger lastHilbertValue, @Nullable Tuple lastKey, @Nonnull Predicate<Rectangle> mbrPredicate, @Nonnull BiPredicate<Tuple, Tuple> suffixPredicate) {
        List<Deque<ChildSlot>> toBeProcessed = traversalState.getToBeProcessed();
        AtomicReference<Object> leafNode = new AtomicReference<Object>(null);
        return AsyncUtil.whileTrue(() -> {
            ChildSlot nextChildSlot = RTree.resolveNextIdForFetch(toBeProcessed, mbrPredicate, suffixPredicate, this.onReadListener);
            if (nextChildSlot == null) {
                return AsyncUtil.READY_FALSE;
            }
            return this.fetchLeftmostPathToLeaf(readTransaction, nextChildSlot.getChildId(), lastHilbertValue, lastKey, mbrPredicate, suffixPredicate).thenApply(nestedTraversalState -> {
                if (nestedTraversalState.isEnd()) {
                    return true;
                }
                leafNode.set(nestedTraversalState.getCurrentLeafNode());
                toBeProcessed.addAll(nestedTraversalState.getToBeProcessed());
                return false;
            });
        }, this.executor).thenApply(v -> leafNode.get() == null ? TraversalState.end() : TraversalState.of(toBeProcessed, (LeafNode)leafNode.get()));
    }

    @Nullable
    private static ChildSlot resolveNextIdForFetch(@Nonnull List<Deque<ChildSlot>> toBeProcessed, @Nonnull Predicate<Rectangle> mbrPredicate, @Nonnull BiPredicate<Tuple, Tuple> suffixPredicate, @Nonnull OnReadListener onReadListener) {
        for (int level = toBeProcessed.size() - 1; level >= 0; --level) {
            Deque<ChildSlot> toBeProcessedThisLevel = toBeProcessed.get(level);
            while (!toBeProcessedThisLevel.isEmpty()) {
                ChildSlot childSlot = toBeProcessedThisLevel.pollFirst();
                if (!mbrPredicate.test(childSlot.getMbr())) {
                    onReadListener.onChildNodeDiscard(childSlot);
                    continue;
                }
                if (childSlot.suffixPredicateCanBeApplied() && !suffixPredicate.test(childSlot.getSmallestKeySuffix(), childSlot.getLargestKeySuffix())) {
                    onReadListener.onChildNodeDiscard(childSlot);
                    continue;
                }
                toBeProcessed.subList(level + 1, toBeProcessed.size()).clear();
                return childSlot;
            }
        }
        return null;
    }

    @Nonnull
    public CompletableFuture<Void> insertOrUpdate(@Nonnull TransactionContext tc, @Nonnull Point point, @Nonnull Tuple keySuffix, @Nonnull Tuple value) {
        BigInteger hilbertValue = this.hilbertValueFunction.apply(point);
        Tuple itemKey = Tuple.from(point.getCoordinates(), keySuffix);
        return tc.runAsync(transaction -> this.fetchPathForModification((Transaction)transaction, hilbertValue, itemKey, true).thenCompose(leafNode -> {
            if (leafNode == null) {
                leafNode = new LeafNode(rootId, Lists.newArrayList());
            }
            return this.insertOrUpdateSlot((Transaction)transaction, (LeafNode)leafNode, point, hilbertValue, itemKey, value);
        }));
    }

    @Nonnull
    private CompletableFuture<Void> insertOrUpdateSlot(@Nonnull Transaction transaction, @Nonnull LeafNode targetNode, @Nonnull Point point, @Nonnull BigInteger hilbertValue, @Nonnull Tuple key, @Nonnull Tuple value) {
        Verify.verify(targetNode.size() <= this.config.getMaxM());
        AtomicInteger level = new AtomicInteger(0);
        ItemSlot newSlot = new ItemSlot(hilbertValue, point, key, value);
        AtomicInteger insertSlotIndex = new AtomicInteger(RTree.findInsertUpdateItemSlotIndex(targetNode, hilbertValue, key));
        if (insertSlotIndex.get() < 0) {
            this.storageAdapter.writeLeafNodeSlot(transaction, targetNode, newSlot);
            return AsyncUtil.DONE;
        }
        AtomicReference<LeafNode> currentNode = new AtomicReference<LeafNode>(targetNode);
        AtomicReference<ItemSlot> parentSlot = new AtomicReference<ItemSlot>(newSlot);
        return AsyncUtil.whileTrue(() -> {
            NodeSlot currentNewSlot = (NodeSlot)parentSlot.get();
            if (currentNewSlot != null) {
                return this.insertSlotIntoTargetNode(transaction, level.get(), hilbertValue, key, (Node)currentNode.get(), currentNewSlot, insertSlotIndex.get()).thenApply(nodeOrAdjust -> {
                    if (((Node)currentNode.get()).isRoot()) {
                        return false;
                    }
                    currentNode.set((LeafNode)((Object)((Node)currentNode.get()).getParentNode()));
                    parentSlot.set((ItemSlot)((Object)nodeOrAdjust.getSlotInParent()));
                    insertSlotIndex.set(nodeOrAdjust.getSplitNode() == null ? -1 : nodeOrAdjust.getSplitNode().getSlotIndexInParent());
                    level.incrementAndGet();
                    return nodeOrAdjust.getSplitNode() != null || nodeOrAdjust.parentNeedsAdjustment();
                });
            }
            return this.updateSlotsAndAdjustNode(transaction, level.get(), hilbertValue, key, (Node)currentNode.get(), true).thenApply(nodeOrAdjust -> {
                Verify.verify(nodeOrAdjust.getSlotInParent() == null);
                if (((Node)currentNode.get()).isRoot()) {
                    return false;
                }
                currentNode.set((LeafNode)((Object)((Node)currentNode.get()).getParentNode()));
                level.incrementAndGet();
                return nodeOrAdjust.parentNeedsAdjustment();
            });
        }, this.executor);
    }

    @Nonnull
    private CompletableFuture<NodeOrAdjust> insertSlotIntoTargetNode(@Nonnull Transaction transaction, int level, @Nonnull BigInteger hilbertValue, @Nonnull Tuple key, @Nonnull Node targetNode, @Nonnull NodeSlot newSlot, int slotIndexInTargetNode) {
        if (targetNode.size() < this.config.getMaxM()) {
            if (logger.isTraceEnabled()) {
                logger.trace("regular insert without splitting; node={}; size={}", (Object)targetNode, (Object)targetNode.size());
            }
            targetNode.insertSlot(this.storageAdapter, level - 1, slotIndexInTargetNode, newSlot);
            if (targetNode.getKind() == NodeKind.INTERMEDIATE) {
                this.storageAdapter.writeNodes(transaction, Collections.singletonList(targetNode));
            } else {
                Verify.verify(targetNode.getKind() == NodeKind.LEAF);
                this.storageAdapter.writeLeafNodeSlot(transaction, (LeafNode)targetNode, (ItemSlot)newSlot);
            }
            if (!targetNode.isRoot()) {
                return this.fetchParentNodeIfNecessary(transaction, targetNode, level, hilbertValue, key, true).thenApply(ignored -> this.adjustSlotInParent(targetNode, level) ? NodeOrAdjust.ADJUST : NodeOrAdjust.NONE);
            }
            return CompletableFuture.completedFuture(NodeOrAdjust.NONE);
        }
        if (targetNode.isRoot()) {
            if (logger.isTraceEnabled()) {
                logger.trace("splitting root node; size={}", (Object)targetNode.size());
            }
            targetNode.insertSlot(this.storageAdapter, level - 1, slotIndexInTargetNode, newSlot);
            this.splitRootNode(transaction, level, targetNode);
            return CompletableFuture.completedFuture(NodeOrAdjust.NONE);
        }
        CompletionStage siblings = this.fetchParentNodeIfNecessary(transaction, targetNode, level, hilbertValue, key, true).thenCompose(ignored -> this.fetchSiblings(transaction, targetNode));
        return ((CompletableFuture)siblings).thenApply(siblingNodes -> {
            ArrayList<Node> newSiblingNodes;
            Node splitNode;
            int numSlots = Math.toIntExact(siblingNodes.stream().mapToLong(Node::size).sum());
            if (numSlots == siblingNodes.size() * this.config.getMaxM()) {
                if (logger.isTraceEnabled()) {
                    logger.trace("splitting node; node={}, siblings={}", (Object)targetNode, (Object)siblingNodes.stream().map(Object::toString).collect(Collectors.joining(",")));
                }
                splitNode = targetNode.newOfSameKind(this.nodeIdSupplier.get());
                splitNode.linkToParent(Objects.requireNonNull(targetNode.getParentNode()), ((Node)siblingNodes.get(siblingNodes.size() - 1)).getSlotIndexInParent() + 1);
                newSiblingNodes = Lists.newArrayList(siblingNodes);
                newSiblingNodes.add(splitNode);
            } else {
                if (logger.isTraceEnabled()) {
                    logger.trace("handling overflow; node={}, numSlots={}, siblings={}", targetNode, numSlots, siblingNodes.stream().map(Object::toString).collect(Collectors.joining(",")));
                }
                splitNode = null;
                newSiblingNodes = siblingNodes;
            }
            targetNode.insertSlot(this.storageAdapter, level - 1, slotIndexInTargetNode, newSlot);
            Iterator slotIterator = siblingNodes.stream().flatMap(Node::slotsStream).iterator();
            int base = ++numSlots / newSiblingNodes.size();
            int rest = numSlots % newSiblingNodes.size();
            ArrayList newNodeSlotLists = Lists.newArrayList();
            ArrayList<NodeSlot> currentNodeSlots = Lists.newArrayList();
            while (slotIterator.hasNext()) {
                NodeSlot slot = (NodeSlot)slotIterator.next();
                currentNodeSlots.add(slot);
                if (currentNodeSlots.size() != base + (rest > 0 ? 1 : 0)) continue;
                if (rest > 0) {
                    --rest;
                }
                newNodeSlotLists.add(currentNodeSlots);
                currentNodeSlots = Lists.newArrayList();
            }
            Verify.verify(newSiblingNodes.size() == newNodeSlotLists.size());
            Iterator newSiblingNodesIterator = newSiblingNodes.iterator();
            Iterator newNodeSlotsIterator = newNodeSlotLists.iterator();
            while (newSiblingNodesIterator.hasNext()) {
                Node newSiblingNode = (Node)newSiblingNodesIterator.next();
                Verify.verify(newNodeSlotsIterator.hasNext());
                List newNodeSlots = (List)newNodeSlotsIterator.next();
                newSiblingNode.moveOutAllSlots(this.storageAdapter);
                newSiblingNode.moveInSlots(this.storageAdapter, newNodeSlots);
            }
            this.storageAdapter.writeNodes(transaction, newSiblingNodes);
            for (Node siblingNode : siblingNodes) {
                this.adjustSlotInParent(siblingNode, level);
            }
            if (splitNode == null) {
                return NodeOrAdjust.ADJUST;
            }
            NodeSlot firstSlotOfSplitNode = splitNode.getSlot(0);
            NodeSlot lastSlotOfSplitNode = splitNode.getSlot(splitNode.size() - 1);
            return new NodeOrAdjust(new ChildSlot(firstSlotOfSplitNode.getSmallestHilbertValue(), firstSlotOfSplitNode.getSmallestKey(), lastSlotOfSplitNode.getLargestHilbertValue(), lastSlotOfSplitNode.getLargestKey(), splitNode.getId(), NodeHelpers.computeMbr(splitNode.getSlots())), splitNode, true);
        });
    }

    private void splitRootNode(@Nonnull Transaction transaction, int level, @Nonnull Node oldRootNode) {
        Node leftNode = oldRootNode.newOfSameKind(this.nodeIdSupplier.get());
        Node rightNode = oldRootNode.newOfSameKind(this.nodeIdSupplier.get());
        int leftSize = oldRootNode.size() / 2;
        ImmutableList<? extends NodeSlot> leftSlots = ImmutableList.copyOf(oldRootNode.getSlots(0, leftSize));
        leftNode.moveInSlots(this.storageAdapter, leftSlots);
        int rightSize = oldRootNode.size() - leftSize;
        ImmutableList<? extends NodeSlot> rightSlots = ImmutableList.copyOf(oldRootNode.getSlots(leftSize, leftSize + rightSize));
        rightNode.moveInSlots(this.storageAdapter, rightSlots);
        NodeSlot firstSlotOfLeftNode = (NodeSlot)leftSlots.get(0);
        NodeSlot lastSlotOfLeftNode = (NodeSlot)leftSlots.get(leftSlots.size() - 1);
        NodeSlot firstSlotOfRightNode = (NodeSlot)rightSlots.get(0);
        NodeSlot lastSlotOfRightNode = (NodeSlot)rightSlots.get(rightSlots.size() - 1);
        ChildSlot leftChildSlot = new ChildSlot(firstSlotOfLeftNode.getSmallestHilbertValue(), firstSlotOfLeftNode.getSmallestKey(), lastSlotOfLeftNode.getLargestHilbertValue(), lastSlotOfLeftNode.getLargestKey(), leftNode.getId(), NodeHelpers.computeMbr(leftNode.getSlots()));
        ChildSlot rightChildSlot = new ChildSlot(firstSlotOfRightNode.getSmallestHilbertValue(), firstSlotOfRightNode.getSmallestKey(), lastSlotOfRightNode.getLargestHilbertValue(), lastSlotOfRightNode.getLargestKey(), rightNode.getId(), NodeHelpers.computeMbr(rightNode.getSlots()));
        oldRootNode.moveOutAllSlots(this.storageAdapter);
        IntermediateNode newRootNode = (IntermediateNode)((IntermediateNode)new IntermediateNode(rootId).insertSlot(this.storageAdapter, level, 0, leftChildSlot)).insertSlot(this.storageAdapter, level, 1, rightChildSlot);
        this.storageAdapter.writeNodes(transaction, Lists.newArrayList(oldRootNode, newRootNode, leftNode, rightNode));
    }

    @Nonnull
    public CompletableFuture<Void> delete(@Nonnull TransactionContext tc, @Nonnull Point point, @Nonnull Tuple keySuffix) {
        BigInteger hilbertValue = this.hilbertValueFunction.apply(point);
        Tuple itemKey = Tuple.from(point.getCoordinates(), keySuffix);
        return tc.runAsync(transaction -> this.fetchPathForModification((Transaction)transaction, hilbertValue, itemKey, false).thenCompose(leafNode -> {
            if (leafNode == null) {
                return AsyncUtil.DONE;
            }
            return this.deleteSlotIfExists((Transaction)transaction, (LeafNode)leafNode, hilbertValue, itemKey);
        }));
    }

    @Nonnull
    private CompletableFuture<Void> deleteSlotIfExists(@Nonnull Transaction transaction, @Nonnull LeafNode targetNode, @Nonnull BigInteger hilbertValue, @Nonnull Tuple key) {
        Verify.verify(targetNode.size() <= this.config.getMaxM());
        AtomicInteger level = new AtomicInteger(0);
        AtomicInteger deleteSlotIndex = new AtomicInteger(RTree.findDeleteItemSlotIndex(targetNode, hilbertValue, key));
        if (deleteSlotIndex.get() < 0) {
            return AsyncUtil.DONE;
        }
        Object deleteSlot = targetNode.getSlot(deleteSlotIndex.get());
        AtomicReference<LeafNode> currentNode = new AtomicReference<LeafNode>(targetNode);
        AtomicReference parentSlot = new AtomicReference(deleteSlot);
        return AsyncUtil.whileTrue(() -> {
            NodeSlot currentDeleteSlot = (NodeSlot)parentSlot.get();
            if (currentDeleteSlot != null) {
                return this.deleteSlotFromTargetNode(transaction, level.get(), hilbertValue, key, (Node)currentNode.get(), currentDeleteSlot, deleteSlotIndex.get()).thenApply(nodeOrAdjust -> {
                    if (((Node)currentNode.get()).isRoot()) {
                        return false;
                    }
                    currentNode.set((LeafNode)((Object)((Node)currentNode.get()).getParentNode()));
                    parentSlot.set(nodeOrAdjust.getSlotInParent());
                    deleteSlotIndex.set(nodeOrAdjust.getTombstoneNode() == null ? -1 : nodeOrAdjust.getTombstoneNode().getSlotIndexInParent());
                    level.incrementAndGet();
                    return nodeOrAdjust.getTombstoneNode() != null || nodeOrAdjust.parentNeedsAdjustment();
                });
            }
            return this.updateSlotsAndAdjustNode(transaction, level.get(), hilbertValue, key, (Node)currentNode.get(), false).thenApply(nodeOrAdjust -> {
                Verify.verify(nodeOrAdjust.getSlotInParent() == null);
                if (((Node)currentNode.get()).isRoot()) {
                    return false;
                }
                currentNode.set((LeafNode)((Object)((Node)currentNode.get()).getParentNode()));
                level.incrementAndGet();
                return nodeOrAdjust.parentNeedsAdjustment();
            });
        }, this.executor);
    }

    @Nonnull
    private CompletableFuture<NodeOrAdjust> deleteSlotFromTargetNode(@Nonnull Transaction transaction, int level, BigInteger hilbertValue, Tuple key, @Nonnull Node targetNode, @Nonnull NodeSlot deleteSlot, int slotIndexInTargetNode) {
        if (targetNode.isRoot() || targetNode.size() > this.config.getMinM()) {
            if (logger.isTraceEnabled()) {
                logger.trace("regular delete; node={}; size={}", (Object)targetNode, (Object)targetNode.size());
            }
            targetNode.deleteSlot(this.storageAdapter, level - 1, slotIndexInTargetNode);
            if (targetNode.getKind() == NodeKind.INTERMEDIATE) {
                Verify.verify(!targetNode.isRoot() || targetNode.size() >= 2);
                this.storageAdapter.writeNodes(transaction, Collections.singletonList(targetNode));
            } else {
                Verify.verify(targetNode.getKind() == NodeKind.LEAF);
                this.storageAdapter.clearLeafNodeSlot(transaction, (LeafNode)targetNode, (ItemSlot)deleteSlot);
            }
            if (!targetNode.isRoot()) {
                return this.fetchParentNodeIfNecessary(transaction, targetNode, level, hilbertValue, key, false).thenApply(ignored -> this.adjustSlotInParent(targetNode, level) ? NodeOrAdjust.ADJUST : NodeOrAdjust.NONE);
            }
            return CompletableFuture.completedFuture(NodeOrAdjust.NONE);
        }
        CompletionStage siblings = this.fetchParentNodeIfNecessary(transaction, targetNode, level, hilbertValue, key, false).thenCompose(ignored -> this.fetchSiblings(transaction, targetNode));
        return ((CompletableFuture)siblings).thenApply(siblingNodes -> {
            List newSiblingNodes;
            Node tombstoneNode;
            int numSlots = Math.toIntExact(siblingNodes.stream().mapToLong(Node::size).sum());
            if (numSlots == siblingNodes.size() * this.config.getMinM()) {
                if (logger.isTraceEnabled()) {
                    logger.trace("fusing nodes; node={}, siblings={}", (Object)targetNode, (Object)siblingNodes.stream().map(Object::toString).collect(Collectors.joining(",")));
                }
                tombstoneNode = (Node)siblingNodes.get(siblingNodes.size() - 1);
                newSiblingNodes = siblingNodes.subList(0, siblingNodes.size() - 1);
            } else {
                if (logger.isTraceEnabled()) {
                    logger.trace("handling underflow; node={}, numSlots={}, siblings={}", targetNode, numSlots, siblingNodes.stream().map(Object::toString).collect(Collectors.joining(",")));
                }
                tombstoneNode = null;
                newSiblingNodes = siblingNodes;
            }
            targetNode.deleteSlot(this.storageAdapter, level - 1, slotIndexInTargetNode);
            Iterator slotIterator = siblingNodes.stream().flatMap(Node::slotsStream).iterator();
            int base = --numSlots / newSiblingNodes.size();
            int rest = numSlots % newSiblingNodes.size();
            ArrayList newNodeSlotLists = Lists.newArrayList();
            ArrayList<NodeSlot> currentNodeSlots = Lists.newArrayList();
            while (slotIterator.hasNext()) {
                NodeSlot slot = (NodeSlot)slotIterator.next();
                currentNodeSlots.add(slot);
                if (currentNodeSlots.size() != base + (rest > 0 ? 1 : 0)) continue;
                if (rest > 0) {
                    --rest;
                }
                newNodeSlotLists.add(currentNodeSlots);
                currentNodeSlots = Lists.newArrayList();
            }
            Verify.verify(newSiblingNodes.size() == newNodeSlotLists.size());
            if (tombstoneNode != null) {
                tombstoneNode.moveOutAllSlots(this.storageAdapter);
                this.storageAdapter.writeNodes(transaction, Collections.singletonList(tombstoneNode));
            }
            Iterator newSiblingNodesIterator = newSiblingNodes.iterator();
            Iterator newNodeSlotsIterator = newNodeSlotLists.iterator();
            while (newSiblingNodesIterator.hasNext()) {
                Node newSiblingNode = (Node)newSiblingNodesIterator.next();
                Verify.verify(newNodeSlotsIterator.hasNext());
                List newNodeSlots = (List)newNodeSlotsIterator.next();
                newSiblingNode.moveOutAllSlots(this.storageAdapter);
                newSiblingNode.moveInSlots(this.storageAdapter, newNodeSlots);
            }
            IntermediateNode parentNode = Objects.requireNonNull(targetNode.getParentNode());
            if (parentNode.isRoot() && parentNode.size() == 2 && tombstoneNode != null) {
                Node toBePromotedNode = (Node)Iterables.getOnlyElement(newSiblingNodes);
                this.promoteNodeToRoot(transaction, level, parentNode, toBePromotedNode);
                return NodeOrAdjust.NONE;
            }
            this.storageAdapter.writeNodes(transaction, newSiblingNodes);
            for (Node newSiblingNode : newSiblingNodes) {
                this.adjustSlotInParent(newSiblingNode, level);
            }
            if (tombstoneNode == null) {
                return NodeOrAdjust.ADJUST;
            }
            return new NodeOrAdjust((ChildSlot)parentNode.getSlot(tombstoneNode.getSlotIndexInParent()), tombstoneNode, true);
        });
    }

    private void promoteNodeToRoot(@Nonnull Transaction transaction, int level, IntermediateNode oldRootNode, Node toBePromotedNode) {
        oldRootNode.deleteAllSlots(this.storageAdapter, level);
        ImmutableList<? extends NodeSlot> newRootSlots = ImmutableList.copyOf(toBePromotedNode.getSlots());
        toBePromotedNode.moveOutAllSlots(this.storageAdapter);
        Node newRootNode = toBePromotedNode.newOfSameKind(rootId).moveInSlots(this.storageAdapter, newRootSlots);
        this.storageAdapter.writeNodes(transaction, ImmutableList.of(oldRootNode, newRootNode, toBePromotedNode));
    }

    @Nonnull
    private CompletableFuture<NodeOrAdjust> updateSlotsAndAdjustNode(@Nonnull Transaction transaction, int level, @Nonnull BigInteger hilbertValue, @Nonnull Tuple key, @Nonnull Node targetNode, boolean isInsertUpdate) {
        this.storageAdapter.writeNodes(transaction, Collections.singletonList(targetNode));
        if (targetNode.isRoot()) {
            return CompletableFuture.completedFuture(NodeOrAdjust.NONE);
        }
        return this.fetchParentNodeIfNecessary(transaction, targetNode, level, hilbertValue, key, isInsertUpdate).thenApply(ignored -> this.adjustSlotInParent(targetNode, level) ? NodeOrAdjust.ADJUST : NodeOrAdjust.NONE);
    }

    private boolean adjustSlotInParent(@Nonnull Node targetNode, int level) {
        Preconditions.checkArgument(!targetNode.isRoot());
        IntermediateNode parentNode = Objects.requireNonNull(targetNode.getParentNode());
        int slotIndexInParent = targetNode.getSlotIndexInParent();
        ChildSlot childSlot = (ChildSlot)parentNode.getSlot(slotIndexInParent);
        Rectangle newMbr = NodeHelpers.computeMbr(targetNode.getSlots());
        boolean slotHasChanged = !childSlot.getMbr().equals(newMbr);
        NodeSlot firstSlotOfTargetNode = targetNode.getSlot(0);
        slotHasChanged |= !childSlot.getSmallestHilbertValue().equals(firstSlotOfTargetNode.getSmallestHilbertValue());
        slotHasChanged |= !childSlot.getSmallestKey().equals(firstSlotOfTargetNode.getSmallestKey());
        NodeSlot lastSlotOfTargetNode = targetNode.getSlot(targetNode.size() - 1);
        slotHasChanged |= !childSlot.getLargestHilbertValue().equals(lastSlotOfTargetNode.getLargestHilbertValue());
        if (slotHasChanged |= !childSlot.getLargestKey().equals(lastSlotOfTargetNode.getLargestKey())) {
            parentNode.updateSlot(this.storageAdapter, level, slotIndexInParent, new ChildSlot(firstSlotOfTargetNode.getSmallestHilbertValue(), firstSlotOfTargetNode.getSmallestKey(), lastSlotOfTargetNode.getLargestHilbertValue(), lastSlotOfTargetNode.getLargestKey(), childSlot.getChildId(), newMbr));
        }
        return slotHasChanged;
    }

    @Nonnull
    private CompletableFuture<LeafNode> fetchPathForModification(@Nonnull Transaction transaction, @Nonnull BigInteger hilbertValue, @Nonnull Tuple key, boolean isInsertUpdate) {
        if (this.config.isUseNodeSlotIndex()) {
            return this.scanIndexAndFetchLeafNode(transaction, hilbertValue, key, isInsertUpdate);
        }
        return this.fetchUpdatePathToLeaf(transaction, hilbertValue, key, isInsertUpdate);
    }

    @Nonnull
    private CompletableFuture<LeafNode> scanIndexAndFetchLeafNode(@Nonnull ReadTransaction transaction, @Nonnull BigInteger hilbertValue, @Nonnull Tuple key, boolean isInsertUpdate) {
        return this.storageAdapter.scanNodeIndexAndFetchNode(transaction, 0, hilbertValue, key, isInsertUpdate).thenApply(node -> {
            Verify.verify(node == null || node.getKind() == NodeKind.LEAF && node instanceof LeafNode);
            return (LeafNode)node;
        });
    }

    @Nonnull
    private CompletableFuture<IntermediateNode> scanIndexAndFetchIntermediateNode(@Nonnull ReadTransaction transaction, int level, @Nonnull BigInteger hilbertValue, @Nonnull Tuple key, boolean isInsertUpdate) {
        Verify.verify(level > 0);
        return this.storageAdapter.scanNodeIndexAndFetchNode(transaction, level, hilbertValue, key, isInsertUpdate).thenApply(node -> {
            Verify.verify(node.getKind() == NodeKind.INTERMEDIATE && node instanceof IntermediateNode);
            return (IntermediateNode)node;
        });
    }

    @Nonnull
    private CompletableFuture<IntermediateNode> fetchParentNodeIfNecessary(@Nonnull ReadTransaction transaction, @Nonnull Node node, int level, @Nonnull BigInteger hilbertValue, @Nonnull Tuple key, boolean isInsertUpdate) {
        Verify.verify(!node.isRoot());
        IntermediateNode linkedParentNode = node.getParentNode();
        if (linkedParentNode != null) {
            return CompletableFuture.completedFuture(linkedParentNode);
        }
        Verify.verify(this.getConfig().isUseNodeSlotIndex());
        return this.scanIndexAndFetchIntermediateNode(transaction, level + 1, hilbertValue, key, isInsertUpdate).thenApply(parentNode -> {
            int slotIndexInParent = RTree.findChildSlotIndex(parentNode, node.getId());
            Verify.verify(slotIndexInParent >= 0);
            node.linkToParent((IntermediateNode)parentNode, slotIndexInParent);
            return parentNode;
        });
    }

    @Nonnull
    private CompletableFuture<LeafNode> fetchUpdatePathToLeaf(@Nonnull Transaction transaction, @Nonnull BigInteger hilbertValue, @Nonnull Tuple key, boolean isInsertUpdate) {
        AtomicReference<Object> parentNode = new AtomicReference<Object>(null);
        AtomicInteger slotInParent = new AtomicInteger(-1);
        AtomicReference<byte[]> currentId = new AtomicReference<byte[]>(rootId);
        AtomicReference<Object> leafNode = new AtomicReference<Object>(null);
        return AsyncUtil.whileTrue(() -> this.storageAdapter.fetchNode(transaction, (byte[])currentId.get()).thenApply(node -> {
            if (node == null) {
                if (Arrays.equals((byte[])currentId.get(), rootId)) {
                    Verify.verify(leafNode.get() == null);
                    return false;
                }
                throw new IllegalStateException("unable to fetch node for insert or update");
            }
            if (parentNode.get() != null) {
                node.linkToParent((IntermediateNode)parentNode.get(), slotInParent.get());
            }
            if (node.getKind() == NodeKind.INTERMEDIATE) {
                IntermediateNode intermediateNode = (IntermediateNode)node;
                int slotIndex = RTree.findChildSlotIndex(intermediateNode, hilbertValue, key, isInsertUpdate);
                if (slotIndex < 0) {
                    Verify.verify(!isInsertUpdate);
                    return false;
                }
                parentNode.set(intermediateNode);
                slotInParent.set(slotIndex);
                ChildSlot childSlot = (ChildSlot)intermediateNode.getSlot(slotIndex);
                currentId.set(childSlot.getChildId());
                return true;
            }
            leafNode.set(((LeafNode)node));
            return false;
        }), this.executor).thenApply(ignored -> {
            LeafNode node = (LeafNode)leafNode.get();
            if (logger.isTraceEnabled()) {
                logger.trace("update path; path={}", (Object)NodeHelpers.nodeIdPath(node));
            }
            return node;
        });
    }

    @Nonnull
    private CompletableFuture<List<Node>> fetchSiblings(@Nonnull Transaction transaction, @Nonnull Node node) {
        ArrayDeque<byte[]> toBeProcessed = new ArrayDeque<byte[]>();
        ArrayList working = Lists.newArrayList();
        int numSiblings = this.config.getSplitS();
        Node[] siblings = new Node[numSiblings];
        IntermediateNode parentNode = Objects.requireNonNull(node.getParentNode());
        int slotIndexInParent = node.getSlotIndexInParent();
        int start = slotIndexInParent - numSiblings / 2;
        int end = start + numSiblings;
        if (start < 0) {
            start = 0;
            end = numSiblings;
        } else if (end > parentNode.size()) {
            end = parentNode.size();
            start = end - numSiblings;
        }
        int minSibling = start;
        for (int i = start; i < end; ++i) {
            toBeProcessed.addLast(((ChildSlot)parentNode.getSlot(i)).getChildId());
        }
        return AsyncUtil.whileTrue(() -> {
            working.removeIf(CompletableFuture::isDone);
            while (working.size() <= 16) {
                int index = numSiblings - toBeProcessed.size();
                byte[] currentId = (byte[])toBeProcessed.pollFirst();
                if (currentId == null) break;
                int slotIndex = minSibling + index;
                if (slotIndex != slotIndexInParent) {
                    working.add(this.storageAdapter.fetchNode(transaction, currentId).thenAccept(siblingNode -> {
                        Objects.requireNonNull(siblingNode);
                        siblingNode.linkToParent(parentNode, slotIndex);
                        siblings[index] = siblingNode;
                    }));
                    continue;
                }
                siblings[index] = node;
            }
            if (working.isEmpty()) {
                return AsyncUtil.READY_FALSE;
            }
            return AsyncUtil.whenAny(working).thenApply(v -> true);
        }, this.executor).thenApply(vignore -> Lists.newArrayList(siblings));
    }

    public int depth(@Nonnull TransactionContext transactionContext) {
        Node node = transactionContext.run(tr -> this.fetchUpdatePathToLeaf((Transaction)tr, BigInteger.ONE, new Tuple(), true).join());
        if (node == null) {
            logger.trace("R-tree is empty.");
            return 0;
        }
        int numLevels = 1;
        while (node.getParentNode() != null) {
            ++numLevels;
            node = node.getParentNode();
        }
        Verify.verify(node.isRoot(), "end of update path should be the root", new Object[0]);
        logger.trace("numLevels = {}", (Object)numLevels);
        return numLevels;
    }

    public void validate(@Nonnull Database db) {
        this.validate(db, Integer.MAX_VALUE);
    }

    public void validate(@Nonnull Database db, int maxNumNodesToBeValidated) {
        ArrayDeque<ValidationTraversalState> toBeProcessed = new ArrayDeque<ValidationTraversalState>();
        toBeProcessed.addLast(new ValidationTraversalState(this.depth(db) - 1, null, rootId));
        while (!toBeProcessed.isEmpty()) {
            db.run(tr -> this.validate((Transaction)tr, maxNumNodesToBeValidated, toBeProcessed).join());
        }
    }

    @Nonnull
    private CompletableFuture<ArrayDeque<ValidationTraversalState>> validate(@Nonnull Transaction transaction, int maxNumNodesToBeValidated, @Nonnull ArrayDeque<ValidationTraversalState> toBeProcessed) {
        AtomicInteger numNodesEnqueued = new AtomicInteger(0);
        ArrayList working = Lists.newArrayList();
        return AsyncUtil.whileTrue(() -> {
            ValidationTraversalState currentValidationTraversalState;
            Iterator workingIterator = working.iterator();
            while (workingIterator.hasNext()) {
                CompletableFuture nextFuture = (CompletableFuture)workingIterator.next();
                if (!nextFuture.isDone()) continue;
                toBeProcessed.addAll((Collection)nextFuture.join());
                workingIterator.remove();
            }
            while (working.size() <= 16 && numNodesEnqueued.get() < maxNumNodesToBeValidated && (currentValidationTraversalState = (ValidationTraversalState)toBeProcessed.pollFirst()) != null) {
                int slotIndexInParent;
                ChildSlot childSlotInParentNode;
                IntermediateNode parentNode = currentValidationTraversalState.getParentNode();
                int level = currentValidationTraversalState.getLevel();
                if (parentNode != null) {
                    int slotIndex;
                    ChildSlot childSlot = null;
                    for (slotIndex = 0; slotIndex < parentNode.size() && !Arrays.equals((childSlot = (ChildSlot)parentNode.getSlot(slotIndex)).getChildId(), currentValidationTraversalState.getChildId()); ++slotIndex) {
                    }
                    if (slotIndex == parentNode.size()) {
                        throw new IllegalStateException("child slot not found in parent for child node");
                    }
                    childSlotInParentNode = childSlot;
                    slotIndexInParent = slotIndex;
                } else {
                    childSlotInParentNode = null;
                    slotIndexInParent = -1;
                }
                CompletableFuture fetchedNodeFuture = this.onReadListener.onAsyncRead(((CompletableFuture)this.storageAdapter.fetchNode(transaction, currentValidationTraversalState.getChildId()).thenApply(node -> {
                    if (parentNode != null) {
                        Objects.requireNonNull(node);
                        node.linkToParent(parentNode, slotIndexInParent);
                    }
                    return node;
                })).thenCompose(childNode -> {
                    if (parentNode != null && this.getConfig().isUseNodeSlotIndex()) {
                        ChildSlot childSlot = (ChildSlot)parentNode.getSlot(slotIndexInParent);
                        return this.storageAdapter.scanNodeIndexAndFetchNode(transaction, level, childSlot.getLargestHilbertValue(), childSlot.getLargestKey(), false).thenApply(nodeFromIndex -> {
                            Objects.requireNonNull(nodeFromIndex);
                            if (!Arrays.equals(nodeFromIndex.getId(), childNode.getId())) {
                                logger.warn("corrupt node slot index at level {}, parentNode = {}", (Object)level, (Object)parentNode);
                                throw new IllegalStateException("corrupt node index");
                            }
                            return childNode;
                        });
                    }
                    return CompletableFuture.completedFuture(childNode);
                }));
                working.add(fetchedNodeFuture.thenApply(childNode -> {
                    if (childNode == null) {
                        return ImmutableList.of();
                    }
                    childNode.validate();
                    childNode.validateParentNode(parentNode, childSlotInParentNode);
                    if (childNode.getKind() == NodeKind.INTERMEDIATE) {
                        return ((IntermediateNode)childNode).getSlots().stream().map(childSlot -> new ValidationTraversalState(level - 1, (IntermediateNode)childNode, childSlot.getChildId())).collect(ImmutableList.toImmutableList());
                    }
                    return ImmutableList.of();
                }));
                numNodesEnqueued.addAndGet(1);
            }
            if (working.isEmpty()) {
                return AsyncUtil.READY_FALSE;
            }
            return AsyncUtil.whenAny(working).thenApply(v -> true);
        }, this.executor).thenApply(vignore -> toBeProcessed);
    }

    private static int findChildSlotIndex(@Nonnull IntermediateNode intermediateNode, @Nonnull BigInteger hilbertValue, @Nonnull Tuple key, boolean isInsertUpdate) {
        ChildSlot firstChildSlot;
        int compare;
        Verify.verify(!intermediateNode.isEmpty());
        if (!isInsertUpdate && (compare = NodeSlot.compareHilbertValueKeyPair((firstChildSlot = (ChildSlot)intermediateNode.getSlot(0)).getSmallestHilbertValue(), firstChildSlot.getSmallestKey(), hilbertValue, key)) > 0) {
            return -1;
        }
        for (int slotIndex = 0; slotIndex < intermediateNode.size(); ++slotIndex) {
            ChildSlot childSlot = (ChildSlot)intermediateNode.getSlot(slotIndex);
            int compare2 = NodeSlot.compareHilbertValueKeyPair(childSlot.getLargestHilbertValue(), childSlot.getLargestKey(), hilbertValue, key);
            if (compare2 < 0) continue;
            return slotIndex;
        }
        return isInsertUpdate ? intermediateNode.size() - 1 : -1;
    }

    private static int findChildSlotIndex(@Nonnull IntermediateNode parentNode, @Nonnull byte[] childId) {
        for (int slotIndex = 0; slotIndex < parentNode.size(); ++slotIndex) {
            ChildSlot childSlot = (ChildSlot)parentNode.getSlot(slotIndex);
            if (!Arrays.equals(childSlot.getChildId(), childId)) continue;
            return slotIndex;
        }
        return -1;
    }

    private static int findInsertUpdateItemSlotIndex(@Nonnull LeafNode leafNode, @Nonnull BigInteger hilbertValue, @Nonnull Tuple key) {
        for (int slotIndex = 0; slotIndex < leafNode.size(); ++slotIndex) {
            ItemSlot slot = (ItemSlot)leafNode.getSlot(slotIndex);
            int compare = NodeSlot.compareHilbertValueKeyPair(slot.getHilbertValue(), slot.getKey(), hilbertValue, key);
            if (compare == 0) {
                return -1;
            }
            if (compare <= 0) continue;
            return slotIndex;
        }
        return leafNode.size();
    }

    private static int findDeleteItemSlotIndex(@Nonnull LeafNode leafNode, @Nonnull BigInteger hilbertValue, @Nonnull Tuple key) {
        for (int slotIndex = 0; slotIndex < leafNode.size(); ++slotIndex) {
            ItemSlot slot = (ItemSlot)leafNode.getSlot(slotIndex);
            int compare = NodeSlot.compareHilbertValueKeyPair(slot.getHilbertValue(), slot.getKey(), hilbertValue, key);
            if (compare == 0) {
                return slotIndex;
            }
            if (compare <= 0) continue;
            return -1;
        }
        return -1;
    }

    @CanIgnoreReturnValue
    public static class ConfigBuilder {
        private boolean useNodeSlotIndex = false;
        private int minM = 16;
        private int maxM = 32;
        private int splitS = 2;
        @Nonnull
        private Storage storage = DEFAULT_STORAGE;
        private boolean storeHilbertValues = true;

        public ConfigBuilder() {
        }

        public ConfigBuilder(boolean useNodeSlotIndex, int minM, int maxM, int splitS, @Nonnull Storage storage, boolean storeHilbertValues) {
            this.useNodeSlotIndex = useNodeSlotIndex;
            this.minM = minM;
            this.maxM = maxM;
            this.splitS = splitS;
            this.storage = storage;
            this.storeHilbertValues = storeHilbertValues;
        }

        public int getMinM() {
            return this.minM;
        }

        public ConfigBuilder setMinM(int minM) {
            this.minM = minM;
            return this;
        }

        public int getMaxM() {
            return this.maxM;
        }

        public ConfigBuilder setMaxM(int maxM) {
            this.maxM = maxM;
            return this;
        }

        public int getSplitS() {
            return this.splitS;
        }

        public ConfigBuilder setSplitS(int splitS) {
            this.splitS = splitS;
            return this;
        }

        @Nonnull
        public Storage getStorage() {
            return this.storage;
        }

        public ConfigBuilder setStorage(@Nonnull Storage storage) {
            this.storage = storage;
            return this;
        }

        public boolean isStoreHilbertValues() {
            return this.storeHilbertValues;
        }

        public ConfigBuilder setStoreHilbertValues(boolean storeHilbertValues) {
            this.storeHilbertValues = storeHilbertValues;
            return this;
        }

        public boolean isUseNodeSlotIndex() {
            return this.useNodeSlotIndex;
        }

        public ConfigBuilder setUseNodeSlotIndex(boolean useNodeSlotIndex) {
            this.useNodeSlotIndex = useNodeSlotIndex;
            return this;
        }

        public Config build() {
            return new Config(this.isUseNodeSlotIndex(), this.getMinM(), this.getMaxM(), this.getSplitS(), this.getStorage(), this.isStoreHilbertValues());
        }
    }

    public static class Config {
        private final boolean useNodeSlotIndex;
        private final int minM;
        private final int maxM;
        private final int splitS;
        @Nonnull
        private final Storage storage;
        private final boolean storeHilbertValues;

        protected Config() {
            this.useNodeSlotIndex = false;
            this.minM = 16;
            this.maxM = 32;
            this.splitS = 2;
            this.storage = DEFAULT_STORAGE;
            this.storeHilbertValues = true;
        }

        protected Config(boolean useNodeSlotIndex, int minM, int maxM, int splitS, @Nonnull Storage storage, boolean storeHilbertValues) {
            this.useNodeSlotIndex = useNodeSlotIndex;
            this.minM = minM;
            this.maxM = maxM;
            this.splitS = splitS;
            this.storage = storage;
            this.storeHilbertValues = storeHilbertValues;
        }

        public boolean isUseNodeSlotIndex() {
            return this.useNodeSlotIndex;
        }

        public int getMinM() {
            return this.minM;
        }

        public int getMaxM() {
            return this.maxM;
        }

        public int getSplitS() {
            return this.splitS;
        }

        @Nonnull
        public Storage getStorage() {
            return this.storage;
        }

        public boolean isStoreHilbertValues() {
            return this.storeHilbertValues;
        }

        public ConfigBuilder toBuilder() {
            return new ConfigBuilder(this.useNodeSlotIndex, this.minM, this.maxM, this.splitS, this.storage, this.storeHilbertValues);
        }

        public String toString() {
            return String.valueOf((Object)this.storage) + ", M=" + this.minM + "-" + this.maxM + ", S=" + this.splitS + (this.useNodeSlotIndex ? ", slotIndex" : "") + (this.storeHilbertValues ? ", storeHV" : "");
        }
    }

    public static enum Storage {
        BY_SLOT(BySlotStorageAdapter::new),
        BY_NODE(ByNodeStorageAdapter::new);

        @Nonnull
        private final StorageAdapterCreator storageAdapterCreator;

        private Storage(StorageAdapterCreator storageAdapterCreator) {
            this.storageAdapterCreator = storageAdapterCreator;
        }

        @Nonnull
        private StorageAdapter newStorageAdapter(@Nonnull Config config, @Nonnull Subspace subspace, @Nonnull Subspace nodeSlotIndexSubspace, @Nonnull Function<Point, BigInteger> hilbertValueFunction, @Nonnull OnWriteListener onWriteListener, @Nonnull OnReadListener onReadListener) {
            return this.storageAdapterCreator.create(config, subspace, nodeSlotIndexSubspace, hilbertValueFunction, onWriteListener, onReadListener);
        }
    }

    private class LeafIterator
    implements AsyncIterator<LeafNode> {
        @Nonnull
        private final ReadTransaction readTransaction;
        @Nonnull
        private final byte[] rootId;
        @Nullable
        private final BigInteger lastHilbertValue;
        @Nullable
        private final Tuple lastKey;
        @Nonnull
        private final Predicate<Rectangle> mbrPredicate;
        @Nonnull
        private final BiPredicate<Tuple, Tuple> suffixKeyPredicate;
        @Nullable
        private TraversalState currentState;
        @Nullable
        private CompletableFuture<TraversalState> nextStateFuture;

        @SpotBugsSuppressWarnings(value={"EI_EXPOSE_REP2"})
        public LeafIterator(@Nonnull ReadTransaction readTransaction, @Nullable byte[] rootId, @Nullable BigInteger lastHilbertValue, @Nonnull Tuple lastKey, @Nonnull Predicate<Rectangle> mbrPredicate, BiPredicate<Tuple, Tuple> suffixKeyPredicate) {
            Preconditions.checkArgument(lastHilbertValue == null && lastKey == null || lastHilbertValue != null && lastKey != null);
            this.readTransaction = readTransaction;
            this.rootId = rootId;
            this.lastHilbertValue = lastHilbertValue;
            this.lastKey = lastKey;
            this.mbrPredicate = mbrPredicate;
            this.suffixKeyPredicate = suffixKeyPredicate;
            this.currentState = null;
            this.nextStateFuture = null;
        }

        @Override
        public CompletableFuture<Boolean> onHasNext() {
            if (this.nextStateFuture == null) {
                this.nextStateFuture = this.currentState == null ? RTree.this.fetchLeftmostPathToLeaf(this.readTransaction, this.rootId, this.lastHilbertValue, this.lastKey, this.mbrPredicate, this.suffixKeyPredicate) : RTree.this.fetchNextPathToLeaf(this.readTransaction, this.currentState, this.lastHilbertValue, this.lastKey, this.mbrPredicate, this.suffixKeyPredicate);
            }
            return this.nextStateFuture.thenApply(traversalState -> !traversalState.isEnd());
        }

        @Override
        public boolean hasNext() {
            return this.onHasNext().join();
        }

        @Override
        public LeafNode next() {
            if (this.hasNext()) {
                this.currentState = Objects.requireNonNull(this.nextStateFuture).join();
                this.nextStateFuture = null;
                return this.currentState.getCurrentLeafNode();
            }
            throw new NoSuchElementException("called next() on exhausted iterator");
        }

        @Override
        public void cancel() {
            if (this.nextStateFuture != null) {
                this.nextStateFuture.cancel(false);
            }
        }
    }

    public static class ItemSlotIterator
    implements AsyncIterator<ItemSlot> {
        @Nonnull
        private final AsyncIterator<LeafNode> leafIterator;
        @Nullable
        private LeafNode currentLeafNode;
        @Nullable
        private Iterator<ItemSlot> currenLeafItemsIterator;

        private ItemSlotIterator(@Nonnull AsyncIterator<LeafNode> leafIterator) {
            this.leafIterator = leafIterator;
            this.currentLeafNode = null;
            this.currenLeafItemsIterator = null;
        }

        @Override
        public CompletableFuture<Boolean> onHasNext() {
            if (this.currenLeafItemsIterator != null && this.currenLeafItemsIterator.hasNext()) {
                return CompletableFuture.completedFuture(true);
            }
            return this.leafIterator.onHasNext().thenApply(hasNext -> {
                if (hasNext.booleanValue()) {
                    this.currentLeafNode = this.leafIterator.next();
                    this.currenLeafItemsIterator = this.currentLeafNode.getSlots().iterator();
                    return this.currenLeafItemsIterator.hasNext();
                }
                return false;
            });
        }

        @Override
        public boolean hasNext() {
            return this.onHasNext().join();
        }

        @Override
        public ItemSlot next() {
            if (this.hasNext()) {
                return Objects.requireNonNull(this.currenLeafItemsIterator).next();
            }
            throw new NoSuchElementException("called next() on exhausted iterator");
        }

        @Override
        public void cancel() {
            this.leafIterator.cancel();
        }
    }

    private static class TraversalState {
        @Nullable
        private final List<Deque<ChildSlot>> toBeProcessed;
        @Nullable
        private final LeafNode currentLeafNode;

        private TraversalState(@Nullable List<Deque<ChildSlot>> toBeProcessed, @Nullable LeafNode currentLeafNode) {
            this.toBeProcessed = toBeProcessed;
            this.currentLeafNode = currentLeafNode;
        }

        @Nonnull
        public List<Deque<ChildSlot>> getToBeProcessed() {
            return Objects.requireNonNull(this.toBeProcessed);
        }

        @Nonnull
        public LeafNode getCurrentLeafNode() {
            return Objects.requireNonNull(this.currentLeafNode);
        }

        public boolean isEnd() {
            return this.currentLeafNode == null;
        }

        public static TraversalState of(@Nonnull List<Deque<ChildSlot>> toBeProcessed, @Nonnull LeafNode currentLeafNode) {
            return new TraversalState(toBeProcessed, currentLeafNode);
        }

        public static TraversalState end() {
            return new TraversalState(null, null);
        }
    }

    public static class Rectangle {
        @Nonnull
        private final Tuple ranges;

        public Rectangle(Tuple ranges) {
            Preconditions.checkArgument(!ranges.isEmpty() && ranges.size() % 2 == 0);
            this.ranges = ranges;
        }

        public int getNumDimensions() {
            return this.ranges.size() >> 1;
        }

        @Nonnull
        public Tuple getRanges() {
            return this.ranges;
        }

        @Nonnull
        public Object getLow(int dimension) {
            return this.ranges.get(dimension);
        }

        @Nonnull
        public Object getHigh(int dimension) {
            return this.ranges.get((this.ranges.size() >> 1) + dimension);
        }

        @Nonnull
        public BigInteger area() {
            BigInteger currentArea = BigInteger.ONE;
            for (int d = 0; d < this.getNumDimensions(); ++d) {
                currentArea = currentArea.multiply(BigInteger.valueOf(((Number)this.getHigh(d)).longValue() - ((Number)this.getLow(d)).longValue()));
            }
            return currentArea;
        }

        @Nonnull
        public Rectangle unionWith(@Nonnull Point point) {
            Preconditions.checkArgument(this.getNumDimensions() == point.getNumDimensions());
            boolean isModified = false;
            Object[] ranges = new Object[this.getNumDimensions() << 1];
            for (int d = 0; d < this.getNumDimensions(); ++d) {
                Object low;
                Tuple lowTuple;
                Object coordinate = point.getCoordinate(d);
                Tuple coordinateTuple = Tuple.from(coordinate);
                if (TupleHelpers.compare(coordinateTuple, lowTuple = Tuple.from(low = this.getLow(d))) < 0) {
                    ranges[d] = coordinate;
                    isModified = true;
                } else {
                    ranges[d] = low;
                }
                Object high = this.getHigh(d);
                Tuple highTuple = Tuple.from(high);
                if (TupleHelpers.compare(coordinateTuple, highTuple) > 0) {
                    ranges[this.getNumDimensions() + d] = coordinate;
                    isModified = true;
                    continue;
                }
                ranges[this.getNumDimensions() + d] = high;
            }
            if (!isModified) {
                return this;
            }
            return new Rectangle(Tuple.from(ranges));
        }

        @Nonnull
        public Rectangle unionWith(@Nonnull Rectangle other) {
            Preconditions.checkArgument(this.getNumDimensions() == other.getNumDimensions());
            boolean isModified = false;
            Object[] ranges = new Object[this.getNumDimensions() << 1];
            for (int d = 0; d < this.getNumDimensions(); ++d) {
                Object otherLow = other.getLow(d);
                Tuple otherLowTuple = Tuple.from(otherLow);
                Object otherHigh = other.getHigh(d);
                Tuple otherHighTuple = Tuple.from(otherHigh);
                Object low = this.getLow(d);
                Tuple lowTuple = Tuple.from(low);
                if (TupleHelpers.compare(otherLowTuple, lowTuple) < 0) {
                    ranges[d] = otherLow;
                    isModified = true;
                } else {
                    ranges[d] = low;
                }
                Object high = this.getHigh(d);
                Tuple highTuple = Tuple.from(high);
                if (TupleHelpers.compare(otherHighTuple, highTuple) > 0) {
                    ranges[this.getNumDimensions() + d] = otherHigh;
                    isModified = true;
                    continue;
                }
                ranges[this.getNumDimensions() + d] = high;
            }
            if (!isModified) {
                return this;
            }
            return new Rectangle(Tuple.from(ranges));
        }

        public boolean isOverlapping(@Nonnull Rectangle other) {
            Preconditions.checkArgument(this.getNumDimensions() == other.getNumDimensions());
            for (int d = 0; d < this.getNumDimensions(); ++d) {
                Tuple otherLowTuple = Tuple.from(other.getLow(d));
                Tuple otherHighTuple = Tuple.from(other.getHigh(d));
                Tuple lowTuple = Tuple.from(this.getLow(d));
                Tuple highTuple = Tuple.from(this.getHigh(d));
                if (TupleHelpers.compare(highTuple, otherLowTuple) >= 0 && TupleHelpers.compare(lowTuple, otherHighTuple) <= 0) continue;
                return false;
            }
            return true;
        }

        public boolean contains(@Nonnull Point point) {
            Preconditions.checkArgument(this.getNumDimensions() == point.getNumDimensions());
            for (int d = 0; d < this.getNumDimensions(); ++d) {
                Tuple otherTuple = Tuple.from(point.getCoordinate(d));
                Tuple lowTuple = Tuple.from(this.getLow(d));
                Tuple highTuple = Tuple.from(this.getHigh(d));
                if (TupleHelpers.compare(highTuple, otherTuple) >= 0 && TupleHelpers.compare(lowTuple, otherTuple) <= 0) continue;
                return false;
            }
            return true;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof Rectangle)) {
                return false;
            }
            Rectangle rectangle = (Rectangle)o;
            return TupleHelpers.equals(this.ranges, rectangle.ranges);
        }

        public int hashCode() {
            return this.ranges.hashCode();
        }

        @Nonnull
        public String toPlotString() {
            int d;
            StringBuilder builder = new StringBuilder();
            for (d = 0; d < this.getNumDimensions(); ++d) {
                builder.append(((Number)this.getLow(d)).longValue());
                if (d + 1 >= this.getNumDimensions()) continue;
                builder.append(",");
            }
            builder.append(",");
            for (d = 0; d < this.getNumDimensions(); ++d) {
                builder.append(((Number)this.getHigh(d)).longValue());
                if (d + 1 >= this.getNumDimensions()) continue;
                builder.append(",");
            }
            return builder.toString();
        }

        @Nonnull
        public String toString() {
            return this.ranges.toString();
        }

        @Nonnull
        public static Rectangle fromPoint(@Nonnull Point point) {
            Object[] mbrRanges = new Object[point.getNumDimensions() * 2];
            for (int d = 0; d < point.getNumDimensions(); ++d) {
                Object coordinate;
                mbrRanges[d] = coordinate = point.getCoordinate(d);
                mbrRanges[point.getNumDimensions() + d] = coordinate;
            }
            return new Rectangle(Tuple.from(mbrRanges));
        }
    }

    public static class Point {
        @Nonnull
        private final Tuple coordinates;

        public Point(@Nonnull Tuple coordinates) {
            Preconditions.checkArgument(!coordinates.isEmpty());
            this.coordinates = coordinates;
        }

        @Nonnull
        public Tuple getCoordinates() {
            return this.coordinates;
        }

        public int getNumDimensions() {
            return this.coordinates.size();
        }

        @Nullable
        public Object getCoordinate(int dimension) {
            return this.coordinates.get(dimension);
        }

        @Nullable
        public Number getCoordinateAsNumber(int dimension) {
            return (Number)this.getCoordinate(dimension);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof Point)) {
                return false;
            }
            Point point = (Point)o;
            return TupleHelpers.equals(this.coordinates, point.coordinates);
        }

        public int hashCode() {
            return this.coordinates.hashCode();
        }

        @Nonnull
        public String toString() {
            return this.coordinates.toString();
        }
    }

    private static class NodeOrAdjust {
        public static final NodeOrAdjust NONE = new NodeOrAdjust(null, null, false);
        public static final NodeOrAdjust ADJUST = new NodeOrAdjust(null, null, true);
        @Nullable
        private final ChildSlot slotInParent;
        @Nullable
        private final Node node;
        private final boolean parentNeedsAdjustment;

        private NodeOrAdjust(@Nullable ChildSlot slotInParent, @Nullable Node node, boolean parentNeedsAdjustment) {
            Verify.verify(slotInParent == null && node == null || slotInParent != null && node != null);
            this.slotInParent = slotInParent;
            this.node = node;
            this.parentNeedsAdjustment = parentNeedsAdjustment;
        }

        @Nullable
        public ChildSlot getSlotInParent() {
            return this.slotInParent;
        }

        @Nullable
        public Node getSplitNode() {
            return this.node;
        }

        @Nullable
        public Node getTombstoneNode() {
            return this.node;
        }

        public boolean parentNeedsAdjustment() {
            return this.parentNeedsAdjustment;
        }
    }

    private static class ValidationTraversalState {
        final int level;
        @Nullable
        private final IntermediateNode parentNode;
        @Nonnull
        private final byte[] childId;

        public ValidationTraversalState(int level, @Nullable IntermediateNode parentNode, @Nonnull byte[] childId) {
            this.level = level;
            this.parentNode = parentNode;
            this.childId = childId;
        }

        public int getLevel() {
            return this.level;
        }

        @Nullable
        public IntermediateNode getParentNode() {
            return this.parentNode;
        }

        @Nonnull
        public byte[] getChildId() {
            return this.childId;
        }
    }

    private static interface StorageAdapterCreator {
        public StorageAdapter create(@Nonnull Config var1, @Nonnull Subspace var2, @Nonnull Subspace var3, @Nonnull Function<Point, BigInteger> var4, @Nonnull OnWriteListener var5, @Nonnull OnReadListener var6);
    }
}

