/*
 * Decompiled with CFR 0.152.
 */
package com.apple.foundationdb.record.provider.foundationdb;

import com.apple.foundationdb.KeyValue;
import com.apple.foundationdb.Transaction;
import com.apple.foundationdb.annotation.API;
import com.apple.foundationdb.async.AsyncUtil;
import com.apple.foundationdb.directory.DirectoryLayer;
import com.apple.foundationdb.record.ExecuteProperties;
import com.apple.foundationdb.record.IsolationLevel;
import com.apple.foundationdb.record.RecordCoreException;
import com.apple.foundationdb.record.RecordCoreRetriableTransactionException;
import com.apple.foundationdb.record.ScanProperties;
import com.apple.foundationdb.record.TupleRange;
import com.apple.foundationdb.record.logging.KeyValueLogMessage;
import com.apple.foundationdb.record.logging.LogMessageKeys;
import com.apple.foundationdb.record.provider.common.StoreTimer;
import com.apple.foundationdb.record.provider.foundationdb.FDBDatabase;
import com.apple.foundationdb.record.provider.foundationdb.FDBRecordContext;
import com.apple.foundationdb.record.provider.foundationdb.FDBStoreTimer;
import com.apple.foundationdb.record.provider.foundationdb.KeyValueCursor;
import com.apple.foundationdb.record.provider.foundationdb.keyspace.LocatableResolver;
import com.apple.foundationdb.record.provider.foundationdb.keyspace.ScopedValue;
import com.apple.foundationdb.subspace.Subspace;
import com.apple.foundationdb.tuple.Tuple;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Charsets;
import java.util.Collections;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@API(value=API.Status.INTERNAL)
public class FDBReverseDirectoryCache {
    private static final Logger LOGGER = LoggerFactory.getLogger(FDBReverseDirectoryCache.class);
    public static final String REVERSE_DIRECTORY_CACHE_ENTRY = new String(new byte[]{114, 101, 99, 100, 98, 95, 114, 100, 95, 99, 97, 99, 104, 101}, Charsets.US_ASCII);
    public static final int MAX_ROWS_PER_TRANSACTION = 10000;
    public static final long MAX_MILLIS_PER_TRANSACTION = TimeUnit.SECONDS.toMillis(3L);
    private FDBDatabase fdb;
    private CompletableFuture<Long> reverseDirectoryCacheEntry;
    private AtomicLong persistentCacheMissCount = new AtomicLong(0L);
    private AtomicLong persistentCacheHitCount = new AtomicLong(0L);
    private int maxRowsPerTransaction;
    private long maxMillisPerTransaction;

    public FDBReverseDirectoryCache(@Nonnull FDBDatabase fdb) {
        this(fdb, 10000, MAX_MILLIS_PER_TRANSACTION);
    }

    public FDBReverseDirectoryCache(@Nonnull FDBDatabase fdb, int maxRowsPerTransaction, long maxMillisPerTransaction) {
        this.fdb = fdb;
        this.maxRowsPerTransaction = maxRowsPerTransaction;
        this.maxMillisPerTransaction = maxMillisPerTransaction;
        this.reverseDirectoryCacheEntry = fdb.runAsync(context -> DirectoryLayer.getDefault().createOrOpen(context.ensureActive(), Collections.singletonList(REVERSE_DIRECTORY_CACHE_ENTRY)).thenApply(subspace -> Tuple.fromBytes(subspace.getKey()).getLong(0)));
    }

    public int getMaxRowsPerTransaction() {
        return this.maxRowsPerTransaction;
    }

    public void setMaxRowsPerTransaction(int maxRowsPerTransaction) {
        this.maxRowsPerTransaction = maxRowsPerTransaction;
    }

    public long getMaxMillisPerTransaction() {
        return this.maxMillisPerTransaction;
    }

    public void setMaxMillisPerTransaction(long maxMillisPerTransaction) {
        this.maxMillisPerTransaction = maxMillisPerTransaction;
    }

    public long getPersistentCacheMissCount() {
        return this.persistentCacheMissCount.get();
    }

    public long getPersistentCacheHitCount() {
        return this.persistentCacheHitCount.get();
    }

    public void clearStats() {
        this.persistentCacheMissCount.set(0L);
        this.persistentCacheHitCount.set(0L);
    }

    @Nonnull
    public CompletableFuture<Optional<String>> getInReverseDirectoryCacheSubspace(@Nullable FDBStoreTimer timer, @Nonnull ScopedValue<Long> scopedReverseDirectoryKey) {
        FDBRecordContext context = this.fdb.openContext(null, timer);
        return ((CompletableFuture)this.getReverseCacheSubspace(scopedReverseDirectoryKey.getScope()).thenCompose(subspace -> this.getFromSubspace(context, (Subspace)subspace, scopedReverseDirectoryKey))).whenComplete((result, exception) -> context.close());
    }

    @Nonnull
    public CompletableFuture<Optional<String>> get(@Nonnull ScopedValue<Long> scopedReverseDirectoryKey) {
        return this.get((FDBStoreTimer)null, scopedReverseDirectoryKey);
    }

    @Nonnull
    public CompletableFuture<Optional<String>> get(@Nullable FDBStoreTimer timer, @Nonnull ScopedValue<Long> scopedReverseDirectoryKey) {
        FDBRecordContext context = this.fdb.openContext(null, timer);
        return this.get(context, scopedReverseDirectoryKey).whenComplete((result, exception) -> context.close());
    }

    @Nonnull
    public CompletableFuture<Optional<String>> get(@Nonnull FDBRecordContext context, @Nonnull ScopedValue<Long> scopedReverseDirectoryKey) {
        CompletableFuture<Subspace> reverseCacheSubspaceFuture = this.getReverseCacheSubspace(scopedReverseDirectoryKey.getScope());
        return reverseCacheSubspaceFuture.thenCompose(subspace -> this.getFromSubspace(context, (Subspace)subspace, scopedReverseDirectoryKey));
    }

    private void logStatsToStoreTimer(@Nonnull FDBRecordContext context, @Nonnull StoreTimer.Count event) {
        if (context.getTimer() != null) {
            context.getTimer().increment(event);
        }
    }

    private CompletableFuture<Optional<String>> getFromSubspace(@Nonnull FDBRecordContext context, @Nonnull Subspace reverseCacheSubspace, @Nonnull ScopedValue<Long> scopedReverseDirectoryKey) {
        Long reverseDirectoryKeyData = scopedReverseDirectoryKey.getData();
        return context.ensureActive().snapshot().get(reverseCacheSubspace.pack(reverseDirectoryKeyData)).thenApply(valueBytes -> {
            if (valueBytes != null) {
                String dirString = Tuple.fromBytes(valueBytes).getString(0);
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("Cache miss, but found path '" + dirString + "'' in reverse lookup for value '" + String.valueOf(scopedReverseDirectoryKey) + "'");
                }
                this.persistentCacheHitCount.incrementAndGet();
                this.logStatsToStoreTimer(context, FDBStoreTimer.Counts.REVERSE_DIR_PERSISTENT_CACHE_HIT_COUNT);
                return Optional.of(dirString);
            }
            this.persistentCacheMissCount.incrementAndGet();
            this.logStatsToStoreTimer(context, FDBStoreTimer.Counts.REVERSE_DIR_PERSISTENT_CACHE_MISS_COUNT);
            return Optional.empty();
        });
    }

    public CompletableFuture<Void> put(@Nonnull FDBRecordContext context, @Nonnull ScopedValue<String> pathKey) {
        LocatableResolver scope = pathKey.getScope();
        String key = pathKey.getData();
        return this.getReverseCacheSubspace(scope).thenCompose(reverseCacheSubspace -> scope.mustResolve(context, key).thenApply(value -> {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug(KeyValueLogMessage.of("Adding value to reverse directory cache", new Object[]{LogMessageKeys.KEY, pathKey, LogMessageKeys.VALUE, value}));
            }
            context.ensureActive().set(reverseCacheSubspace.pack(value), Tuple.from(pathKey.getData()).pack());
            return null;
        }));
    }

    @VisibleForTesting
    public CompletableFuture<Void> deleteForTesting(@Nonnull FDBRecordContext context, @Nonnull ScopedValue<Long> value) {
        LocatableResolver scope = value.getScope();
        return this.getReverseCacheSubspace(scope).thenApply(subspace -> {
            context.ensureActive().clear(subspace.pack(value.getData()));
            return null;
        });
    }

    @VisibleForTesting
    public CompletableFuture<Void> putOrReplaceForTesting(@Nonnull FDBRecordContext context, @Nonnull ScopedValue<String> scopedPathString, @Nonnull Long value) {
        LocatableResolver scope = scopedPathString.getScope();
        String pathString = scopedPathString.getData();
        return this.getReverseCacheSubspace(scope).thenApply(reverseCacheSubspace -> {
            context.ensureActive().set(reverseCacheSubspace.pack(value), Tuple.from(pathString).pack());
            return null;
        });
    }

    public CompletableFuture<Void> putIfNotExists(@Nonnull FDBRecordContext context, @Nonnull ScopedValue<String> scopedPathString, @Nonnull Long pathValue) {
        LocatableResolver scope = scopedPathString.getScope();
        String pathString = scopedPathString.getData();
        String cachedString = context.getDatabase().getReverseDirectoryInMemoryCache().getIfPresent(scope.wrap(pathValue));
        if (cachedString != null) {
            if (!cachedString.equals(pathString)) {
                throw new RecordCoreRetriableTransactionException("Provided value for path key does not match existing value in reverse directory layer in-memory cache").addLogInfo(new Object[]{LogMessageKeys.RESOLVER, scope}).addLogInfo(new Object[]{LogMessageKeys.RESOLVER_PATH, pathString}).addLogInfo(new Object[]{LogMessageKeys.RESOLVER_KEY, pathValue}).addLogInfo(new Object[]{LogMessageKeys.CACHED_KEY, cachedString});
            }
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("In-memory cache contains '" + pathString + "' -> '" + pathValue + "' mapping. No need to put");
            }
            return AsyncUtil.DONE;
        }
        return this.getReverseCacheSubspace(scope).thenCompose(subspace -> this.putToSubspace(context, (Subspace)subspace, scopedPathString, pathValue));
    }

    private CompletableFuture<Void> putToSubspace(@Nonnull FDBRecordContext context, @Nonnull Subspace reverseCacheSubspace, @Nonnull ScopedValue<String> scopedPathString, @Nonnull Long pathValue) {
        String pathString = scopedPathString.getData();
        Transaction transaction = context.ensureActive();
        return transaction.snapshot().get(reverseCacheSubspace.pack(pathValue)).thenApply(valueBytes -> {
            if (valueBytes != null) {
                String readValue = Tuple.fromBytes(valueBytes).getString(0);
                if (!readValue.equals(pathString)) {
                    throw new RecordCoreException("Provided value for path key does not match existing value in reverse directory layer cache", new Object[0]).addLogInfo(new Object[]{LogMessageKeys.RESOLVER, scopedPathString.getScope()}).addLogInfo(new Object[]{LogMessageKeys.RESOLVER_PATH, pathString}).addLogInfo(new Object[]{LogMessageKeys.RESOLVER_KEY, pathValue}).addLogInfo(new Object[]{LogMessageKeys.CACHED_KEY, readValue});
                }
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("Put unnecessary, found path '" + readValue + "'' in reverse lookup for value '" + pathValue + "'");
                }
                this.persistentCacheHitCount.incrementAndGet();
                this.logStatsToStoreTimer(context, FDBStoreTimer.Counts.REVERSE_DIR_PERSISTENT_CACHE_HIT_COUNT);
            } else {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("Adding '" + pathValue + "' to reverse lookup with key " + pathString);
                }
                transaction.set(reverseCacheSubspace.pack(pathValue), Tuple.from(pathString).pack());
                this.persistentCacheMissCount.incrementAndGet();
                this.logStatsToStoreTimer(context, FDBStoreTimer.Counts.REVERSE_DIR_PERSISTENT_CACHE_MISS_COUNT);
            }
            return null;
        });
    }

    @VisibleForTesting
    public void rebuild(LocatableResolver scope) {
        try (FDBRecordContext context = this.fdb.openContext();){
            Subspace reverseCacheSubspace = this.fdb.asyncToSync(null, null, this.getReverseCacheSubspace(scope));
            context.ensureActive().clear(reverseCacheSubspace.range());
            context.getDatabase().clearForwardDirectoryCache();
            this.persistentCacheMissCount.set(0L);
            this.persistentCacheHitCount.set(0L);
            this.populate(context, reverseCacheSubspace);
        }
    }

    @VisibleForTesting
    public void waitUntilReadyForTesting() {
        this.reverseDirectoryCacheEntry.join();
    }

    private CompletableFuture<Subspace> getReverseCacheSubspace(LocatableResolver scope) {
        return this.reverseDirectoryCacheEntry.thenCombine(scope.getBaseSubspaceAsync(), (entry, subspace) -> subspace.subspace(Tuple.from(entry)));
    }

    private void populate(FDBRecordContext initialContext, Subspace directory) {
        byte[] prefix = new byte[]{-2};
        Subspace subdirs = new Subspace(Tuple.from(prefix, 0L), prefix);
        this.fdb.asyncToSync(initialContext.getTimer(), FDBStoreTimer.Waits.WAIT_REVERSE_DIRECTORY_SCAN, this.populate(initialContext, subdirs, directory, null));
    }

    @Nonnull
    private CompletableFuture<byte[]> populate(@Nonnull FDBRecordContext context, @Nonnull Subspace directorySubspace, @Nonnull Subspace reverseDirectorySubspace, @Nullable byte[] continuation) {
        return this.populateRegion(context, directorySubspace, reverseDirectorySubspace, continuation).thenCompose(nextContinuation -> context.commitAsync().thenCompose(ignored -> {
            context.close();
            if (nextContinuation == null) {
                return CompletableFuture.completedFuture(null);
            }
            FDBRecordContext nextContext = this.fdb.openContext();
            return this.populate(nextContext, directorySubspace, reverseDirectorySubspace, (byte[])nextContinuation);
        }));
    }

    @Nonnull
    private CompletableFuture<byte[]> populateRegion(@Nonnull FDBRecordContext context, @Nonnull Subspace directorySubspace, @Nonnull Subspace reverseDirectorySubspace, @Nullable byte[] continuation) {
        KeyValueCursor cursor = ((KeyValueCursor.Builder)((KeyValueCursor.Builder)((KeyValueCursor.Builder)((KeyValueCursor.Builder)KeyValueCursor.Builder.withSubspace(directorySubspace).setContext(context)).setRange(TupleRange.ALL)).setContinuation(continuation)).setScanProperties(new ScanProperties(ExecuteProperties.newBuilder().setReturnedRowLimit(this.maxRowsPerTransaction).setIsolationLevel(IsolationLevel.SNAPSHOT).build()))).build();
        return cursor.forEachResult(result -> {
            KeyValue kv = (KeyValue)result.get();
            String dirName = directorySubspace.unpack(kv.getKey()).getString(0);
            Object dirValue = Tuple.fromBytes(kv.getValue()).get(0);
            context.ensureActive().set(reverseDirectorySubspace.pack(dirValue), Tuple.from(dirName).pack());
        }).thenApply(result -> result.getContinuation().toBytes());
    }
}

