package com.foxcode.android.common.nointernet.ui.base

import android.app.Activity
import android.app.Dialog
import android.content.DialogInterface
import android.graphics.Color
import android.graphics.drawable.ColorDrawable
import android.os.Bundle
import android.view.Gravity
import android.view.ViewGroup
import android.view.Window
import androidx.annotation.StyleRes
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleObserver
import com.foxcode.android.common.nointernet.callbacks.ConnectionCallback
import com.foxcode.android.common.nointernet.components.NoInternetObserveComponent
import com.foxcode.android.common.nointernet.ui.DialogProperties
import com.foxcode.android.common.nointernet.utils.NoInternetUtils

abstract class BaseNoInternetDialog(
    private val activity: Activity,
    private val lifecycle: Lifecycle,
    private val dialogProperties: DialogProperties,
    @StyleRes themeResId: Int
) : Dialog(activity, themeResId), LifecycleObserver {
    private var noInternetObserveComponent: NoInternetObserveComponent
    private var connectionCallback: ConnectionCallback? = dialogProperties.connectionCallback

    init {
        noInternetObserveComponent = NoInternetObserveComponent(
            activity.applicationContext,
            lifecycle,
            object : NoInternetObserveComponent.NoInternetObserverListener {
                override fun onStart() {
                    // no-op
                }

                override fun onConnected() {
                    connectionCallback?.hasActiveConnection(true)
                    dismiss()
                }

                override fun onDisconnected(isAirplaneModeOn: Boolean) {
                    connectionCallback?.hasActiveConnection(false)
                    show()
                }

                override fun onStop() {
                    destroy()
                }
            }
        )
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        requestWindowFeature(Window.FEATURE_NO_TITLE)
        lifecycle.addObserver(this)

        setLayout()
        initView()
        setProperties()
        setupDialog()
    }

    private fun setProperties() {
        setCancelable(dialogProperties.cancelable)
    }

    private fun setupDialog() {
        window?.apply {
            setLayout(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT)
            setGravity(Gravity.CENTER)
            setBackgroundDrawable(ColorDrawable(Color.TRANSPARENT))
        }
    }

    /**
     * Show the dialog.
     *
     * Note: We didn't check the Internet connection in this function.
     * So, call it if you know what you are doing.
     */
    override fun show() {
        if (!activity.isFinishing && lifecycle.currentState.isAtLeast(Lifecycle.State.RESUMED)) {
            super.show()
            onShow(NoInternetUtils.isAirplaneModeOn(activity))
        }
    }

    override fun setOnDismissListener(listener: DialogInterface.OnDismissListener?) {
        super.setOnDismissListener(listener)
    }

    override fun onStop() {
        super.onStop()

        onDismiss()
    }

    /**
     * Destroy the dialog.
     */
    fun destroy() {
        onDestroy()
        dismiss()
    }

    /**
     * Manually start listening.
     */
    fun start() {
        noInternetObserveComponent.start()
    }

    /**
     * Manually stop listening.
     */
    fun stop() {
        noInternetObserveComponent.stop()
    }

    /**
     * Set content view here.
     */
    internal abstract fun setLayout()

    /**
     * Initialize view elements.
     */
    internal abstract fun initView()

    /**
     * Thing to do after dialog `show()` called, like start the animations etc.
     */
    internal abstract fun onShow(isAirplaneModeOn: Boolean)

    /**
     * Thing to do on dismiss dialog, like stop the animations etc.
     */
    internal abstract fun onDismiss()

    /**
     * Thing to do on destroy dialog.
     */
    internal abstract fun onDestroy()
}
