/**
 * Copyright (c) 2012 Harman International (http://www.harman.com).
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.franca.deploymodel.core;

import com.google.common.base.Objects;
import java.util.List;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function0;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.franca.deploymodel.core.FDModelUtils;
import org.franca.deploymodel.core.GenericPropertyAccessor;
import org.franca.deploymodel.core.PropertyMappings;
import org.franca.deploymodel.dsl.fDeploy.FDBoolean;
import org.franca.deploymodel.dsl.fDeploy.FDComplexValue;
import org.franca.deploymodel.dsl.fDeploy.FDElement;
import org.franca.deploymodel.dsl.fDeploy.FDEnumType;
import org.franca.deploymodel.dsl.fDeploy.FDEnumerator;
import org.franca.deploymodel.dsl.fDeploy.FDGeneric;
import org.franca.deploymodel.dsl.fDeploy.FDInteger;
import org.franca.deploymodel.dsl.fDeploy.FDProperty;
import org.franca.deploymodel.dsl.fDeploy.FDPropertyDecl;
import org.franca.deploymodel.dsl.fDeploy.FDSpecification;
import org.franca.deploymodel.dsl.fDeploy.FDString;
import org.franca.deploymodel.dsl.fDeploy.FDType;
import org.franca.deploymodel.dsl.fDeploy.FDValue;
import org.franca.deploymodel.dsl.fDeploy.FDValueArray;
import org.franca.deploymodel.dsl.fDeploy.FDeployFactory;

/**
 * Helper functions to build deploy models.
 * 
 * This will be moved to org.franca.deploymodel in future.
 */
@SuppressWarnings("all")
public class DeployModelBuilder {
  /**
   * Set an integer deployment property for a given FDElement.
   * If the property is declared with a default value and the value
   * which is to be set is exactly the default value, the property will not
   * be set.
   * 
   * @param elem      the FDElement where the property should be set
   * @param spec      the deployment specification where the property is declared
   * @param property  the name of the property which should be set
   * @param value     the actual value to be set
   */
  public static Boolean setProperty(final FDElement elem, final FDSpecification spec, final String property, final int value) {
    final Function1<FDValue, Boolean> _function = new Function1<FDValue, Boolean>() {
      @Override
      public Boolean apply(final FDValue dflt) {
        int _value = ((FDInteger) dflt).getValue();
        return Boolean.valueOf((_value != value));
      }
    };
    final Function0<FDValue> _function_1 = new Function0<FDValue>() {
      @Override
      public FDValue apply() {
        return DeployModelBuilder.createFDValue(value);
      }
    };
    return DeployModelBuilder.setSingleProperty(elem, spec, property, _function, _function_1);
  }
  
  /**
   * Set a long integer deployment property for a given FDElement.
   * If the property is declared with a default value and the value
   * which is to be set is exactly the default value, the property will not
   * be set.
   * 
   * @param elem      the FDElement where the property should be set
   * @param spec      the deployment specification where the property is declared
   * @param property  the name of the property which should be set
   * @param value     the actual value to be set
   */
  public static Boolean setProperty(final FDElement elem, final FDSpecification spec, final String property, final long value) {
    return DeployModelBuilder.setProperty(elem, spec, property, ((int) value));
  }
  
  /**
   * Set a boolean deployment property for a given FDElement.
   * If the property is declared with a default value and the value
   * which is to be set is exactly the default value, the property will not
   * be set.
   * 
   * @param elem      the FDElement where the property should be set
   * @param spec      the deployment specification where the property is declared
   * @param property  the name of the property which should be set
   * @param value     the actual value to be set
   */
  public static Boolean setProperty(final FDElement elem, final FDSpecification spec, final String property, final boolean value) {
    final Function1<FDValue, Boolean> _function = new Function1<FDValue, Boolean>() {
      @Override
      public Boolean apply(final FDValue dflt) {
        boolean _eval = DeployModelBuilder.eval(((FDBoolean) dflt));
        return Boolean.valueOf((_eval != value));
      }
    };
    final Function0<FDValue> _function_1 = new Function0<FDValue>() {
      @Override
      public FDValue apply() {
        return DeployModelBuilder.createFDValue(value);
      }
    };
    return DeployModelBuilder.setSingleProperty(elem, spec, property, _function, _function_1);
  }
  
  private static boolean eval(final FDBoolean it) {
    String _value = it.getValue();
    return Objects.equal(_value, "true");
  }
  
  /**
   * Set an enum or string deployment property for a given FDElement.
   * If the property is declared with a String type, the value will
   * be set as a raw string. If the property is declared with an
   * Enumeration type, the value will be interpreted as an enumerator.
   * In the latter case, the value should be defined as enumerator in
   * the property's declaration.
   * In both cases, if the property is declared with a default value
   * and the value which is to be set is exactly the default value,
   * the property will not be set.
   * 
   * @param elem      the FDElement where the property should be set
   * @param spec      the deployment specification where the property is declared
   * @param property  the name of the property which should be set
   * @param value     the actual value to be set
   */
  public static Boolean setProperty(final FDElement elem, final FDSpecification spec, final String property, final String value) {
    Boolean _xblockexpression = null;
    {
      FDPropertyDecl decl = DeployModelBuilder.getPropertyDecl(elem, spec, property);
      Boolean _xifexpression = null;
      FDType _complex = decl.getType().getComplex();
      boolean _tripleEquals = (_complex == null);
      if (_tripleEquals) {
        final Function1<FDValue, Boolean> _function = new Function1<FDValue, Boolean>() {
          @Override
          public Boolean apply(final FDValue dflt) {
            String _value = ((FDString) dflt).getValue();
            return Boolean.valueOf((!Objects.equal(_value, value)));
          }
        };
        final Function0<FDValue> _function_1 = new Function0<FDValue>() {
          @Override
          public FDValue apply() {
            return DeployModelBuilder.createFDValue(value);
          }
        };
        _xifexpression = DeployModelBuilder.setPropertyGeneric(decl, elem, _function, _function_1);
      } else {
        Boolean _xblockexpression_1 = null;
        {
          FDType _complex_1 = decl.getType().getComplex();
          final FDEnumType enumtype = ((FDEnumType) _complex_1);
          final Function1<FDEnumerator, Boolean> _function_2 = new Function1<FDEnumerator, Boolean>() {
            @Override
            public Boolean apply(final FDEnumerator e) {
              String _name = e.getName();
              return Boolean.valueOf(Objects.equal(_name, value));
            }
          };
          final FDEnumerator evalue = IterableExtensions.<FDEnumerator>findFirst(enumtype.getEnumerators(), _function_2);
          Boolean _xifexpression_1 = null;
          if ((evalue != null)) {
            final Function1<FDValue, Boolean> _function_3 = new Function1<FDValue, Boolean>() {
              @Override
              public Boolean apply(final FDValue dflt) {
                FDEnumerator _enumerator = FDModelUtils.getEnumerator(dflt);
                return Boolean.valueOf((!Objects.equal(_enumerator, evalue)));
              }
            };
            final Function0<FDValue> _function_4 = new Function0<FDValue>() {
              @Override
              public FDValue apply() {
                return DeployModelBuilder.createFDValue(evalue);
              }
            };
            _xifexpression_1 = DeployModelBuilder.setPropertyGeneric(decl, elem, _function_3, _function_4);
          }
          _xblockexpression_1 = _xifexpression_1;
        }
        _xifexpression = _xblockexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  /**
   * Generic helper function.
   */
  private static Boolean setSingleProperty(final FDElement elem, final FDSpecification spec, final String property, final Function1<? super FDValue, ? extends Boolean> isNonDefault, final Function0<? extends FDValue> getNewValue) {
    Boolean _xblockexpression = null;
    {
      FDPropertyDecl decl = DeployModelBuilder.getPropertyDecl(elem, spec, property);
      _xblockexpression = DeployModelBuilder.setPropertyGeneric(decl, elem, isNonDefault, getNewValue);
    }
    return _xblockexpression;
  }
  
  /**
   * Find property declaration for element by property name.
   */
  public static FDPropertyDecl getPropertyDecl(final FDElement elem, final FDSpecification spec, final String property) {
    try {
      List<FDPropertyDecl> propertyDecls = PropertyMappings.getAllPropertyDecls(spec, elem);
      final Function1<FDPropertyDecl, Boolean> _function = new Function1<FDPropertyDecl, Boolean>() {
        @Override
        public Boolean apply(final FDPropertyDecl d) {
          String _name = d.getName();
          return Boolean.valueOf(Objects.equal(_name, property));
        }
      };
      final FDPropertyDecl pdecl = IterableExtensions.<FDPropertyDecl>findFirst(propertyDecls, _function);
      if ((pdecl == null)) {
        String _name = spec.getName();
        String _plus = ((("DeployModelBuilder: No property \'" + property) + 
          "\' in specification \'") + _name);
        String _plus_1 = (_plus + "\' for element ");
        String _string = elem.toString();
        String _plus_2 = (_plus_1 + _string);
        throw new Exception(_plus_2);
      }
      return pdecl;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private static Boolean setPropertyGeneric(final FDPropertyDecl decl, final FDElement elem, final Function1<? super FDValue, ? extends Boolean> isNonDefault, final Function0<? extends FDValue> getNewValue) {
    boolean _xifexpression = false;
    String _array = decl.getType().getArray();
    boolean _tripleNotEquals = (_array != null);
    if (_tripleNotEquals) {
      boolean _xblockexpression = false;
      {
        FDProperty prop = DeployModelBuilder.getProperty(elem, decl);
        FDValueArray _array_1 = prop.getValue().getArray();
        boolean _tripleEquals = (_array_1 == null);
        if (_tripleEquals) {
          FDComplexValue _value = prop.getValue();
          _value.setArray(FDeployFactory.eINSTANCE.createFDValueArray());
        }
        _xblockexpression = prop.getValue().getArray().getValues().add(getNewValue.apply());
      }
      _xifexpression = _xblockexpression;
    } else {
      FDComplexValue dflt = GenericPropertyAccessor.getDefault(decl);
      if (((dflt == null) || (isNonDefault.apply(dflt.getSingle())).booleanValue())) {
        FDProperty prop = DeployModelBuilder.getProperty(elem, decl);
        FDComplexValue _value = prop.getValue();
        _value.setSingle(getNewValue.apply());
      }
    }
    return Boolean.valueOf(_xifexpression);
  }
  
  /**
   * Create property object for element.
   */
  public static FDProperty getProperty(final FDElement element, final FDPropertyDecl decl) {
    final Function1<FDProperty, Boolean> _function = new Function1<FDProperty, Boolean>() {
      @Override
      public Boolean apply(final FDProperty p) {
        FDPropertyDecl _decl = p.getDecl();
        return Boolean.valueOf(Objects.equal(_decl, decl));
      }
    };
    FDProperty prop = IterableExtensions.<FDProperty>findFirst(element.getProperties().getItems(), _function);
    if ((prop == null)) {
      prop = FDeployFactory.eINSTANCE.createFDProperty();
      prop.setDecl(decl);
      element.getProperties().getItems().add(prop);
    }
    FDComplexValue _value = prop.getValue();
    boolean _tripleEquals = (_value == null);
    if (_tripleEquals) {
      prop.setValue(FDeployFactory.eINSTANCE.createFDComplexValue());
    }
    return prop;
  }
  
  private static FDInteger createFDValue(final int value) {
    FDInteger _xblockexpression = null;
    {
      final FDInteger v = FDeployFactory.eINSTANCE.createFDInteger();
      v.setValue(value);
      _xblockexpression = v;
    }
    return _xblockexpression;
  }
  
  private static FDBoolean createFDValue(final boolean value) {
    FDBoolean _xblockexpression = null;
    {
      final FDBoolean v = FDeployFactory.eINSTANCE.createFDBoolean();
      v.setValue(Boolean.toString(value));
      _xblockexpression = v;
    }
    return _xblockexpression;
  }
  
  private static FDString createFDValue(final String value) {
    FDString _xblockexpression = null;
    {
      final FDString v = FDeployFactory.eINSTANCE.createFDString();
      v.setValue(value);
      _xblockexpression = v;
    }
    return _xblockexpression;
  }
  
  private static FDGeneric createFDValue(final FDEnumerator value) {
    FDGeneric _xblockexpression = null;
    {
      final FDGeneric v = FDeployFactory.eINSTANCE.createFDGeneric();
      v.setValue(value);
      _xblockexpression = v;
    }
    return _xblockexpression;
  }
}
