/**
 * Copyright (c) 2015 itemis AG (http://www.itemis.de).
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.franca.deploymodel.dsl.generator;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.util.Set;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.generator.IFileSystemAccess;
import org.eclipse.xtext.generator.IFileSystemAccess2;
import org.eclipse.xtext.generator.IGenerator2;
import org.eclipse.xtext.generator.IGeneratorContext;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.StringExtensions;
import org.franca.deploymodel.core.FDPropertyHost;
import org.franca.deploymodel.dsl.fDeploy.FDDeclaration;
import org.franca.deploymodel.dsl.fDeploy.FDEnumType;
import org.franca.deploymodel.dsl.fDeploy.FDEnumerator;
import org.franca.deploymodel.dsl.fDeploy.FDExtensionRoot;
import org.franca.deploymodel.dsl.fDeploy.FDPropertyDecl;
import org.franca.deploymodel.dsl.fDeploy.FDSpecification;
import org.franca.deploymodel.dsl.fDeploy.FDType;
import org.franca.deploymodel.dsl.generator.internal.GeneratorHelper;
import org.franca.deploymodel.dsl.generator.internal.HelperGenerator;
import org.franca.deploymodel.dsl.generator.internal.HostLogic;
import org.franca.deploymodel.dsl.generator.internal.IDataGenerator;
import org.franca.deploymodel.dsl.generator.internal.ImportManager;
import org.franca.deploymodel.dsl.generator.internal.InterfaceAccessorGenerator;
import org.franca.deploymodel.dsl.generator.internal.OverwriteAccessorGenerator;
import org.franca.deploymodel.dsl.generator.internal.RootElementAccessorGenerator;
import org.franca.deploymodel.dsl.generator.internal.TypeCollectionAccessorGenerator;
import org.franca.deploymodel.extensions.ExtensionRegistry;
import org.franca.deploymodel.extensions.IFDeployExtension;

/**
 * Generator for PropertyAccessor class from deployment specification.
 * The deployment specification is defined as fdepl model (spec part).
 * 
 * The accessor classes generated by this generator will be useful when
 * traversing a fidl model and getting the deployment properties for this
 * model.
 */
@SuppressWarnings("all")
public class FDeployGenerator implements IGenerator2 {
  @Inject
  @Extension
  private ImportManager _importManager;
  
  @Inject
  private IDataGenerator genInterface;
  
  @Inject
  private HelperGenerator genHelper;
  
  @Inject
  private TypeCollectionAccessorGenerator genTCAcc;
  
  @Inject
  private InterfaceAccessorGenerator genInterfaceAcc;
  
  @Inject
  private RootElementAccessorGenerator genRootElementAcc;
  
  @Inject
  private OverwriteAccessorGenerator genOverwriteAcc;
  
  private final static String PA_INTERFACE = "Interface";
  
  private final static String PA_TYPE_COLLECTION = "TypeCollection";
  
  @Override
  public void beforeGenerate(final Resource input, final IFileSystemAccess2 fsa, final IGeneratorContext context) {
  }
  
  @Override
  public void doGenerate(final Resource resource, final IFileSystemAccess2 fsa, final IGeneratorContext context) {
    Iterable<FDSpecification> _filter = Iterables.<FDSpecification>filter(IteratorExtensions.<EObject>toIterable(resource.getAllContents()), FDSpecification.class);
    for (final FDSpecification m : _filter) {
      {
        this.generateAll(fsa, m);
        this.generateLegacy(fsa, m, FDeployGenerator.PA_INTERFACE);
        this.generateLegacy(fsa, m, FDeployGenerator.PA_TYPE_COLLECTION);
      }
    }
  }
  
  @Override
  public void afterGenerate(final Resource input, final IFileSystemAccess2 fsa, final IGeneratorContext context) {
  }
  
  public void generateAll(final IFileSystemAccess fsa, final FDSpecification spec) {
    this._importManager.initImportManager();
    final String path = GeneratorHelper.getPackage(spec).replace(".", "/");
    final String code = this.generateCombinedClass(spec).toString();
    String header = this.generateHeader(spec, null).toString();
    String _classname = GeneratorHelper.classname(spec);
    String _plus = ((path + "/") + _classname);
    String _plus_1 = (_plus + ".java");
    fsa.generateFile(_plus_1, (header + code));
  }
  
  private CharSequence generateCombinedClass(final FDSpecification spec) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("/**");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* This is a collection of all interfaces and classes needed for");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* accessing deployment properties according to deployment specification");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* \'");
    String _name = spec.getName();
    _builder.append(_name, " ");
    _builder.append("\'.");
    _builder.newLineIfNotEmpty();
    _builder.append(" ");
    _builder.append("*/");
    _builder.newLine();
    _builder.append("public class ");
    String _classname = GeneratorHelper.classname(spec);
    _builder.append(_classname);
    _builder.append(" {");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("\t");
    CharSequence _genEnumInterface = this.genEnumInterface(spec);
    _builder.append(_genEnumInterface, "\t");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("\t");
    CharSequence _generate = this.genInterface.generate(spec);
    _builder.append(_generate, "\t");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("\t");
    CharSequence _generate_1 = this.genHelper.generate(spec);
    _builder.append(_generate_1, "\t");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("\t");
    CharSequence _generate_2 = this.genTCAcc.generate(spec);
    _builder.append(_generate_2, "\t");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("\t");
    CharSequence _generate_3 = this.genInterfaceAcc.generate(spec);
    _builder.append(_generate_3, "\t");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    {
      Set<IFDeployExtension.RootDef> _keySet = ExtensionRegistry.getRoots().keySet();
      for(final IFDeployExtension.RootDef root : _keySet) {
        _builder.append("\t");
        final Function1<FDPropertyHost, Boolean> _function = new Function1<FDPropertyHost, Boolean>() {
          @Override
          public Boolean apply(final FDPropertyHost it) {
            return Boolean.valueOf(HostLogic.isHostFor(it, root));
          }
        };
        CharSequence _generate_4 = this.genRootElementAcc.generate(spec, 
          FDExtensionRoot.class, 
          root.getTag(), 
          root.getExtension().getShortDescription(), _function);
        _builder.append(_generate_4, "\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.newLine();
      }
    }
    {
      Set<EClass> _nonFrancaMixinRoots = ExtensionRegistry.getNonFrancaMixinRoots();
      for(final EClass clazz : _nonFrancaMixinRoots) {
        _builder.append("\t");
        final String prefix = ExtensionRegistry.getNonFrancaMixinPrefix(clazz);
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        Class<?> _instanceClass = clazz.getInstanceClass();
        final Function1<FDPropertyHost, Boolean> _function_1 = new Function1<FDPropertyHost, Boolean>() {
          @Override
          public Boolean apply(final FDPropertyHost it) {
            return Boolean.valueOf(HostLogic.isHostFor(it, clazz));
          }
        };
        CharSequence _generate_5 = this.genRootElementAcc.generate(spec, 
          ((Class<? extends EObject>) _instanceClass), prefix, 
          ExtensionRegistry.getMixinExtension(clazz).getShortDescription(), _function_1);
        _builder.append(_generate_5, "\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.newLine();
      }
    }
    _builder.append("\t");
    CharSequence _generate_6 = this.genOverwriteAcc.generate(spec);
    _builder.append(_generate_6, "\t");
    _builder.newLineIfNotEmpty();
    _builder.append("}");
    _builder.newLine();
    _builder.append("\t");
    _builder.newLine();
    return _builder;
  }
  
  private CharSequence genEnumInterface(final FDSpecification spec) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("/**");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* Enumerations for deployment specification ");
    String _name = spec.getName();
    _builder.append(_name, " ");
    _builder.append(".");
    _builder.newLineIfNotEmpty();
    _builder.append(" ");
    _builder.append("*/");
    _builder.newLine();
    _builder.append("public interface Enums");
    _builder.newLine();
    _builder.append("\t");
    {
      FDSpecification _base = spec.getBase();
      boolean _tripleNotEquals = (_base != null);
      if (_tripleNotEquals) {
        _builder.append("extends ");
        String _qualifiedClassname = GeneratorHelper.getQualifiedClassname(spec.getBase());
        _builder.append(_qualifiedClassname, "\t");
        _builder.append(".Enums");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("{");
    _builder.newLine();
    {
      EList<FDDeclaration> _declarations = spec.getDeclarations();
      for(final FDDeclaration d : _declarations) {
        {
          EList<FDPropertyDecl> _properties = d.getProperties();
          for(final FDPropertyDecl p : _properties) {
            _builder.append("\t");
            CharSequence _genStaticEnum = this.genStaticEnum(p, d.getHost());
            _builder.append(_genStaticEnum, "\t");
            _builder.newLineIfNotEmpty();
          }
        }
      }
    }
    _builder.append("}");
    _builder.newLine();
    return _builder;
  }
  
  private CharSequence genStaticEnum(final FDPropertyDecl it, final FDPropertyHost host) {
    CharSequence _xifexpression = null;
    boolean _isEnum = GeneratorHelper.isEnum(it);
    if (_isEnum) {
      CharSequence _xblockexpression = null;
      {
        final String enumType = StringExtensions.toFirstUpper(it.getName());
        FDType _complex = it.getType().getComplex();
        final FDEnumType enumerator = ((FDEnumType) _complex);
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("public enum ");
        _builder.append(enumType);
        _builder.append(" {");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        {
          EList<FDEnumerator> _enumerators = enumerator.getEnumerators();
          boolean _hasElements = false;
          for(final FDEnumerator e : _enumerators) {
            if (!_hasElements) {
              _hasElements = true;
            } else {
              _builder.appendImmediate(", ", "\t");
            }
            String _name = e.getName();
            _builder.append(_name, "\t");
          }
        }
        _builder.newLineIfNotEmpty();
        _builder.append("}");
        _builder.newLine();
        _builder.append(" ");
        _builder.newLine();
        _xblockexpression = _builder;
      }
      _xifexpression = _xblockexpression;
    } else {
      _xifexpression = "";
    }
    return _xifexpression;
  }
  
  /**
   * Generate legacy classes needed for users of Franca 0.9.1 and earlier.
   */
  public void generateLegacy(final IFileSystemAccess fsa, final FDSpecification spec, final String type) {
    final String path = GeneratorHelper.getPackage(spec).replace(".", "/");
    final String code = this.generateLegacyClass(spec, type).toString();
    String header = this.generateHeader(spec, type).toString();
    String _legacyClassname = this.getLegacyClassname(spec, type);
    String _plus = ((path + "/") + _legacyClassname);
    String _plus_1 = (_plus + ".java");
    fsa.generateFile(_plus_1, (header + code));
  }
  
  public CharSequence generateHeader(final FDSpecification spec, final String type) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("/*******************************************************************************");
    _builder.newLine();
    _builder.append("* This file has been generated by Franca\'s FDeployGenerator.");
    _builder.newLine();
    _builder.append("* Source: deployment specification \'");
    String _name = spec.getName();
    _builder.append(_name);
    _builder.append("\'");
    _builder.newLineIfNotEmpty();
    _builder.append("*******************************************************************************/");
    _builder.newLine();
    {
      boolean _isEmpty = GeneratorHelper.getPackage(spec).isEmpty();
      boolean _not = (!_isEmpty);
      if (_not) {
        _builder.append("package ");
        String _package = GeneratorHelper.getPackage(spec);
        _builder.append(_package);
        _builder.append(";");
        _builder.newLineIfNotEmpty();
        _builder.newLine();
      }
    }
    {
      if ((type != null)) {
        _builder.append("import org.franca.deploymodel.core.");
        String _supportingClass = this.getSupportingClass(type);
        _builder.append(_supportingClass);
        _builder.append(";");
        _builder.newLineIfNotEmpty();
      } else {
        CharSequence _genImports = this._importManager.genImports();
        _builder.append(_genImports);
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence generateLegacyClass(final FDSpecification spec, final String type) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("/**");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* Accessor for deployment properties for \'");
    String _name = spec.getName();
    _builder.append(_name, " ");
    _builder.append("\' specification");
    _builder.newLineIfNotEmpty();
    _builder.append(" ");
    _builder.append("*");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* @deprecated use class ");
    String _qualifiedClassname = GeneratorHelper.getQualifiedClassname(spec);
    _builder.append(_qualifiedClassname, " ");
    _builder.append(".");
    _builder.append(type, " ");
    _builder.append("PropertyAccessor instead");
    _builder.newLineIfNotEmpty();
    _builder.append(" ");
    _builder.append("*/");
    _builder.newLine();
    _builder.append("public class ");
    String _legacyClassname = this.getLegacyClassname(spec, type);
    _builder.append(_legacyClassname);
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("extends ");
    String _qualifiedClassname_1 = GeneratorHelper.getQualifiedClassname(spec);
    _builder.append(_qualifiedClassname_1, "\t");
    _builder.append(".");
    _builder.append(type, "\t");
    _builder.append("PropertyAccessor");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("implements ");
    String _qualifiedClassname_2 = GeneratorHelper.getQualifiedClassname(spec);
    _builder.append(_qualifiedClassname_2, "\t");
    _builder.append(".Enums");
    _builder.newLineIfNotEmpty();
    _builder.append("{");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("public ");
    String _legacyClassname_1 = this.getLegacyClassname(spec, type);
    _builder.append(_legacyClassname_1, "\t");
    _builder.append("(");
    String _supportingClass = this.getSupportingClass(type);
    _builder.append(_supportingClass, "\t");
    _builder.append(" target) {");
    _builder.newLineIfNotEmpty();
    _builder.append("\t\t");
    _builder.append("super(target);");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("}");
    _builder.newLine();
    _builder.append("}");
    _builder.newLine();
    _builder.append("   ");
    _builder.newLine();
    return _builder;
  }
  
  private String getLegacyClassname(final FDSpecification it, final String type) {
    String _xblockexpression = null;
    {
      final int sep = it.getName().lastIndexOf(".");
      String _xifexpression = null;
      if ((sep > 0)) {
        _xifexpression = it.getName().substring((sep + 1));
      } else {
        _xifexpression = it.getName();
      }
      final String basename = _xifexpression;
      String _firstUpper = StringExtensions.toFirstUpper(basename);
      String _plus = (_firstUpper + type);
      _xblockexpression = (_plus + "PropertyAccessor");
    }
    return _xblockexpression;
  }
  
  private String getSupportingClass(final String type) {
    String _switchResult = null;
    boolean _matched = false;
    if (Objects.equal(type, FDeployGenerator.PA_TYPE_COLLECTION)) {
      _matched=true;
      _switchResult = "FDeployedTypeCollection";
    }
    if (!_matched) {
      if (Objects.equal(type, FDeployGenerator.PA_INTERFACE)) {
        _matched=true;
        _switchResult = "FDeployedInterface";
      }
    }
    if (!_matched) {
      _switchResult = "";
    }
    return _switchResult;
  }
}
