/**
 * Copyright (c) 2015 itemis AG (http://www.itemis.de).
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.franca.deploymodel.dsl.generator.internal;

import com.google.common.collect.Iterables;
import java.util.Collections;
import java.util.Set;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.franca.core.franca.FArgument;
import org.franca.core.franca.FAttribute;
import org.franca.core.franca.FBroadcast;
import org.franca.core.franca.FEnumerationType;
import org.franca.core.franca.FEnumerator;
import org.franca.core.franca.FField;
import org.franca.core.franca.FInterface;
import org.franca.core.franca.FMapType;
import org.franca.core.franca.FMethod;
import org.franca.core.franca.FStructType;
import org.franca.core.franca.FTypeCollection;
import org.franca.core.franca.FTypeDef;
import org.franca.core.franca.FUnionType;
import org.franca.deploymodel.core.FDPropertyHost;
import org.franca.deploymodel.dsl.fDeploy.FDBuiltInPropertyHost;
import org.franca.deploymodel.dsl.fDeploy.FDeployPackage;
import org.franca.deploymodel.dsl.generator.internal.SuperclassFinder;
import org.franca.deploymodel.extensions.ExtensionRegistry;
import org.franca.deploymodel.extensions.IFDeployExtension;

/**
 * This class defines how deployment properties are mapped to Franca IDL objects
 * by the PropertyAccessor generator.<p/>
 * 
 * E.g., each property host implies a defined argument type for the property
 * getter in the accessor. E.g., for the host 'arguments' the Franca type FArgument
 * is used.
 */
@SuppressWarnings("all")
public class HostLogic {
  /**
   * The context for which an argument type for a given host should be computed.</p>
   * 
   * NB: FRANCA_INTERFACE logically includes FRANCA_TYPE, but not vice versa.
   */
  public enum Context {
    FRANCA_TYPE,
    
    FRANCA_INTERFACE,
    
    NON_FRANCA;
  }
  
  private final static Set<Class<? extends EObject>> interfaceSpecificClasses = Collections.<Class<? extends EObject>>unmodifiableSet(CollectionLiterals.<Class<? extends EObject>>newHashSet(FInterface.class, FAttribute.class, FMethod.class, FBroadcast.class, FArgument.class));
  
  /**
   * Get the argument type for the property accessor method for a given deployment host.
   */
  public static Class<? extends EObject> getArgumentType(final FDPropertyHost host, final HostLogic.Context context) {
    Class<? extends EObject> _xblockexpression = null;
    {
      final FDBuiltInPropertyHost builtIn = host.getBuiltIn();
      Class<? extends EObject> _xifexpression = null;
      if ((builtIn != null)) {
        Class<? extends EObject> _switchResult = null;
        if (builtIn != null) {
          switch (builtIn) {
            case TYPE_COLLECTIONS:
              _switchResult = FTypeCollection.class;
              break;
            case INTERFACES:
              _switchResult = FInterface.class;
              break;
            case ATTRIBUTES:
              _switchResult = FAttribute.class;
              break;
            case METHODS:
              _switchResult = FMethod.class;
              break;
            case BROADCASTS:
              _switchResult = FBroadcast.class;
              break;
            case ARGUMENTS:
              _switchResult = FArgument.class;
              break;
            case STRUCTS:
              _switchResult = FStructType.class;
              break;
            case UNIONS:
              _switchResult = FUnionType.class;
              break;
            case STRUCT_FIELDS:
              _switchResult = FField.class;
              break;
            case UNION_FIELDS:
              _switchResult = FField.class;
              break;
            case FIELDS:
              _switchResult = FField.class;
              break;
            case ENUMERATIONS:
              _switchResult = FEnumerationType.class;
              break;
            case ENUMERATORS:
              _switchResult = FEnumerator.class;
              break;
            case MAPS:
              _switchResult = FMapType.class;
              break;
            case MAP_KEYS:
              _switchResult = FMapType.class;
              break;
            case MAP_VALUES:
              _switchResult = FMapType.class;
              break;
            case TYPEDEFS:
              _switchResult = FTypeDef.class;
              break;
            default:
              _switchResult = EObject.class;
              break;
          }
        } else {
          _switchResult = EObject.class;
        }
        _xifexpression = _switchResult;
      } else {
        _xifexpression = HostLogic.getArgumentTypeForExtensionHost(host, context);
      }
      final Class<? extends EObject> result = _xifexpression;
      Class<? extends EObject> _xifexpression_1 = null;
      if ((result == null)) {
        _xifexpression_1 = null;
      } else {
        Class<? extends EObject> _xifexpression_2 = null;
        if (((context != HostLogic.Context.FRANCA_INTERFACE) && HostLogic.interfaceSpecificClasses.contains(result))) {
          _xifexpression_2 = null;
        } else {
          _xifexpression_2 = result;
        }
        _xifexpression_1 = _xifexpression_2;
      }
      _xblockexpression = _xifexpression_1;
    }
    return _xblockexpression;
  }
  
  private static Class<? extends EObject> getArgumentTypeForExtensionHost(final FDPropertyHost host, final HostLogic.Context context) {
    Class<EObject> _xblockexpression = null;
    {
      final IFDeployExtension.Host hostDef = ExtensionRegistry.findHost(host.getName());
      if ((hostDef != null)) {
        final Set<EClass> classes = ExtensionRegistry.getHostingClasses(hostDef);
        Iterable<EClass> _xifexpression = null;
        if ((context == HostLogic.Context.NON_FRANCA)) {
          final Function1<EClass, Boolean> _function = new Function1<EClass, Boolean>() {
            @Override
            public Boolean apply(final EClass it) {
              return Boolean.valueOf((ExtensionRegistry.isNonFrancaMixinHost(it) || FDeployPackage.eINSTANCE.getFDAbstractExtensionElement().isSuperTypeOf(it)));
            }
          };
          _xifexpression = IterableExtensions.<EClass>filter(classes, _function);
        } else {
          final Function1<EClass, Boolean> _function_1 = new Function1<EClass, Boolean>() {
            @Override
            public Boolean apply(final EClass it) {
              boolean _isNonFrancaMixinHost = ExtensionRegistry.isNonFrancaMixinHost(it);
              return Boolean.valueOf((!_isNonFrancaMixinHost));
            }
          };
          _xifexpression = IterableExtensions.<EClass>filter(classes, _function_1);
        }
        final Iterable<EClass> filtered = _xifexpression;
        boolean _isEmpty = IterableExtensions.isEmpty(filtered);
        if (_isEmpty) {
          return null;
        }
        final Function1<EClass, EClassifier> _function_2 = new Function1<EClass, EClassifier>() {
          @Override
          public EClassifier apply(final EClass it) {
            return ExtensionRegistry.getAccessorArgumentType(it);
          }
        };
        final Iterable<EClassifier> targetClasses = IterableExtensions.<EClass, EClassifier>map(filtered, _function_2);
        boolean _isEmpty_1 = IterableExtensions.isEmpty(targetClasses);
        boolean _not = (!_isEmpty_1);
        if (_not) {
          final SuperclassFinder sd = new SuperclassFinder();
          final EClassifier superclass = sd.findCommonSuperclass(targetClasses);
          if ((superclass != null)) {
            final Class<?> instClass = superclass.getInstanceClass();
            boolean _isAssignableFrom = EObject.class.isAssignableFrom(instClass);
            if (_isAssignableFrom) {
              return ((Class<? extends EObject>) instClass);
            }
          }
        }
      }
      _xblockexpression = EObject.class;
    }
    return _xblockexpression;
  }
  
  public static boolean isInterfaceOnly(final FDPropertyHost host) {
    return ((HostLogic.getArgumentType(host, HostLogic.Context.FRANCA_TYPE) == null) && 
      (HostLogic.getArgumentType(host, HostLogic.Context.FRANCA_INTERFACE) != null));
  }
  
  public static boolean isHostFor(final FDPropertyHost host, final IFDeployExtension.AbstractElementDef elementDef) {
    boolean _xifexpression = false;
    FDBuiltInPropertyHost _builtIn = host.getBuiltIn();
    boolean _tripleNotEquals = (_builtIn != null);
    if (_tripleNotEquals) {
      _xifexpression = false;
    } else {
      boolean _xblockexpression = false;
      {
        final IFDeployExtension.Host hostDef = ExtensionRegistry.findHost(host.getName());
        _xblockexpression = ExtensionRegistry.hasHostSubtree(elementDef, hostDef);
      }
      _xifexpression = _xblockexpression;
    }
    return _xifexpression;
  }
  
  public static boolean isHostFor(final FDPropertyHost host, final EClass mixinRoot) {
    boolean _xifexpression = false;
    FDBuiltInPropertyHost _builtIn = host.getBuiltIn();
    boolean _tripleNotEquals = (_builtIn != null);
    if (_tripleNotEquals) {
      _xifexpression = false;
    } else {
      boolean _xblockexpression = false;
      {
        final IFDeployExtension.Host hostDef = ExtensionRegistry.findHost(host.getName());
        final Iterable<EClass> classes = ExtensionRegistry.getMixinClasses(mixinRoot);
        final Function1<EClass, Iterable<IFDeployExtension.Host>> _function = new Function1<EClass, Iterable<IFDeployExtension.Host>>() {
          @Override
          public Iterable<IFDeployExtension.Host> apply(final EClass it) {
            return ExtensionRegistry.getAdditionalHosts(it);
          }
        };
        final Set<IFDeployExtension.Host> supportedHosts = IterableExtensions.<IFDeployExtension.Host>toSet(Iterables.<IFDeployExtension.Host>concat(IterableExtensions.<EClass, Iterable<IFDeployExtension.Host>>map(classes, _function)));
        _xblockexpression = supportedHosts.contains(hostDef);
      }
      _xifexpression = _xblockexpression;
    }
    return _xifexpression;
  }
}
