/**
 * Copyright (c) 2018 itemis AG (http://www.itemis.de).
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.franca.deploymodel.dsl.generator.internal;

import com.google.common.base.Objects;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EcorePackage;
import org.eclipse.xtext.xbase.lib.CollectionExtensions;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.Functions.Function2;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.MapExtensions;
import org.eclipse.xtext.xbase.lib.Pair;

/**
 * Analyse EClassifier super type hierarchy and find a common superclass for
 * a given input set of classifiers.</p>
 * 
 * @author Klaus Birken (itemis AG)
 */
@SuppressWarnings("all")
public class SuperclassFinder {
  private final Map<EClassifier, List<Integer>> stepsUntilReached = CollectionLiterals.<EClassifier, List<Integer>>newHashMap();
  
  /**
   * Find a common superclass for a given input set of EClassifiers.</p>
   * 
   * This method finds the common superclass of the given set of input EClassifiers.
   * Among all common superclasses, it chooses the one where the overall number
   * of inheritance relations (over all input classes) is minimal.</p>
   * 
   * Note that due to multiple inheritance this is not necessarily unique.
   * If there are more than one results, we pick the one with the lexicographically
   * minimal name.</p>
   * 
   * The fallback result is EObject, in case no more specific common superclass
   * can be found.</p>
   */
  public EClassifier findCommonSuperclass(final Iterable<EClassifier> classes) {
    EClassifier _xblockexpression = null;
    {
      boolean _isEmpty = IterableExtensions.isEmpty(classes);
      if (_isEmpty) {
        return null;
      }
      int _size = IterableExtensions.size(classes);
      boolean _equals = (_size == 1);
      if (_equals) {
        return ((EClassifier[])Conversions.unwrapArray(classes, EClassifier.class))[0];
      }
      final Iterable<EClassifier> result = this.findSuperclassSet(classes);
      EClassifier _xifexpression = null;
      boolean _isEmpty_1 = IterableExtensions.isEmpty(result);
      if (_isEmpty_1) {
        _xifexpression = EcorePackage.eINSTANCE.getEObject();
      } else {
        final Comparator<EClassifier> _function = new Comparator<EClassifier>() {
          @Override
          public int compare(final EClassifier a, final EClassifier b) {
            return a.getName().compareTo(b.getName());
          }
        };
        _xifexpression = IterableExtensions.<EClassifier>head(IterableExtensions.<EClassifier>sortWith(result, _function));
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  /**
   * Find the set of all superclasses of the classifiers in the given set
   * which have a minimal overall number of inheritance steps.</p>
   */
  private Iterable<EClassifier> findSuperclassSet(final Iterable<EClassifier> classes) {
    Iterable<EClassifier> _xblockexpression = null;
    {
      this.stepsUntilReached.clear();
      final Consumer<EClassifier> _function = new Consumer<EClassifier>() {
        @Override
        public void accept(final EClassifier it) {
          SuperclassFinder.this.traverse(it);
        }
      };
      classes.forEach(_function);
      final int n = IterableExtensions.size(classes);
      final Function2<EClassifier, List<Integer>, Boolean> _function_1 = new Function2<EClassifier, List<Integer>, Boolean>() {
        @Override
        public Boolean apply(final EClassifier clazz, final List<Integer> reach) {
          int _size = reach.size();
          return Boolean.valueOf((_size == n));
        }
      };
      final Set<EClassifier> candidates = MapExtensions.<EClassifier, List<Integer>>filter(this.stepsUntilReached, _function_1).keySet();
      final Function1<EClassifier, EClassifier> _function_2 = new Function1<EClassifier, EClassifier>() {
        @Override
        public EClassifier apply(final EClassifier it) {
          return it;
        }
      };
      final Function1<EClassifier, Integer> _function_3 = new Function1<EClassifier, Integer>() {
        @Override
        public Integer apply(final EClassifier it) {
          final Function2<Integer, Integer, Integer> _function = new Function2<Integer, Integer, Integer>() {
            @Override
            public Integer apply(final Integer p1, final Integer p2) {
              return Integer.valueOf(((p1).intValue() + (p2).intValue()));
            }
          };
          return IterableExtensions.<Integer>reduce(SuperclassFinder.this.stepsUntilReached.get(it), _function);
        }
      };
      final Map<EClassifier, Integer> sums = IterableExtensions.<EClassifier, EClassifier, Integer>toMap(candidates, _function_2, _function_3);
      boolean _isEmpty = sums.isEmpty();
      if (_isEmpty) {
        return CollectionLiterals.<EClassifier>newHashSet();
      }
      final Integer minSteps = IterableExtensions.<Integer>min(sums.values());
      final Function1<EClassifier, Boolean> _function_4 = new Function1<EClassifier, Boolean>() {
        @Override
        public Boolean apply(final EClassifier it) {
          Integer _get = sums.get(it);
          return Boolean.valueOf(Objects.equal(_get, minSteps));
        }
      };
      _xblockexpression = IterableExtensions.<EClassifier>filter(sums.keySet(), _function_4);
    }
    return _xblockexpression;
  }
  
  /**
   * Traverse inheritance graph and find number of inheritance steps from the starting classifier.</p>
   */
  private void traverse(final EClassifier start) {
    final Queue<Pair<EClassifier, Integer>> work = CollectionLiterals.<Pair<EClassifier, Integer>>newLinkedList();
    final Set<EClassifier> visited = CollectionLiterals.<EClassifier>newHashSet();
    Pair<EClassifier, Integer> _mappedTo = Pair.<EClassifier, Integer>of(start, Integer.valueOf(0));
    CollectionExtensions.<Pair<EClassifier, Integer>>addAll(work, _mappedTo);
    while ((!work.isEmpty())) {
      {
        final Pair<EClassifier, Integer> w = work.poll();
        final EClassifier clazz = w.getKey();
        final Integer nSteps = w.getValue();
        boolean _contains = visited.contains(clazz);
        boolean _not = (!_contains);
        if (_not) {
          visited.add(clazz);
          boolean _containsKey = this.stepsUntilReached.containsKey(clazz);
          boolean _not_1 = (!_containsKey);
          if (_not_1) {
            this.stepsUntilReached.put(clazz, CollectionLiterals.<Integer>newArrayList());
          }
          this.stepsUntilReached.get(clazz).add(nSteps);
          if ((clazz instanceof EClass)) {
            EList<EClass> _eSuperTypes = ((EClass)clazz).getESuperTypes();
            for (final EClass base : _eSuperTypes) {
              Pair<EClassifier, Integer> _mappedTo_1 = Pair.<EClassifier, Integer>of(base, Integer.valueOf(((nSteps).intValue() + 1)));
              work.add(_mappedTo_1);
            }
          }
        }
      }
    }
  }
}
