/**
 * Copyright (c) 2018 itemis AG (http://www.itemis.de).
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.franca.deploymodel.dsl.validation;

import com.google.common.collect.Lists;
import java.util.Collection;
import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.validation.CheckMode;
import org.eclipse.xtext.validation.ValidationMessageAcceptor;
import org.franca.core.FrancaModelExtensions;
import org.franca.core.franca.FArgument;
import org.franca.core.franca.FArrayType;
import org.franca.core.franca.FAttribute;
import org.franca.core.franca.FBroadcast;
import org.franca.core.franca.FEnumerationType;
import org.franca.core.franca.FEnumerator;
import org.franca.core.franca.FField;
import org.franca.core.franca.FInterface;
import org.franca.core.franca.FMapType;
import org.franca.core.franca.FMethod;
import org.franca.core.franca.FStructType;
import org.franca.core.franca.FType;
import org.franca.core.franca.FTypeDef;
import org.franca.core.franca.FUnionType;
import org.franca.deploymodel.core.FDModelUtils;
import org.franca.deploymodel.core.FDPropertyHost;
import org.franca.deploymodel.core.PropertyMappings;
import org.franca.deploymodel.dsl.FDMapper;
import org.franca.deploymodel.dsl.FDSpecificationExtender;
import org.franca.deploymodel.dsl.fDeploy.FDAbstractExtensionElement;
import org.franca.deploymodel.dsl.fDeploy.FDArgument;
import org.franca.deploymodel.dsl.fDeploy.FDArgumentList;
import org.franca.deploymodel.dsl.fDeploy.FDArray;
import org.franca.deploymodel.dsl.fDeploy.FDAttribute;
import org.franca.deploymodel.dsl.fDeploy.FDBoolean;
import org.franca.deploymodel.dsl.fDeploy.FDBroadcast;
import org.franca.deploymodel.dsl.fDeploy.FDBuiltInPropertyHost;
import org.franca.deploymodel.dsl.fDeploy.FDComplexValue;
import org.franca.deploymodel.dsl.fDeploy.FDDeclaration;
import org.franca.deploymodel.dsl.fDeploy.FDElement;
import org.franca.deploymodel.dsl.fDeploy.FDEnumType;
import org.franca.deploymodel.dsl.fDeploy.FDEnumValue;
import org.franca.deploymodel.dsl.fDeploy.FDEnumeration;
import org.franca.deploymodel.dsl.fDeploy.FDEnumerationOverwrites;
import org.franca.deploymodel.dsl.fDeploy.FDEnumerator;
import org.franca.deploymodel.dsl.fDeploy.FDExtensionElement;
import org.franca.deploymodel.dsl.fDeploy.FDExtensionRoot;
import org.franca.deploymodel.dsl.fDeploy.FDExtensionType;
import org.franca.deploymodel.dsl.fDeploy.FDField;
import org.franca.deploymodel.dsl.fDeploy.FDInteger;
import org.franca.deploymodel.dsl.fDeploy.FDInterface;
import org.franca.deploymodel.dsl.fDeploy.FDInterfaceRef;
import org.franca.deploymodel.dsl.fDeploy.FDMap;
import org.franca.deploymodel.dsl.fDeploy.FDMapKey;
import org.franca.deploymodel.dsl.fDeploy.FDMapValue;
import org.franca.deploymodel.dsl.fDeploy.FDMethod;
import org.franca.deploymodel.dsl.fDeploy.FDModel;
import org.franca.deploymodel.dsl.fDeploy.FDOverwriteElement;
import org.franca.deploymodel.dsl.fDeploy.FDPlainTypeOverwrites;
import org.franca.deploymodel.dsl.fDeploy.FDPredefinedTypeId;
import org.franca.deploymodel.dsl.fDeploy.FDProperty;
import org.franca.deploymodel.dsl.fDeploy.FDPropertyDecl;
import org.franca.deploymodel.dsl.fDeploy.FDPropertyFlag;
import org.franca.deploymodel.dsl.fDeploy.FDPropertySet;
import org.franca.deploymodel.dsl.fDeploy.FDRootElement;
import org.franca.deploymodel.dsl.fDeploy.FDSpecification;
import org.franca.deploymodel.dsl.fDeploy.FDString;
import org.franca.deploymodel.dsl.fDeploy.FDStruct;
import org.franca.deploymodel.dsl.fDeploy.FDStructOverwrites;
import org.franca.deploymodel.dsl.fDeploy.FDType;
import org.franca.deploymodel.dsl.fDeploy.FDTypeOverwrites;
import org.franca.deploymodel.dsl.fDeploy.FDTypeRef;
import org.franca.deploymodel.dsl.fDeploy.FDTypedef;
import org.franca.deploymodel.dsl.fDeploy.FDTypes;
import org.franca.deploymodel.dsl.fDeploy.FDUnion;
import org.franca.deploymodel.dsl.fDeploy.FDUnionOverwrites;
import org.franca.deploymodel.dsl.fDeploy.FDValue;
import org.franca.deploymodel.dsl.fDeploy.FDValueArray;
import org.franca.deploymodel.dsl.fDeploy.FDeployPackage;
import org.franca.deploymodel.dsl.validation.AbstractFDeployValidator;
import org.franca.deploymodel.dsl.validation.FDeployValidatorAux;
import org.franca.deploymodel.dsl.validation.FrancaQuickFixConstants;
import org.franca.deploymodel.dsl.validation.IFDeployExternalValidator;
import org.franca.deploymodel.dsl.validation.PropertyDefChecker;
import org.franca.deploymodel.dsl.validation.ValidationHelpers;
import org.franca.deploymodel.dsl.validation.ValidationMessageReporter;
import org.franca.deploymodel.dsl.validation.internal.ValidatorRegistry;
import org.franca.deploymodel.extensions.ExtensionRegistry;
import org.franca.deploymodel.extensions.IFDeployExtension;

/**
 * This class contains custom validation rules for the Franca deployment DSL.</p>
 * 
 * See https://www.eclipse.org/Xtext/documentation/303_runtime_concepts.html#validation
 * </p>
 */
@SuppressWarnings("all")
public class FDeployValidator extends AbstractFDeployValidator implements ValidationMessageReporter {
  public final static String UPPERCASE_PROPERTYNAME_QUICKFIX = "UPPERCASE_PROPERTYNAME_QUICKFIX";
  
  public final static String METHOD_ARGUMENT_QUICKFIX = "METHOD_ARGUMENT_QUICKFIX";
  
  public final static String METHOD_ARGUMENT_QUICKFIX_MESSAGE = "Method argument is missing for method ";
  
  public final static String BROADCAST_ARGUMENT_QUICKFIX = "BROADCAST_ARGUMENT_QUICKFIX";
  
  public final static String BROADCAST_ARGUMENT_QUICKFIX_MESSAGE = "Broadcast argument is missing for broadcast ";
  
  public final static String COMPOUND_FIELD_QUICKFIX = "COMPOUND_FIELD_QUICKFIX";
  
  public final static String COMPOUND_FIELD_QUICKFIX_MESSAGE = "Field is missing for compound ";
  
  public final static String ENUMERATOR_ENUM_QUICKFIX = "ENUMERATOR_ENUM_QUICKFIX";
  
  public final static String ENUMERATOR_ENUM_QUICKFIX_MESSAGE = "Enumerator element is missing for enum ";
  
  public final static String MAP_KEY_QUICKFIX = "MAP_KEY_QUICKFIX";
  
  public final static String MAP_KEY_QUICKFIX_MESSAGE = "Map key section is missing for map ";
  
  public final static String MAP_VALUE_QUICKFIX = "MAP_VALUE_QUICKFIX";
  
  public final static String MAP_VALUE_QUICKFIX_MESSAGE = "Map value section is missing for map ";
  
  public final static String MANDATORY_PROPERTY_QUICKFIX = "MANDATORY_PROPERTIES_QUICKFIX";
  
  public final static String MANDATORY_PROPERTY_QUICKFIX_MESSAGE = "Mandatory properties are missing for element ";
  
  public final static String DEPLOYMENT_ELEMENT_QUICKFIX = "DEPLOYMENT_ELEMENT_QUICKFIX";
  
  public final static String DEPLOYMENT_ELEMENT_QUICKFIX_MESSAGE = "Missing specification element ";
  
  public final static String DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX = "DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX";
  
  public final static String DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX_MESSAGE = "There are multiple issues with element ";
  
  FDeployValidatorAux deployValidator = new FDeployValidatorAux(this);
  
  /**
   * Call external validators (those which have been installed via the Eclipse
   * 'deploymentValidator' extension point).
   */
  @Check
  public void checkExtensionValidators(final FDModel model) {
    CheckMode mode = this.getCheckMode();
    Collection<IFDeployExternalValidator> _get = ValidatorRegistry.getValidatorMap().get(mode);
    for (final IFDeployExternalValidator validator : _get) {
      validator.validateModel(model, this.getMessageAcceptor());
    }
  }
  
  @Check
  public void checkSpecNamesUnique(final FDModel model) {
    ValidationHelpers.<FDSpecification>checkDuplicates(this, model.getSpecifications(), FDeployPackage.Literals.FD_SPECIFICATION__NAME, "specification name");
  }
  
  @Check
  public void checkPropertyHosts(final FDDeclaration decl) {
    FDPropertyHost host = decl.getHost();
    FDBuiltInPropertyHost _builtIn = host.getBuiltIn();
    boolean _tripleEquals = (_builtIn == null);
    if (_tripleEquals) {
      IFDeployExtension.Host _findHost = ExtensionRegistry.findHost(host.getName());
      boolean _tripleEquals_1 = (_findHost == null);
      if (_tripleEquals_1) {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("Invalid property host \'");
        String _name = host.getName();
        _builder.append(_name);
        _builder.append("\'");
        this.error(_builder.toString(), decl, FDeployPackage.Literals.FD_DECLARATION__HOST, (-1));
      }
    }
  }
  
  @Check
  public void checkExtensionRootAvailable(final FDExtensionRoot root) {
    String tag = root.getTag();
    IFDeployExtension.RootDef rootDef = ExtensionRegistry.findRoot(tag);
    if ((rootDef == null)) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Invalid root \'");
      _builder.append(tag);
      _builder.append("\', no matching deployment extension has been configured");
      this.error(_builder.toString(), root, 
        FDeployPackage.Literals.FD_ABSTRACT_EXTENSION_ELEMENT__TAG, (-1));
    } else {
      if (((root.getName() != null) && (!rootDef.mayHaveName()))) {
        StringConcatenation _builder_1 = new StringConcatenation();
        _builder_1.append("Root \'");
        _builder_1.append(tag);
        _builder_1.append("\' must not have a name");
        this.error(_builder_1.toString(), root, 
          FDeployPackage.Literals.FD_ROOT_ELEMENT__NAME, (-1));
      }
      if ((rootDef.mustHaveName() && (root.getName() == null))) {
        StringConcatenation _builder_2 = new StringConcatenation();
        _builder_2.append("Root \'");
        _builder_2.append(tag);
        _builder_2.append("\' must have a name");
        this.error(_builder_2.toString(), root, 
          FDeployPackage.Literals.FD_ABSTRACT_EXTENSION_ELEMENT__TAG, (-1));
      }
    }
  }
  
  @Check
  public void checkExtensionElementHierarchy(final FDExtensionElement elem) {
    String tag = elem.getTag();
    EObject _eContainer = elem.eContainer();
    FDAbstractExtensionElement parent = ((FDAbstractExtensionElement) _eContainer);
    IFDeployExtension.AbstractElementDef parentDef = ExtensionRegistry.getElement(parent);
    boolean _hasChild = this.hasChild(parentDef, tag);
    boolean _not = (!_hasChild);
    if (_not) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Invalid element tag \'");
      _builder.append(tag);
      _builder.append("\' for parent \'");
      String _tag = parentDef.getTag();
      _builder.append(_tag);
      _builder.append("\'");
      this.error(_builder.toString(), elem, 
        FDeployPackage.Literals.FD_ABSTRACT_EXTENSION_ELEMENT__TAG, (-1));
      return;
    }
    IFDeployExtension.AbstractElementDef elemDef = ExtensionRegistry.getElement(elem);
    if (((elem.getName() != null) && (!elemDef.mayHaveName()))) {
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append("Element \'");
      _builder_1.append(tag);
      _builder_1.append("\' must not have a name");
      this.error(_builder_1.toString(), elem, 
        FDeployPackage.Literals.FD_EXTENSION_ELEMENT__NAME, (-1));
    }
    if ((elemDef.mustHaveName() && (elem.getName() == null))) {
      StringConcatenation _builder_2 = new StringConcatenation();
      _builder_2.append("Element \'");
      _builder_2.append(tag);
      _builder_2.append("\' must have a name");
      this.error(_builder_2.toString(), elem, 
        FDeployPackage.Literals.FD_ABSTRACT_EXTENSION_ELEMENT__TAG, (-1));
    }
  }
  
  private boolean hasChild(final IFDeployExtension.AbstractElementDef elemDef, final String tag) {
    Collection<IFDeployExtension.ElementDef> _children = elemDef.getChildren();
    for (final IFDeployExtension.AbstractElementDef c : _children) {
      boolean _equals = c.getTag().equals(tag);
      if (_equals) {
        return true;
      }
    }
    return false;
  }
  
  @Check
  public void checkRootElementNamesUnique(final FDModel model) {
    ValidationHelpers.<FDRootElement>checkDuplicates(this, model.getDeployments(), 
      FDeployPackage.Literals.FD_ROOT_ELEMENT__NAME, "definition name");
  }
  
  @Check
  public void checkRootElement(final FDRootElement elem) {
    this.deployValidator.checkRootElement(elem);
  }
  
  @Check
  public void checkMethodArgs(final FDMethod method) {
    FDArgumentList _inArguments = method.getInArguments();
    boolean _tripleNotEquals = (_inArguments != null);
    if (_tripleNotEquals) {
      EList<FDArgument> _arguments = method.getInArguments().getArguments();
      for (final FDArgument arg : _arguments) {
        boolean _contains = method.getTarget().getInArgs().contains(arg.getTarget());
        boolean _not = (!_contains);
        if (_not) {
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("Invalid input argument \'");
          String _name = arg.getTarget().getName();
          _builder.append(_name);
          _builder.append("\'");
          this.error(_builder.toString(), arg, FDeployPackage.Literals.FD_ARGUMENT__TARGET, (-1));
        }
      }
    }
    FDArgumentList _outArguments = method.getOutArguments();
    boolean _tripleNotEquals_1 = (_outArguments != null);
    if (_tripleNotEquals_1) {
      EList<FDArgument> _arguments_1 = method.getOutArguments().getArguments();
      for (final FDArgument arg_1 : _arguments_1) {
        boolean _contains_1 = method.getTarget().getOutArgs().contains(arg_1.getTarget());
        boolean _not_1 = (!_contains_1);
        if (_not_1) {
          StringConcatenation _builder_1 = new StringConcatenation();
          _builder_1.append("Invalid output argument \'");
          String _name_1 = arg_1.getTarget().getName();
          _builder_1.append(_name_1);
          _builder_1.append("\'");
          this.error(_builder_1.toString(), arg_1, FDeployPackage.Literals.FD_ARGUMENT__TARGET, (-1));
        }
      }
    }
  }
  
  @Check
  public void checkBroadcastArgs(final FDBroadcast broadcast) {
    FDArgumentList _outArguments = broadcast.getOutArguments();
    boolean _tripleNotEquals = (_outArguments != null);
    if (_tripleNotEquals) {
      EList<FDArgument> _arguments = broadcast.getOutArguments().getArguments();
      for (final FDArgument arg : _arguments) {
        boolean _contains = broadcast.getTarget().getOutArgs().contains(arg.getTarget());
        boolean _not = (!_contains);
        if (_not) {
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("Invalid output argument \'");
          String _name = arg.getTarget().getName();
          _builder.append(_name);
          _builder.append("\'");
          this.error(_builder.toString(), arg, FDeployPackage.Literals.FD_ARGUMENT__TARGET, (-1));
        }
      }
    }
  }
  
  @Check
  public void checkDuplicateProperties(final FDPropertySet properties) {
    final ValidationHelpers.NameList names = ValidationHelpers.createNameList();
    EList<FDProperty> _items = properties.getItems();
    for (final FDProperty p : _items) {
      boolean _eIsProxy = p.getDecl().eIsProxy();
      if (_eIsProxy) {
      } else {
        names.add(p, p.getDecl().getName());
      }
    }
    ValidationHelpers.checkDuplicates(this, names, FDeployPackage.Literals.FD_PROPERTY__DECL, "property");
  }
  
  @Check
  public void checkPropertyName(final FDPropertyDecl prop) {
    boolean _isUpperCase = Character.isUpperCase(prop.getName().charAt(0));
    boolean _not = (!_isUpperCase);
    if (_not) {
      this.error("Property names must begin with an uppercase character", 
        FDeployPackage.Literals.FD_PROPERTY_DECL__NAME, FDeployValidator.UPPERCASE_PROPERTYNAME_QUICKFIX, prop.getName());
    }
  }
  
  @Check
  public void checkClashingProperties(final FDSpecification spec) {
    this.deployValidator.checkClashingProperties(spec);
  }
  
  @Check
  public void checkBaseSpec(final FDSpecification spec) {
    final FDSpecification cycleSpec = this.deployValidator.getCyclicBaseSpec(spec);
    if ((cycleSpec != null)) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Inheritance cycle for specification ");
      String _name = cycleSpec.getName();
      _builder.append(_name);
      this.error(_builder.toString(), cycleSpec, 
        FDeployPackage.Literals.FD_SPECIFICATION__BASE, (-1));
      return;
    }
  }
  
  @Check
  public void checkPropertiesComplete(final FDExtensionRoot elem) {
    final FDSpecification spec = FDModelUtils.getRootElement(elem).getSpec();
    this.checkSpecificationElementProperties(spec, elem, FDeployPackage.Literals.FD_ROOT_ELEMENT__NAME, spec.getName());
    EList<FDExtensionElement> _elements = elem.getElements();
    for (final FDExtensionElement child : _elements) {
      this.checkExtensionElementProperties(spec, child);
    }
  }
  
  private void checkExtensionElementProperties(final FDSpecification spec, final FDExtensionElement elem) {
    this.checkSpecificationElementProperties(spec, elem, FDeployPackage.Literals.FD_ABSTRACT_EXTENSION_ELEMENT__TAG, spec.getName());
    EList<FDExtensionElement> _elements = elem.getElements();
    for (final FDExtensionElement child : _elements) {
      this.checkExtensionElementProperties(spec, child);
    }
  }
  
  @Check
  public void checkPropertiesComplete(final FDTypes elem) {
    final FDSpecification spec = FDModelUtils.getRootElement(elem).getSpec();
    this.checkSpecificationElementProperties(spec, elem, FDeployPackage.Literals.FD_TYPES__TARGET, spec.getName());
    final FDSpecificationExtender specHelper = new FDSpecificationExtender(spec);
    final PropertyDefChecker checker = new PropertyDefChecker(specHelper);
    final FDMapper mapper = new FDMapper(elem);
    final List<FType> targetTypes = elem.getTarget().getTypes();
    this.checkLocalTypes(targetTypes, specHelper, checker, mapper, spec, FDeployPackage.Literals.FD_TYPES__TARGET);
    this.deployValidator.checkUsedTypes(elem, targetTypes, checker);
  }
  
  @Check
  public void checkPropertiesComplete(final FDInterface elem) {
    int lowerLevelErrors = 0;
    final FDSpecification spec = FDModelUtils.getRootElement(elem).getSpec();
    boolean _checkSpecificationElementProperties = this.checkSpecificationElementProperties(spec, elem, FDeployPackage.Literals.FD_INTERFACE__TARGET, 
      spec.getName());
    if (_checkSpecificationElementProperties) {
      lowerLevelErrors++;
    }
    final FDSpecificationExtender specHelper = new FDSpecificationExtender(spec);
    final PropertyDefChecker checker = new PropertyDefChecker(specHelper);
    final FDMapper mapper = new FDMapper(elem);
    final FInterface target = elem.getTarget();
    EList<FAttribute> _attributes = target.getAttributes();
    for (final FAttribute tc : _attributes) {
      {
        FDElement _fDElement = mapper.getFDElement(tc);
        FDAttribute c = ((FDAttribute) _fDElement);
        if ((c == null)) {
          boolean _mustBeDefined = checker.mustBeDefined(tc);
          if (_mustBeDefined) {
            StringConcatenation _builder = new StringConcatenation();
            _builder.append(FDeployValidator.DEPLOYMENT_ELEMENT_QUICKFIX_MESSAGE);
            _builder.append("\'");
            String _name = tc.getName();
            _builder.append(_name);
            _builder.append("\'");
            this.error(_builder.toString(), 
              FDeployPackage.Literals.FD_INTERFACE__TARGET, FDeployValidator.DEPLOYMENT_ELEMENT_QUICKFIX, tc.getName(), 
              FrancaQuickFixConstants.ATTRIBUTE.toString());
            StringConcatenation _builder_1 = new StringConcatenation();
            _builder_1.append(FDeployValidator.DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX_MESSAGE);
            _builder_1.append("\'");
            String _name_1 = tc.getName();
            _builder_1.append(_name_1);
            _builder_1.append("\'");
            this.error(_builder_1.toString(), 
              FDeployPackage.Literals.FD_INTERFACE__TARGET, FDeployValidator.DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX, 
              tc.getName(), FrancaQuickFixConstants.ATTRIBUTE.toString());
            lowerLevelErrors++;
          }
        } else {
          boolean _checkSpecificationElementProperties_1 = this.checkSpecificationElementProperties(spec, c, FDeployPackage.Literals.FD_ATTRIBUTE__TARGET, 
            tc.getName());
          if (_checkSpecificationElementProperties_1) {
            lowerLevelErrors++;
          }
        }
      }
    }
    EList<FMethod> _methods = target.getMethods();
    for (final FMethod tc_1 : _methods) {
      {
        FDElement _fDElement = mapper.getFDElement(tc_1);
        FDMethod c = ((FDMethod) _fDElement);
        if ((c == null)) {
          boolean _mustBeDefined = checker.mustBeDefined(tc_1);
          if (_mustBeDefined) {
            String name = FrancaModelExtensions.getUniqueName(tc_1);
            StringConcatenation _builder = new StringConcatenation();
            _builder.append(FDeployValidator.DEPLOYMENT_ELEMENT_QUICKFIX_MESSAGE);
            _builder.append("\'");
            _builder.append(name);
            _builder.append("\'");
            this.error(_builder.toString(), 
              FDeployPackage.Literals.FD_INTERFACE__TARGET, FDeployValidator.DEPLOYMENT_ELEMENT_QUICKFIX, name, 
              FrancaQuickFixConstants.METHOD.toString());
            StringConcatenation _builder_1 = new StringConcatenation();
            _builder_1.append(FDeployValidator.DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX_MESSAGE);
            _builder_1.append("\'");
            _builder_1.append(name);
            _builder_1.append("\'");
            this.error(_builder_1.toString(), 
              FDeployPackage.Literals.FD_INTERFACE__TARGET, FDeployValidator.DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX, name, 
              FrancaQuickFixConstants.METHOD.toString());
            lowerLevelErrors++;
          }
        } else {
          final String name_1 = FrancaModelExtensions.getUniqueName(tc_1);
          boolean _checkSpecificationElementProperties_1 = this.checkSpecificationElementProperties(spec, c, FDeployPackage.Literals.FD_METHOD__TARGET, name_1);
          if (_checkSpecificationElementProperties_1) {
            lowerLevelErrors++;
          }
          final boolean checkIn = this.checkArgumentList(specHelper, checker, mapper, spec, tc_1.getInArgs(), c, "Input", FDeployPackage.Literals.FD_METHOD__TARGET);
          final boolean checkOut = this.checkArgumentList(specHelper, checker, mapper, spec, tc_1.getOutArgs(), c, "Output", FDeployPackage.Literals.FD_METHOD__TARGET);
          if ((checkIn || checkOut)) {
            StringConcatenation _builder_2 = new StringConcatenation();
            _builder_2.append(FDeployValidator.DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX_MESSAGE);
            _builder_2.append("\'");
            _builder_2.append(name_1);
            _builder_2.append("\'");
            this.error(_builder_2.toString(), c, 
              FDeployPackage.Literals.FD_METHOD__TARGET, FDeployValidator.DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX, name_1, 
              FrancaQuickFixConstants.METHOD.toString());
            lowerLevelErrors++;
          }
        }
      }
    }
    EList<FBroadcast> _broadcasts = target.getBroadcasts();
    for (final FBroadcast tc_2 : _broadcasts) {
      {
        FDElement _fDElement = mapper.getFDElement(tc_2);
        final FDBroadcast c = ((FDBroadcast) _fDElement);
        if ((c == null)) {
          boolean _mustBeDefined = checker.mustBeDefined(tc_2);
          if (_mustBeDefined) {
            final String name = FrancaModelExtensions.getUniqueName(tc_2);
            StringConcatenation _builder = new StringConcatenation();
            _builder.append(FDeployValidator.DEPLOYMENT_ELEMENT_QUICKFIX_MESSAGE);
            _builder.append("\'");
            _builder.append(name);
            _builder.append("\'");
            this.error(_builder.toString(), 
              FDeployPackage.Literals.FD_INTERFACE__TARGET, FDeployValidator.DEPLOYMENT_ELEMENT_QUICKFIX, name, 
              FrancaQuickFixConstants.BROADCAST.toString());
            StringConcatenation _builder_1 = new StringConcatenation();
            _builder_1.append(FDeployValidator.DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX_MESSAGE);
            _builder_1.append("\'");
            _builder_1.append(name);
            _builder_1.append("\'");
            this.error(_builder_1.toString(), 
              FDeployPackage.Literals.FD_INTERFACE__TARGET, FDeployValidator.DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX, name, 
              FrancaQuickFixConstants.BROADCAST.toString());
            lowerLevelErrors++;
          }
        } else {
          final String name_1 = FrancaModelExtensions.getUniqueName(tc_2);
          boolean _checkSpecificationElementProperties_1 = this.checkSpecificationElementProperties(spec, c, FDeployPackage.Literals.FD_BROADCAST__TARGET, name_1);
          if (_checkSpecificationElementProperties_1) {
            lowerLevelErrors++;
          }
          boolean _checkArgumentList = this.checkArgumentList(specHelper, checker, mapper, spec, tc_2.getOutArgs(), c, "Output", 
            FDeployPackage.Literals.FD_BROADCAST__TARGET);
          if (_checkArgumentList) {
            StringConcatenation _builder_2 = new StringConcatenation();
            _builder_2.append(FDeployValidator.DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX_MESSAGE);
            _builder_2.append("\'");
            _builder_2.append(name_1);
            _builder_2.append("\'");
            this.error(_builder_2.toString(), c, 
              FDeployPackage.Literals.FD_BROADCAST__TARGET, FDeployValidator.DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX, name_1, 
              FrancaQuickFixConstants.BROADCAST.toString());
            lowerLevelErrors++;
          }
        }
      }
    }
    boolean _checkLocalTypes = this.checkLocalTypes(target.getTypes(), specHelper, checker, mapper, spec, 
      FDeployPackage.Literals.FD_INTERFACE__TARGET);
    if (_checkLocalTypes) {
      lowerLevelErrors++;
    }
    boolean _checkUsedTypes = this.deployValidator.checkUsedTypes(elem, target.getTypes(), checker);
    if (_checkUsedTypes) {
      lowerLevelErrors++;
    }
    if ((lowerLevelErrors > 0)) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append(FDeployValidator.DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX_MESSAGE);
      _builder.append("\'");
      String _name = spec.getName();
      _builder.append(_name);
      _builder.append("\'");
      this.error(_builder.toString(), elem, 
        FDeployPackage.Literals.FD_INTERFACE__TARGET, FDeployValidator.DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX, spec.getName(), 
        FrancaQuickFixConstants.INTERFACE.toString());
    }
  }
  
  /**
   * Checks the local types of an {@link FDSpecification} instance.
   * @return true if an error is present, false otherwise
   */
  private boolean checkLocalTypes(final List<FType> types, final FDSpecificationExtender specHelper, final PropertyDefChecker checker, final FDMapper mapper, final FDSpecification spec, final EStructuralFeature parentFeature) {
    boolean hasError = false;
    for (final FType tc : types) {
      if ((tc instanceof FArrayType)) {
        FDElement _fDElement = mapper.getFDElement(tc);
        final FDArray c = ((FDArray) _fDElement);
        if ((c == null)) {
          boolean _mustBeDefined = checker.mustBeDefined(((FArrayType)tc));
          if (_mustBeDefined) {
            StringConcatenation _builder = new StringConcatenation();
            _builder.append(FDeployValidator.DEPLOYMENT_ELEMENT_QUICKFIX_MESSAGE);
            _builder.append("\'");
            String _name = ((FArrayType)tc).getName();
            _builder.append(_name);
            _builder.append("\'");
            this.error(_builder.toString(), parentFeature, 
              FDeployValidator.DEPLOYMENT_ELEMENT_QUICKFIX, ((FArrayType)tc).getName(), FrancaQuickFixConstants.ARRAY.toString());
            StringConcatenation _builder_1 = new StringConcatenation();
            _builder_1.append(FDeployValidator.DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX_MESSAGE);
            _builder_1.append("\'");
            String _name_1 = ((FArrayType)tc).getName();
            _builder_1.append(_name_1);
            _builder_1.append("\'");
            this.error(_builder_1.toString(), parentFeature, FDeployValidator.DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX, ((FArrayType)tc).getName(), 
              FrancaQuickFixConstants.ARRAY.toString());
            hasError = true;
          }
        } else {
          boolean _checkSpecificationElementProperties = this.checkSpecificationElementProperties(spec, c, FDeployPackage.Literals.FD_ARRAY__TARGET, ((FArrayType)tc).getName());
          if (_checkSpecificationElementProperties) {
            hasError = true;
          }
        }
      } else {
        if ((tc instanceof FStructType)) {
          FDElement _fDElement_1 = mapper.getFDElement(tc);
          final FDStruct c_1 = ((FDStruct) _fDElement_1);
          if ((c_1 == null)) {
            boolean _mustBeDefined_1 = checker.mustBeDefined(((FStructType)tc));
            if (_mustBeDefined_1) {
              StringConcatenation _builder_2 = new StringConcatenation();
              _builder_2.append(FDeployValidator.DEPLOYMENT_ELEMENT_QUICKFIX_MESSAGE);
              _builder_2.append("\'");
              String _name_2 = ((FStructType)tc).getName();
              _builder_2.append(_name_2);
              _builder_2.append("\'");
              this.error(_builder_2.toString(), parentFeature, 
                FDeployValidator.DEPLOYMENT_ELEMENT_QUICKFIX, ((FStructType)tc).getName(), FrancaQuickFixConstants.STRUCT.toString());
              StringConcatenation _builder_3 = new StringConcatenation();
              _builder_3.append(FDeployValidator.DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX_MESSAGE);
              _builder_3.append("\'");
              String _name_3 = ((FStructType)tc).getName();
              _builder_3.append(_name_3);
              _builder_3.append("\'");
              this.error(_builder_3.toString(), parentFeature, FDeployValidator.DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX, ((FStructType)tc).getName(), 
                FrancaQuickFixConstants.STRUCT.toString());
              hasError = true;
            }
          } else {
            boolean _checkSpecificationElementProperties_1 = this.checkSpecificationElementProperties(spec, c_1, FDeployPackage.Literals.FD_STRUCT__TARGET, ((FStructType)tc).getName());
            if (_checkSpecificationElementProperties_1) {
              hasError = true;
            }
            boolean _checkFieldsList = this.checkFieldsList(specHelper, checker, mapper, spec, ((FStructType)tc).getElements(), c_1, 
              FDeployPackage.Literals.FD_STRUCT__TARGET, "Struct");
            if (_checkFieldsList) {
              StringConcatenation _builder_4 = new StringConcatenation();
              _builder_4.append(FDeployValidator.DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX_MESSAGE);
              _builder_4.append("\'");
              String _name_4 = ((FStructType)tc).getName();
              _builder_4.append(_name_4);
              _builder_4.append("\'");
              this.error(_builder_4.toString(), c_1, 
                FDeployPackage.Literals.FD_STRUCT__TARGET, FDeployValidator.DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX, 
                ((FStructType)tc).getName(), FrancaQuickFixConstants.STRUCT.toString());
              hasError = true;
            }
          }
        } else {
          if ((tc instanceof FUnionType)) {
            FDElement _fDElement_2 = mapper.getFDElement(tc);
            final FDUnion c_2 = ((FDUnion) _fDElement_2);
            if ((c_2 == null)) {
              boolean _mustBeDefined_2 = checker.mustBeDefined(((FUnionType)tc));
              if (_mustBeDefined_2) {
                StringConcatenation _builder_5 = new StringConcatenation();
                _builder_5.append(FDeployValidator.DEPLOYMENT_ELEMENT_QUICKFIX_MESSAGE);
                _builder_5.append("\'");
                String _name_5 = ((FUnionType)tc).getName();
                _builder_5.append(_name_5);
                _builder_5.append("\'");
                this.error(_builder_5.toString(), parentFeature, 
                  FDeployValidator.DEPLOYMENT_ELEMENT_QUICKFIX, ((FUnionType)tc).getName(), FrancaQuickFixConstants.UNION.toString());
                StringConcatenation _builder_6 = new StringConcatenation();
                _builder_6.append(FDeployValidator.DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX_MESSAGE);
                _builder_6.append("\'");
                String _name_6 = ((FUnionType)tc).getName();
                _builder_6.append(_name_6);
                _builder_6.append("\'");
                this.error(_builder_6.toString(), parentFeature, FDeployValidator.DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX, ((FUnionType)tc).getName(), 
                  FrancaQuickFixConstants.UNION.toString());
                hasError = true;
              }
            } else {
              boolean _checkSpecificationElementProperties_2 = this.checkSpecificationElementProperties(spec, c_2, FDeployPackage.Literals.FD_UNION__TARGET, ((FUnionType)tc).getName());
              if (_checkSpecificationElementProperties_2) {
                hasError = true;
              }
              boolean _checkFieldsList_1 = this.checkFieldsList(specHelper, checker, mapper, spec, ((FUnionType)tc).getElements(), c_2, 
                FDeployPackage.Literals.FD_UNION__TARGET, "Union");
              if (_checkFieldsList_1) {
                StringConcatenation _builder_7 = new StringConcatenation();
                _builder_7.append(FDeployValidator.DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX_MESSAGE);
                _builder_7.append("\'");
                String _name_7 = ((FUnionType)tc).getName();
                _builder_7.append(_name_7);
                _builder_7.append("\'");
                this.error(_builder_7.toString(), c_2, 
                  FDeployPackage.Literals.FD_UNION__TARGET, FDeployValidator.DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX, 
                  ((FUnionType)tc).getName(), FrancaQuickFixConstants.UNION.toString());
                hasError = true;
              }
            }
          } else {
            if ((tc instanceof FEnumerationType)) {
              FDElement _fDElement_3 = mapper.getFDElement(tc);
              final FDEnumeration c_3 = ((FDEnumeration) _fDElement_3);
              if ((c_3 == null)) {
                boolean _mustBeDefined_3 = checker.mustBeDefined(((FEnumerationType)tc));
                if (_mustBeDefined_3) {
                  StringConcatenation _builder_8 = new StringConcatenation();
                  _builder_8.append(FDeployValidator.DEPLOYMENT_ELEMENT_QUICKFIX_MESSAGE);
                  _builder_8.append("\'");
                  String _name_8 = ((FEnumerationType)tc).getName();
                  _builder_8.append(_name_8);
                  _builder_8.append("\'");
                  this.error(_builder_8.toString(), parentFeature, 
                    FDeployValidator.DEPLOYMENT_ELEMENT_QUICKFIX, ((FEnumerationType)tc).getName(), FrancaQuickFixConstants.ENUMERATION.toString());
                  StringConcatenation _builder_9 = new StringConcatenation();
                  _builder_9.append(FDeployValidator.DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX_MESSAGE);
                  _builder_9.append("\'");
                  String _name_9 = ((FEnumerationType)tc).getName();
                  _builder_9.append(_name_9);
                  _builder_9.append("\'");
                  this.error(_builder_9.toString(), parentFeature, FDeployValidator.DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX, ((FEnumerationType)tc).getName(), 
                    FrancaQuickFixConstants.ENUMERATION.toString());
                  hasError = true;
                }
              } else {
                boolean _checkSpecificationElementProperties_3 = this.checkSpecificationElementProperties(spec, c_3, FDeployPackage.Literals.FD_ENUMERATION__TARGET, ((FEnumerationType)tc).getName());
                if (_checkSpecificationElementProperties_3) {
                  hasError = true;
                }
                boolean _checkEnumeratorsList = this.checkEnumeratorsList(specHelper, mapper, spec, ((FEnumerationType)tc).getEnumerators(), c_3, 
                  FDeployPackage.Literals.FD_ENUMERATION__TARGET);
                if (_checkEnumeratorsList) {
                  StringConcatenation _builder_10 = new StringConcatenation();
                  _builder_10.append(FDeployValidator.DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX_MESSAGE);
                  _builder_10.append("\'");
                  String _name_10 = ((FEnumerationType)tc).getName();
                  _builder_10.append(_name_10);
                  _builder_10.append("\'");
                  this.error(_builder_10.toString(), c_3, 
                    FDeployPackage.Literals.FD_ENUMERATION__TARGET, FDeployValidator.DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX, 
                    ((FEnumerationType)tc).getName(), FrancaQuickFixConstants.ENUMERATION.toString());
                  hasError = true;
                }
              }
            } else {
              if ((tc instanceof FMapType)) {
                FDElement _fDElement_4 = mapper.getFDElement(tc);
                final FDMap c_4 = ((FDMap) _fDElement_4);
                if ((c_4 == null)) {
                  boolean _mustBeDefined_4 = checker.mustBeDefined(((FMapType)tc));
                  if (_mustBeDefined_4) {
                    StringConcatenation _builder_11 = new StringConcatenation();
                    _builder_11.append(FDeployValidator.DEPLOYMENT_ELEMENT_QUICKFIX_MESSAGE);
                    _builder_11.append("\'");
                    String _name_11 = ((FMapType)tc).getName();
                    _builder_11.append(_name_11);
                    _builder_11.append("\'");
                    this.error(_builder_11.toString(), parentFeature, 
                      FDeployValidator.DEPLOYMENT_ELEMENT_QUICKFIX, ((FMapType)tc).getName(), FrancaQuickFixConstants.MAP.toString());
                    StringConcatenation _builder_12 = new StringConcatenation();
                    _builder_12.append(FDeployValidator.DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX_MESSAGE);
                    _builder_12.append("\'");
                    String _name_12 = ((FMapType)tc).getName();
                    _builder_12.append(_name_12);
                    _builder_12.append("\'");
                    this.error(_builder_12.toString(), parentFeature, FDeployValidator.DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX, ((FMapType)tc).getName(), 
                      FrancaQuickFixConstants.MAP.toString());
                    hasError = true;
                  }
                } else {
                  boolean _checkSpecificationElementProperties_4 = this.checkSpecificationElementProperties(spec, c_4, FDeployPackage.Literals.FD_MAP__TARGET, ((FMapType)tc).getName());
                  if (_checkSpecificationElementProperties_4) {
                    hasError = true;
                  }
                  boolean contentError = false;
                  boolean _mustBeDefined_5 = checker.mustBeDefined(((FMapType)tc).getKeyType());
                  if (_mustBeDefined_5) {
                    FDMapKey _key = c_4.getKey();
                    boolean _tripleEquals = (_key == null);
                    if (_tripleEquals) {
                      StringConcatenation _builder_13 = new StringConcatenation();
                      _builder_13.append(FDeployValidator.MAP_KEY_QUICKFIX_MESSAGE);
                      _builder_13.append("\'");
                      String _name_13 = ((FMapType)tc).getName();
                      _builder_13.append(_name_13);
                      _builder_13.append("\'");
                      this.error(_builder_13.toString(), c_4, 
                        FDeployPackage.Literals.FD_MAP__TARGET, FDeployValidator.MAP_KEY_QUICKFIX, ((FMapType)tc).getName(), 
                        FrancaQuickFixConstants.MAP_KEY.toString());
                      contentError = true;
                    } else {
                      final List<String> missing = this.collectMissingProperties(spec, c_4.getKey());
                      boolean _isEmpty = missing.isEmpty();
                      boolean _not = (!_isEmpty);
                      if (_not) {
                        StringConcatenation _builder_14 = new StringConcatenation();
                        _builder_14.append(FDeployValidator.MANDATORY_PROPERTY_QUICKFIX_MESSAGE);
                        _builder_14.append("\'");
                        String _name_14 = ((FMapType)tc).getName();
                        _builder_14.append(_name_14);
                        _builder_14.append(" key type\'");
                        this.error(_builder_14.toString(), c_4, FDeployPackage.Literals.FD_MAP__KEY, (-1), 
                          FDeployValidator.MANDATORY_PROPERTY_QUICKFIX, ((FMapType)tc).getName(), 
                          FrancaQuickFixConstants.MAP_KEY.toString());
                        hasError = true;
                      }
                    }
                  }
                  boolean _mustBeDefined_6 = checker.mustBeDefined(((FMapType)tc).getValueType());
                  if (_mustBeDefined_6) {
                    FDMapValue _value = c_4.getValue();
                    boolean _tripleEquals_1 = (_value == null);
                    if (_tripleEquals_1) {
                      StringConcatenation _builder_15 = new StringConcatenation();
                      _builder_15.append(FDeployValidator.MAP_VALUE_QUICKFIX_MESSAGE);
                      _builder_15.append("\'");
                      String _name_15 = ((FMapType)tc).getName();
                      _builder_15.append(_name_15);
                      _builder_15.append("\'");
                      this.error(_builder_15.toString(), c_4, 
                        FDeployPackage.Literals.FD_MAP__TARGET, FDeployValidator.MAP_VALUE_QUICKFIX, ((FMapType)tc).getName(), 
                        FrancaQuickFixConstants.MAP_VALUE.toString());
                      contentError = true;
                    } else {
                      final List<String> missing_1 = this.collectMissingProperties(spec, c_4.getValue());
                      boolean _isEmpty_1 = missing_1.isEmpty();
                      boolean _not_1 = (!_isEmpty_1);
                      if (_not_1) {
                        StringConcatenation _builder_16 = new StringConcatenation();
                        _builder_16.append(FDeployValidator.MANDATORY_PROPERTY_QUICKFIX_MESSAGE);
                        _builder_16.append("\'");
                        String _name_16 = ((FMapType)tc).getName();
                        _builder_16.append(_name_16);
                        _builder_16.append(" value type\'");
                        this.error(_builder_16.toString(), c_4, FDeployPackage.Literals.FD_MAP__VALUE, (-1), 
                          FDeployValidator.MANDATORY_PROPERTY_QUICKFIX, ((FMapType)tc).getName(), 
                          FrancaQuickFixConstants.MAP_VALUE.toString());
                        hasError = true;
                      }
                    }
                  }
                  if (contentError) {
                    StringConcatenation _builder_17 = new StringConcatenation();
                    _builder_17.append(FDeployValidator.DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX_MESSAGE);
                    _builder_17.append("\'");
                    String _name_17 = ((FMapType)tc).getName();
                    _builder_17.append(_name_17);
                    _builder_17.append("\'");
                    this.error(_builder_17.toString(), c_4, 
                      FDeployPackage.Literals.FD_MAP__TARGET, FDeployValidator.DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX, ((FMapType)tc).getName(), 
                      FrancaQuickFixConstants.MAP.toString());
                  }
                }
              } else {
                if ((tc instanceof FTypeDef)) {
                  FDElement _fDElement_5 = mapper.getFDElement(tc);
                  final FDTypedef c_5 = ((FDTypedef) _fDElement_5);
                  if ((c_5 == null)) {
                    boolean _mustBeDefined_7 = checker.mustBeDefined(((FTypeDef)tc));
                    if (_mustBeDefined_7) {
                      StringConcatenation _builder_18 = new StringConcatenation();
                      _builder_18.append(FDeployValidator.DEPLOYMENT_ELEMENT_QUICKFIX_MESSAGE);
                      _builder_18.append("\'");
                      String _name_18 = ((FTypeDef)tc).getName();
                      _builder_18.append(_name_18);
                      _builder_18.append("\'");
                      this.error(_builder_18.toString(), parentFeature, 
                        FDeployValidator.DEPLOYMENT_ELEMENT_QUICKFIX, ((FTypeDef)tc).getName(), FrancaQuickFixConstants.TYPEDEF.toString());
                      StringConcatenation _builder_19 = new StringConcatenation();
                      _builder_19.append(FDeployValidator.DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX_MESSAGE);
                      _builder_19.append("\'");
                      String _name_19 = ((FTypeDef)tc).getName();
                      _builder_19.append(_name_19);
                      _builder_19.append("\'");
                      this.error(_builder_19.toString(), parentFeature, FDeployValidator.DEPLOYMENT_ELEMENT_RECURSIVE_QUICKFIX, ((FTypeDef)tc).getName(), 
                        FrancaQuickFixConstants.TYPEDEF.toString());
                      hasError = true;
                    }
                  } else {
                    boolean _checkSpecificationElementProperties_5 = this.checkSpecificationElementProperties(spec, c_5, FDeployPackage.Literals.FD_TYPEDEF__TARGET, ((FTypeDef)tc).getName());
                    if (_checkSpecificationElementProperties_5) {
                      hasError = true;
                    }
                  }
                }
              }
            }
          }
        }
      }
    }
    return hasError;
  }
  
  /**
   * Checks the argument list of {@link FDMethod}s and {@link FDBroadcast}s.
   * @return true if an error is present, false otherwise
   */
  private boolean checkArgumentList(final FDSpecificationExtender specHelper, final PropertyDefChecker checker, final FDMapper mapper, final FDSpecification spec, final List<FArgument> args, final FDElement parent, final String tag, final EStructuralFeature feature) {
    boolean hasError = false;
    for (final FArgument tc : args) {
      {
        FDElement _fDElement = mapper.getFDElement(tc);
        FDArgument c = ((FDArgument) _fDElement);
        if ((c == null)) {
          boolean _mustBeDefined = checker.mustBeDefined(tc);
          if (_mustBeDefined) {
            String opName = "";
            String opType = "";
            String quickfix = "";
            boolean _matched = false;
            if (parent instanceof FDMethod) {
              _matched=true;
              opName = FrancaModelExtensions.getUniqueName(((FDMethod)parent).getTarget());
              opType = "method";
              quickfix = FDeployValidator.METHOD_ARGUMENT_QUICKFIX;
            }
            if (!_matched) {
              if (parent instanceof FDBroadcast) {
                _matched=true;
                opName = FrancaModelExtensions.getUniqueName(((FDBroadcast)parent).getTarget());
                opType = "broadcast";
                quickfix = FDeployValidator.BROADCAST_ARGUMENT_QUICKFIX;
              }
            }
            StringConcatenation _builder = new StringConcatenation();
            _builder.append("Mandatory argument \'");
            String _name = tc.getName();
            _builder.append(_name);
            _builder.append("\' is missing for ");
            _builder.append(opType);
            _builder.append(" \'");
            _builder.append(opName);
            _builder.append("\'");
            this.error(_builder.toString(), parent, feature, (-1), quickfix, opName, tc.getName());
            hasError = true;
          }
        } else {
          boolean _checkSpecificationElementProperties = this.checkSpecificationElementProperties(spec, c, FDeployPackage.Literals.FD_ARGUMENT__TARGET, tc.getName());
          if (_checkSpecificationElementProperties) {
            hasError = true;
          }
        }
      }
    }
    return hasError;
  }
  
  /**
   * Checks the field list of {@link FDUnion}s and {@link FDStruct}s.
   * @return true if an error is present, false otherwise
   */
  private boolean checkFieldsList(final FDSpecificationExtender specHelper, final PropertyDefChecker checker, final FDMapper mapper, final FDSpecification spec, final List<FField> fields, final FDElement parent, final EStructuralFeature feature, final String tag) {
    boolean hasError = false;
    for (final FField tc : fields) {
      {
        FDElement _fDElement = mapper.getFDElement(tc);
        FDField c = ((FDField) _fDElement);
        if ((c == null)) {
          boolean _mustBeDefined = checker.mustBeDefined(tc);
          if (_mustBeDefined) {
            String name = "";
            if ((parent instanceof FDUnion)) {
              name = ((FDUnion)parent).getTarget().getName();
            } else {
              if ((parent instanceof FDStruct)) {
                name = ((FDStruct)parent).getTarget().getName();
              }
            }
            StringConcatenation _builder = new StringConcatenation();
            _builder.append("Mandatory field \'");
            String _name = tc.getName();
            _builder.append(_name);
            _builder.append("\' is missing for compound \'");
            _builder.append(name);
            _builder.append("\'");
            this.error(_builder.toString(), parent, feature, (-1), FDeployValidator.COMPOUND_FIELD_QUICKFIX, name, tc.getName());
            hasError = true;
          }
        } else {
          boolean _checkSpecificationElementProperties = this.checkSpecificationElementProperties(spec, c, FDeployPackage.Literals.FD_FIELD__TARGET, tc.getName());
          if (_checkSpecificationElementProperties) {
            hasError = true;
          }
        }
      }
    }
    return hasError;
  }
  
  /**
   * Checks the enumerator list of {@link FDEnumerator}s.
   * @return true if an error is present, false otherwise
   */
  private boolean checkEnumeratorsList(final FDSpecificationExtender specHelper, final FDMapper mapper, final FDSpecification spec, final List<FEnumerator> enumerators, final FDElement parent, final EStructuralFeature feature) {
    boolean hasError = false;
    for (final FEnumerator tc : enumerators) {
      {
        FDElement _fDElement = mapper.getFDElement(tc);
        FDEnumValue c = ((FDEnumValue) _fDElement);
        if ((c == null)) {
          boolean _isMandatory = specHelper.isMandatory(FDPropertyHost.builtIn(FDBuiltInPropertyHost.ENUMERATORS));
          if (_isMandatory) {
            StringConcatenation _builder = new StringConcatenation();
            _builder.append("Mandatory enumerator \'");
            String _name = tc.getName();
            _builder.append(_name);
            _builder.append("\' is missing for enumeration \'");
            String _name_1 = ((FDEnumeration) parent).getTarget().getName();
            _builder.append(_name_1);
            _builder.append("\'");
            this.error(_builder.toString(), parent, feature, (-1), FDeployValidator.ENUMERATOR_ENUM_QUICKFIX, 
              ((FDEnumeration) parent).getTarget().getName(), tc.getName());
            hasError = true;
          }
        } else {
          boolean _checkSpecificationElementProperties = this.checkSpecificationElementProperties(spec, c, FDeployPackage.Literals.FD_ENUM_VALUE__TARGET, tc.getName());
          if (_checkSpecificationElementProperties) {
            hasError = true;
          }
        }
      }
    }
    return hasError;
  }
  
  /**
   * Checks whether all of the mandatory properties of the given {@link FDSpecification} instance are present.</p>
   * 
   * @param spec the deployment specification
   * @param elem the given element
   * @param feature the corresponding feature instance
   * @param elementName the name of the element for the quickfix message
   * @return true if there was an error (missing property), false otherwise
   */
  protected boolean checkSpecificationElementProperties(final FDSpecification spec, final FDElement elem, final EStructuralFeature feature, final String elementName) {
    final List<String> missing = this.collectMissingProperties(spec, elem);
    boolean _isEmpty = missing.isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append(FDeployValidator.MANDATORY_PROPERTY_QUICKFIX_MESSAGE);
      _builder.append("\'");
      _builder.append(elementName);
      _builder.append("\'");
      this.error(_builder.toString(), elem, feature, (-1), 
        FDeployValidator.MANDATORY_PROPERTY_QUICKFIX, elementName);
      return true;
    }
    return false;
  }
  
  private List<String> collectMissingProperties(final FDSpecification spec, final FDElement elem) {
    List<FDPropertyDecl> decls = PropertyMappings.getAllPropertyDecls(spec, elem);
    List<String> missing = Lists.<String>newArrayList();
    for (final FDPropertyDecl decl : decls) {
      boolean _isMandatory = PropertyMappings.isMandatory(decl);
      if (_isMandatory) {
        boolean _contains = this.contains(elem.getProperties().getItems(), decl);
        boolean _not = (!_contains);
        if (_not) {
          missing.add(decl.getName());
        }
      }
    }
    return missing;
  }
  
  private boolean contains(final List<FDProperty> properties, final FDPropertyDecl decl) {
    for (final FDProperty p : properties) {
      FDPropertyDecl _decl = p.getDecl();
      boolean _tripleEquals = (_decl == decl);
      if (_tripleEquals) {
        return true;
      }
    }
    return false;
  }
  
  @Check
  public void checkOverwriteSections(final FDTypeOverwrites elem) {
    EObject parent = elem.eContainer();
    if ((parent instanceof FDOverwriteElement)) {
      FType targetType = FDModelUtils.getOverwriteTargetType(((FDOverwriteElement)parent));
      if ((targetType == null)) {
        this.error("Cannot determine target type of overwrite section", parent, FDeployPackage.Literals.FD_OVERWRITE_ELEMENT__OVERWRITES);
      } else {
        if ((elem instanceof FDPlainTypeOverwrites)) {
        } else {
          if ((targetType instanceof FStructType)) {
            if ((!(elem instanceof FDStructOverwrites))) {
              this.error("Invalid overwrite tag, use \'#struct\'", parent, FDeployPackage.Literals.FD_OVERWRITE_ELEMENT__OVERWRITES);
            }
          } else {
            if ((targetType instanceof FUnionType)) {
              if ((!(elem instanceof FDUnionOverwrites))) {
                this.error("Invalid overwrite tag, use \'#union\'", parent, FDeployPackage.Literals.FD_OVERWRITE_ELEMENT__OVERWRITES);
              }
            } else {
              if ((targetType instanceof FEnumerationType)) {
                if ((!(elem instanceof FDEnumerationOverwrites))) {
                  this.error("Invalid overwrite tag, use \'#enumeration\'", parent, FDeployPackage.Literals.FD_OVERWRITE_ELEMENT__OVERWRITES);
                }
              }
            }
          }
        }
      }
    }
  }
  
  @Check
  public void checkExtensionType(final FDExtensionType type) {
    final String name = type.getName();
    final IFDeployExtension.TypeDef typeDef = ExtensionRegistry.findType(name);
    if ((typeDef == null)) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Invalid type \'");
      _builder.append(name);
      _builder.append("\', no matching deployment extension has been configured");
      this.error(_builder.toString(), type, FDeployPackage.Literals.FD_EXTENSION_TYPE__NAME, (-1));
    }
  }
  
  @Check
  public void checkPropertyFlagType(final FDPropertyFlag flag) {
    FDComplexValue _default = flag.getDefault();
    boolean _tripleEquals = (_default == null);
    if (_tripleEquals) {
      return;
    }
    EObject _eContainer = flag.eContainer();
    final FDPropertyDecl decl = ((FDPropertyDecl) _eContainer);
    final FDTypeRef typeRef = decl.getType();
    final FDComplexValue value = flag.getDefault();
    FDValue _single = value.getSingle();
    boolean _tripleNotEquals = (_single != null);
    if (_tripleNotEquals) {
      String _array = typeRef.getArray();
      boolean _tripleNotEquals_1 = (_array != null);
      if (_tripleNotEquals_1) {
        this.error("Default must be an array!", FDeployPackage.Literals.FD_PROPERTY_FLAG__DEFAULT);
      } else {
        this.checkValueType(typeRef, value.getSingle(), flag, FDeployPackage.Literals.FD_PROPERTY_FLAG__DEFAULT, (-1));
      }
    } else {
      FDValueArray _array_1 = value.getArray();
      boolean _tripleNotEquals_2 = (_array_1 != null);
      if (_tripleNotEquals_2) {
        String _array_2 = typeRef.getArray();
        boolean _tripleEquals_1 = (_array_2 == null);
        if (_tripleEquals_1) {
          this.error("Default must be a single type, not an array!", FDeployPackage.Literals.FD_PROPERTY_FLAG__DEFAULT);
        } else {
          this.checkValueArrayType(typeRef, value.getArray());
        }
      }
    }
  }
  
  @Check
  public void checkPropertyValueType(final FDProperty prop) {
    final FDTypeRef typeRef = prop.getDecl().getType();
    final FDComplexValue value = prop.getValue();
    FDValue _single = value.getSingle();
    boolean _tripleNotEquals = (_single != null);
    if (_tripleNotEquals) {
      String _array = typeRef.getArray();
      boolean _tripleNotEquals_1 = (_array != null);
      if (_tripleNotEquals_1) {
        this.error("Invalid type, expected array!", FDeployPackage.Literals.FD_PROPERTY__VALUE);
      } else {
        this.checkValueType(typeRef, value.getSingle(), prop, FDeployPackage.Literals.FD_PROPERTY__VALUE, (-1));
      }
    } else {
      FDValueArray _array_1 = value.getArray();
      boolean _tripleNotEquals_2 = (_array_1 != null);
      if (_tripleNotEquals_2) {
        String _array_2 = typeRef.getArray();
        boolean _tripleEquals = (_array_2 == null);
        if (_tripleEquals) {
          this.error("Invalid array type, expected single value!", FDeployPackage.Literals.FD_PROPERTY__VALUE);
        } else {
          this.checkValueArrayType(typeRef, value.getArray());
        }
      }
    }
  }
  
  private void checkValueType(final FDTypeRef typeRef, final FDValue value, final EObject src, final EReference literal, final int index) {
    FDType _complex = typeRef.getComplex();
    boolean _tripleEquals = (_complex == null);
    if (_tripleEquals) {
      final int predefined = typeRef.getPredefined().getValue();
      switch (predefined) {
        case FDPredefinedTypeId.INTEGER_VALUE:
          if ((!(value instanceof FDInteger))) {
            this.error("Invalid type, expected Integer constant", src, literal, index);
          }
          break;
        case FDPredefinedTypeId.STRING_VALUE:
          if ((!(value instanceof FDString))) {
            this.error("Invalid type, expected String constant", src, literal, index);
          }
          break;
        case FDPredefinedTypeId.BOOLEAN_VALUE:
          if ((!(value instanceof FDBoolean))) {
            this.error("Invalid type, expected \'true\' or \'false\'", src, literal, index);
          }
          break;
        case FDPredefinedTypeId.INTERFACE_VALUE:
          if ((!(value instanceof FDInterfaceRef))) {
            this.error("Invalid type, expected reference to Franca interface", src, literal, index);
          }
          break;
      }
    } else {
      final FDType type = typeRef.getComplex();
      if ((type instanceof FDEnumType)) {
        boolean _isEnumerator = FDModelUtils.isEnumerator(value);
        boolean _not = (!_isEnumerator);
        if (_not) {
          this.error("Invalid type, expected enumerator", src, literal, index);
        }
      }
    }
  }
  
  private void checkValueArrayType(final FDTypeRef typeRef, final FDValueArray array) {
    int i = 0;
    EList<FDValue> _values = array.getValues();
    for (final FDValue value : _values) {
      {
        this.checkValueType(typeRef, value, array, FDeployPackage.Literals.FD_VALUE_ARRAY__VALUES, i);
        i++;
      }
    }
  }
  
  @Override
  public void reportError(final String message, final EObject object, final EStructuralFeature feature) {
    this.error(message, object, feature, ValidationMessageAcceptor.INSIGNIFICANT_INDEX);
  }
  
  @Override
  public void reportError(final String message, final EObject object, final EStructuralFeature feature, final int idx) {
    this.error(message, object, feature, idx);
  }
  
  @Override
  public void reportWarning(final String message, final EObject object, final EStructuralFeature feature) {
    this.warning(message, object, feature, ValidationMessageAcceptor.INSIGNIFICANT_INDEX);
  }
}
