/**
 * Copyright (c) 2014 itemis AG (http://www.itemis.de).
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.franca.deploymodel.dsl.validation;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.common.collect.Sets;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.franca.core.FrancaModelExtensions;
import org.franca.core.franca.FArrayType;
import org.franca.core.franca.FEnumerationType;
import org.franca.core.franca.FModel;
import org.franca.core.franca.FStructType;
import org.franca.core.franca.FType;
import org.franca.core.franca.FTypeRef;
import org.franca.core.franca.FUnionType;
import org.franca.core.utils.CycleChecker;
import org.franca.deploymodel.core.FDPropertyHost;
import org.franca.deploymodel.dsl.fDeploy.FDDeclaration;
import org.franca.deploymodel.dsl.fDeploy.FDEnumType;
import org.franca.deploymodel.dsl.fDeploy.FDInterface;
import org.franca.deploymodel.dsl.fDeploy.FDPropertyDecl;
import org.franca.deploymodel.dsl.fDeploy.FDRootElement;
import org.franca.deploymodel.dsl.fDeploy.FDSpecification;
import org.franca.deploymodel.dsl.fDeploy.FDType;
import org.franca.deploymodel.dsl.fDeploy.FDTypes;
import org.franca.deploymodel.dsl.fDeploy.FDeployPackage;
import org.franca.deploymodel.dsl.generator.internal.HostLogic;
import org.franca.deploymodel.dsl.validation.PropertyDefChecker;
import org.franca.deploymodel.dsl.validation.ValidationMessageReporter;

@SuppressWarnings("all")
public class FDeployValidatorAux {
  private ValidationMessageReporter reporter;
  
  public FDeployValidatorAux(final ValidationMessageReporter reporter) {
    this.reporter = reporter;
  }
  
  public void checkRootElement(final FDRootElement it) {
    final Function1<FDRootElement, Collection<FDRootElement>> _function = new Function1<FDRootElement, Collection<FDRootElement>>() {
      @Override
      public Collection<FDRootElement> apply(final FDRootElement e) {
        return e.getUse();
      }
    };
    final List<FDRootElement> path = CycleChecker.<FDRootElement>isReferenced(it, _function);
    if ((path != null)) {
      final int idx = it.getUse().indexOf(path.get(0));
      String _name = it.getName();
      String _plus = ("Cyclic use-relation in element \'" + _name);
      String _plus_1 = (_plus + "\'");
      this.reporter.reportError(_plus_1, it, FDeployPackage.Literals.FD_ROOT_ELEMENT__USE, idx);
    }
    EList<FDRootElement> _use = it.getUse();
    for (final FDRootElement other : _use) {
      if (((it.getSpec() != null) && (other.getSpec() != null))) {
        boolean _isCompatible = this.isCompatible(it.getSpec(), other.getSpec());
        boolean _not = (!_isCompatible);
        if (_not) {
          String _name_1 = other.getName();
          String _plus_2 = ("Use-relation \'" + _name_1);
          String _plus_3 = (_plus_2 + "\' ");
          String _plus_4 = (_plus_3 + 
            "refers to deployment with incompatible specification ");
          String _plus_5 = (_plus_4 + 
            "\'");
          String _name_2 = other.getSpec().getName();
          String _plus_6 = (_plus_5 + _name_2);
          String _plus_7 = (_plus_6 + "\'");
          this.reporter.reportError(_plus_7, it, FDeployPackage.Literals.FD_ROOT_ELEMENT__USE, it.getUse().indexOf(other));
        }
      }
    }
  }
  
  private boolean isCompatible(final FDSpecification spec1, final FDSpecification spec2) {
    FDSpecification _cyclicBaseSpec = this.getCyclicBaseSpec(spec2);
    boolean _tripleNotEquals = (_cyclicBaseSpec != null);
    if (_tripleNotEquals) {
      return true;
    }
    FDSpecification check = spec2;
    do {
      {
        boolean _equals = Objects.equal(spec1, check);
        if (_equals) {
          return true;
        }
        check = check.getBase();
      }
    } while((check != null));
    return false;
  }
  
  /**
   * Check if extends-relation on FDSpecifications has cycles.
   * 
   * @returns FDSpecification which has an extends-cycle, null if the
   *          extends-relation is cycle-free.
   */
  public FDSpecification getCyclicBaseSpec(final FDSpecification spec) {
    Set<FDSpecification> visited = CollectionLiterals.<FDSpecification>newHashSet();
    FDSpecification s = spec;
    FDSpecification last = null;
    do {
      {
        visited.add(s);
        last = s;
        s = s.getBase();
        if (((s != null) && visited.contains(s))) {
          return last;
        }
      }
    } while((s != null));
    return null;
  }
  
  /**
   * Check if a FDSpecification contains properties which lead
   * to clashes in the generated property accessor Java code.
   */
  public void checkClashingProperties(final FDSpecification spec) {
    final Function1<FDDeclaration, EList<FDPropertyDecl>> _function = new Function1<FDDeclaration, EList<FDPropertyDecl>>() {
      @Override
      public EList<FDPropertyDecl> apply(final FDDeclaration it) {
        return it.getProperties();
      }
    };
    final Iterable<FDPropertyDecl> allPropertyDecls = Iterables.<FDPropertyDecl>concat(ListExtensions.<FDDeclaration, EList<FDPropertyDecl>>map(spec.getDeclarations(), _function));
    final Function1<FDPropertyDecl, String> _function_1 = new Function1<FDPropertyDecl, String>() {
      @Override
      public String apply(final FDPropertyDecl it) {
        return it.getName();
      }
    };
    final Map<String, List<FDPropertyDecl>> groups = IterableExtensions.<String, FDPropertyDecl>groupBy(allPropertyDecls, _function_1);
    Set<String> _keySet = groups.keySet();
    for (final String propName : _keySet) {
      {
        final List<FDPropertyDecl> props = groups.get(propName);
        final Function1<FDPropertyDecl, FDPropertyHost> _function_2 = new Function1<FDPropertyDecl, FDPropertyHost>() {
          @Override
          public FDPropertyHost apply(final FDPropertyDecl it) {
            return FDeployValidatorAux.this.getHost(it);
          }
        };
        final Map<FDPropertyHost, List<FDPropertyDecl>> perHost = IterableExtensions.<FDPropertyHost, FDPropertyDecl>groupBy(props, _function_2);
        final Function1<List<FDPropertyDecl>, Boolean> _function_3 = new Function1<List<FDPropertyDecl>, Boolean>() {
          @Override
          public Boolean apply(final List<FDPropertyDecl> it) {
            int _size = it.size();
            return Boolean.valueOf((_size > 1));
          }
        };
        final Set<FDPropertyDecl> directDuplicates = IterableExtensions.<FDPropertyDecl>toSet(Iterables.<FDPropertyDecl>concat(IterableExtensions.<List<FDPropertyDecl>>filter(perHost.values(), _function_3)));
        for (final FDPropertyDecl pd : directDuplicates) {
          String _name = pd.getName();
          String _plus = ("Duplicate property name \'" + _name);
          String _plus_1 = (_plus + "\'");
          this.reporter.reportError(_plus_1, pd, FDeployPackage.Literals.FD_PROPERTY_DECL__NAME);
        }
        final Sets.SetView<FDPropertyDecl> localUnique = Sets.<FDPropertyDecl>difference(IterableExtensions.<FDPropertyDecl>toSet(props), directDuplicates);
        this.checkGroupArgumentType(localUnique);
        final Function1<FDPropertyDecl, Boolean> _function_4 = new Function1<FDPropertyDecl, Boolean>() {
          @Override
          public Boolean apply(final FDPropertyDecl it) {
            return Boolean.valueOf(FDeployValidatorAux.this.isEnumType(it));
          }
        };
        final Iterable<FDPropertyDecl> enumProps = IterableExtensions.<FDPropertyDecl>filter(localUnique, _function_4);
        int _size = IterableExtensions.size(enumProps);
        boolean _greaterThan = (_size > 1);
        if (_greaterThan) {
          for (final FDPropertyDecl ep : enumProps) {
            String _name_1 = ep.getName();
            String _plus_2 = ("Deployment property \'" + _name_1);
            String _plus_3 = (_plus_2 + "\' with an enumeration type has to be unique");
            this.reporter.reportError(_plus_3, ep, FDeployPackage.Literals.FD_PROPERTY_DECL__NAME);
          }
        }
      }
    }
  }
  
  /**
   * Check a group of properties according to the argument type of the
   * resulting property accessor get() method.<p/>
   * 
   * All properties in the input group should have the same name.
   * If the name is different, the actual getter argument type is
   * not relevant.
   * 
   * @param items the to-be-checked group of properties with the same name
   */
  private void checkGroupArgumentType(final Iterable<FDPropertyDecl> items) {
    final Function1<FDPropertyDecl, Class<? extends EObject>> _function = new Function1<FDPropertyDecl, Class<? extends EObject>>() {
      @Override
      public Class<? extends EObject> apply(final FDPropertyDecl it) {
        return HostLogic.getArgumentType(FDeployValidatorAux.this.getHost(it), HostLogic.Context.FRANCA_INTERFACE);
      }
    };
    final Map<Class<? extends EObject>, List<FDPropertyDecl>> argtypeGroups = IterableExtensions.<Class<? extends EObject>, FDPropertyDecl>groupBy(items, _function);
    final Map<Class<? extends EObject>, Iterable<Class<? extends EObject>>> colliding = CollectionLiterals.<Class<? extends EObject>, Iterable<Class<? extends EObject>>>newHashMap();
    final Set<Class<? extends EObject>> argtypes = argtypeGroups.keySet();
    for (final Class<? extends EObject> at : argtypes) {
      {
        final List<Class<? extends EObject>> colliders = CollectionLiterals.<Class<? extends EObject>>newArrayList();
        int _size = argtypeGroups.get(at).size();
        boolean _greaterThan = (_size > 1);
        if (_greaterThan) {
          colliders.add(at);
        }
        final Function1<Class<? extends EObject>, Boolean> _function_1 = new Function1<Class<? extends EObject>, Boolean>() {
          @Override
          public Boolean apply(final Class<? extends EObject> it) {
            return Boolean.valueOf(((!Objects.equal(it, at)) && FDeployValidatorAux.this.willCollide(it, at)));
          }
        };
        Iterable<Class<? extends EObject>> _filter = IterableExtensions.<Class<? extends EObject>>filter(argtypes, _function_1);
        Iterables.<Class<? extends EObject>>addAll(colliders, _filter);
        boolean _isEmpty = colliders.isEmpty();
        boolean _not = (!_isEmpty);
        if (_not) {
          colliding.put(at, colliders);
        }
      }
    }
    Set<Class<? extends EObject>> _keySet = colliding.keySet();
    for (final Class<? extends EObject> h : _keySet) {
      {
        final Function1<Class<? extends EObject>, List<FDPropertyDecl>> _function_1 = new Function1<Class<? extends EObject>, List<FDPropertyDecl>>() {
          @Override
          public List<FDPropertyDecl> apply(final Class<? extends EObject> it) {
            return argtypeGroups.get(it);
          }
        };
        final Set<FDPropertyDecl> collidingProps = IterableExtensions.<FDPropertyDecl>toSet(Iterables.<FDPropertyDecl>concat(IterableExtensions.<Class<? extends EObject>, List<FDPropertyDecl>>map(colliding.get(h), _function_1)));
        List<FDPropertyDecl> _get = argtypeGroups.get(h);
        for (final FDPropertyDecl p : _get) {
          {
            final Function1<FDPropertyDecl, Boolean> _function_2 = new Function1<FDPropertyDecl, Boolean>() {
              @Override
              public Boolean apply(final FDPropertyDecl it) {
                return Boolean.valueOf((!Objects.equal(it, p)));
              }
            };
            final Function1<FDPropertyDecl, FDPropertyHost> _function_3 = new Function1<FDPropertyDecl, FDPropertyHost>() {
              @Override
              public FDPropertyHost apply(final FDPropertyDecl it) {
                return FDeployValidatorAux.this.getHost(it);
              }
            };
            final Function1<FDPropertyHost, String> _function_4 = new Function1<FDPropertyHost, String>() {
              @Override
              public String apply(final FDPropertyHost it) {
                return it.getName();
              }
            };
            final String coll = IterableExtensions.join(IterableExtensions.<String>sort(IterableExtensions.<String>toSet(IterableExtensions.<FDPropertyHost, String>map(IterableExtensions.<FDPropertyDecl, FDPropertyHost>map(IterableExtensions.<FDPropertyDecl>filter(collidingProps, _function_2), _function_3), _function_4))), ", ");
            String _name = p.getName();
            String _plus = ("Name conflict for property \'" + _name);
            String _plus_1 = (_plus + "\' due to conflicting property hosts");
            String _plus_2 = (_plus_1 + 
              " (conflicting: ");
            String _plus_3 = (_plus_2 + coll);
            String _plus_4 = (_plus_3 + ")");
            this.reporter.reportError(_plus_4, p, FDeployPackage.Literals.FD_PROPERTY_DECL__NAME);
          }
        }
      }
    }
  }
  
  /**
   * Check if two types will collide if used as single arguments of
   * methods which use Java's static overloading.<p/>
   * 
   * Example with colliding types:
   * <code><pre>
   * public void get(FArgument a)
   * public void get(EObject b)
   * </pre></code><p/>
   * 
   * Example with independent types:
   * <code><pre>
   * public void get(FArgument a)
   * public void get(FMethod b)
   * </pre></code><p/>
   */
  private boolean willCollide(final Class<? extends EObject> a, final Class<? extends EObject> b) {
    return (a.isAssignableFrom(b) || b.isAssignableFrom(a));
  }
  
  private boolean isEnumType(final FDPropertyDecl decl) {
    boolean _xblockexpression = false;
    {
      final FDType t = decl.getType().getComplex();
      _xblockexpression = ((t != null) && (t instanceof FDEnumType));
    }
    return _xblockexpression;
  }
  
  private FDPropertyHost getHost(final FDPropertyDecl decl) {
    FDPropertyHost _xblockexpression = null;
    {
      EObject _eContainer = decl.eContainer();
      final FDDeclaration declaration = ((FDDeclaration) _eContainer);
      _xblockexpression = declaration.getHost();
    }
    return _xblockexpression;
  }
  
  public boolean checkUsedTypes(final FDTypes rootElem, final List<FType> localTypes, final PropertyDefChecker checker) {
    boolean _xblockexpression = false;
    {
      final Set<FTypeRef> typerefs = CollectionLiterals.<FTypeRef>newHashSet();
      EList<FType> _types = rootElem.getTarget().getTypes();
      for (final FType t : _types) {
        typerefs.addAll(IterableExtensions.<FTypeRef>toSet(Iterables.<FTypeRef>filter(EcoreUtil2.eAllContents(t), FTypeRef.class)));
      }
      _xblockexpression = this.checkUsedTypesRoot(rootElem, typerefs, localTypes, checker);
    }
    return _xblockexpression;
  }
  
  public boolean checkUsedTypes(final FDInterface rootElem, final List<FType> localTypes, final PropertyDefChecker checker) {
    final Set<FTypeRef> typerefs = IterableExtensions.<FTypeRef>toSet(Iterables.<FTypeRef>filter(EcoreUtil2.eAllContents(rootElem.getTarget()), FTypeRef.class));
    return this.checkUsedTypesRoot(rootElem, typerefs, localTypes, checker);
  }
  
  private boolean checkUsedTypesRoot(final FDRootElement rootElem, final Collection<FTypeRef> typerefs, final List<FType> localTypes, final PropertyDefChecker checker) {
    boolean hasError = false;
    final Function1<FTypeRef, FType> _function = new Function1<FTypeRef, FType>() {
      @Override
      public FType apply(final FTypeRef it) {
        return it.getDerived();
      }
    };
    final Function1<FType, Boolean> _function_1 = new Function1<FType, Boolean>() {
      @Override
      public Boolean apply(final FType it) {
        return Boolean.valueOf(FDeployValidatorAux.this.isDeploymentRelevantType(it));
      }
    };
    final Set<FType> referencedTypes = IterableExtensions.<FType>toSet(IterableExtensions.<FType>filter(IterableExtensions.<FType>filterNull(IterableExtensions.<FTypeRef, FType>map(typerefs, _function)), _function_1));
    final Function1<FType, Boolean> _function_2 = new Function1<FType, Boolean>() {
      @Override
      public Boolean apply(final FType it) {
        boolean _contains = localTypes.contains(it);
        return Boolean.valueOf((!_contains));
      }
    };
    final Set<FType> nonLocal = IterableExtensions.<FType>toSet(IterableExtensions.<FType>filter(referencedTypes, _function_2));
    final Set<FType> fromOthers = this.getTypeDefinitionsByTransitiveUse(rootElem);
    final Function1<FType, Boolean> _function_3 = new Function1<FType, Boolean>() {
      @Override
      public Boolean apply(final FType it) {
        boolean _contains = fromOthers.contains(it);
        return Boolean.valueOf((!_contains));
      }
    };
    final Set<FType> remaining = IterableExtensions.<FType>toSet(IterableExtensions.<FType>filter(nonLocal, _function_3));
    for (final FType missing : remaining) {
      boolean _mustBeDefined = this.mustBeDefined(checker, missing);
      if (_mustBeDefined) {
        final FModel model = FrancaModelExtensions.getModel(missing);
        String _name = missing.getName();
        String _plus = ("Deployment for type \'" + _name);
        String _plus_1 = (_plus + "\' is missing, ");
        String _plus_2 = (_plus_1 + 
          "add \'use\' reference for deployment of package \'");
        String _name_1 = model.getName();
        String _plus_3 = (_plus_2 + _name_1);
        String _plus_4 = (_plus_3 + "\'");
        this.reporter.reportError(_plus_4, rootElem, FDeployPackage.Literals.FD_INTERFACE__TARGET);
        hasError = true;
      }
    }
    return hasError;
  }
  
  private boolean mustBeDefined(final PropertyDefChecker checker, final FType t) {
    boolean _switchResult = false;
    boolean _matched = false;
    if (t instanceof FArrayType) {
      _matched=true;
      _switchResult = checker.mustBeDefined(((FArrayType)t));
    }
    if (!_matched) {
      if (t instanceof FStructType) {
        _matched=true;
        _switchResult = checker.mustBeDefined(((FStructType)t));
      }
    }
    if (!_matched) {
      if (t instanceof FUnionType) {
        _matched=true;
        _switchResult = checker.mustBeDefined(((FUnionType)t));
      }
    }
    if (!_matched) {
      if (t instanceof FEnumerationType) {
        _matched=true;
        _switchResult = checker.mustBeDefined(((FEnumerationType)t));
      }
    }
    if (!_matched) {
      _switchResult = true;
    }
    return _switchResult;
  }
  
  private boolean isDeploymentRelevantType(final FType it) {
    return ((((it instanceof FArrayType) || (it instanceof FStructType)) || (it instanceof FUnionType)) || (it instanceof FEnumerationType));
  }
  
  private Set<FType> getTypeDefinitionsByTransitiveUse(final FDRootElement rootElem) {
    Set<FType> _xblockexpression = null;
    {
      Set<FDRootElement> visited = CollectionLiterals.<FDRootElement>newHashSet(rootElem);
      Queue<FDRootElement> work = CollectionLiterals.<FDRootElement>newLinkedList();
      Set<FType> found = CollectionLiterals.<FType>newHashSet();
      work.addAll(rootElem.getUse());
      while ((!work.isEmpty())) {
        {
          final FDRootElement e = work.poll();
          boolean _contains = visited.contains(e);
          boolean _not = (!_contains);
          if (_not) {
            visited.add(e);
            found.addAll(this.getLocalTypes(e));
            work.addAll(e.getUse());
          }
        }
      }
      _xblockexpression = found;
    }
    return _xblockexpression;
  }
  
  /**
   * Get types defined locally in a root element.
   */
  private Collection<FType> getLocalTypes(final FDRootElement rootElem) {
    List<FType> _switchResult = null;
    boolean _matched = false;
    if (rootElem instanceof FDInterface) {
      _matched=true;
      _switchResult = ((FDInterface)rootElem).getTarget().getTypes();
    }
    if (!_matched) {
      if (rootElem instanceof FDTypes) {
        _matched=true;
        _switchResult = ((FDTypes)rootElem).getTarget().getTypes();
      }
    }
    if (!_matched) {
      _switchResult = CollectionLiterals.<FType>newArrayList();
    }
    return _switchResult;
  }
}
