/**
 * Copyright (c) 2013 itemis AG (http://www.itemis.de).
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.franca.deploymodel.dsl.validation;

import java.util.List;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.franca.core.framework.FrancaHelpers;
import org.franca.core.franca.FArgument;
import org.franca.core.franca.FArrayType;
import org.franca.core.franca.FAttribute;
import org.franca.core.franca.FBroadcast;
import org.franca.core.franca.FEnumerationType;
import org.franca.core.franca.FField;
import org.franca.core.franca.FMapType;
import org.franca.core.franca.FMethod;
import org.franca.core.franca.FStructType;
import org.franca.core.franca.FTypeDef;
import org.franca.core.franca.FTypeRef;
import org.franca.core.franca.FUnionType;
import org.franca.deploymodel.core.FDPropertyHost;
import org.franca.deploymodel.dsl.FDSpecificationExtender;
import org.franca.deploymodel.dsl.fDeploy.FDBuiltInPropertyHost;

/**
 * Compute if a given Franca IDL element has to be defined in
 * a deployment definition because of mandatory deployment properties.
 * 
 * This is a helper class for Franca deployment validation. It uses a
 * specification extender in order to get information about which
 * property hosts are mandatory according to a deployment specification.
 */
@SuppressWarnings("all")
public class PropertyDefChecker {
  private FDSpecificationExtender specHelper;
  
  public PropertyDefChecker(final FDSpecificationExtender specHelper) {
    this.specHelper = specHelper;
  }
  
  public boolean mustBeDefined(final FMethod it) {
    boolean _xblockexpression = false;
    {
      boolean _isMandatory = this.specHelper.isMandatory(FDBuiltInPropertyHost.METHODS);
      if (_isMandatory) {
        return true;
      }
      if ((it.getInArgs().isEmpty() && it.getOutArgs().isEmpty())) {
        return false;
      }
      boolean _isMandatory_1 = this.specHelper.isMandatory(FDBuiltInPropertyHost.ARGUMENTS);
      if (_isMandatory_1) {
        return true;
      }
      final Function1<FArgument, Boolean> _function = new Function1<FArgument, Boolean>() {
        @Override
        public Boolean apply(final FArgument it) {
          return Boolean.valueOf(PropertyDefChecker.this.mustBeDefined(it));
        }
      };
      FArgument _findFirst = IterableExtensions.<FArgument>findFirst(it.getInArgs(), _function);
      boolean _tripleNotEquals = (_findFirst != null);
      if (_tripleNotEquals) {
        return true;
      }
      final Function1<FArgument, Boolean> _function_1 = new Function1<FArgument, Boolean>() {
        @Override
        public Boolean apply(final FArgument it) {
          return Boolean.valueOf(PropertyDefChecker.this.mustBeDefined(it));
        }
      };
      FArgument _findFirst_1 = IterableExtensions.<FArgument>findFirst(it.getOutArgs(), _function_1);
      boolean _tripleNotEquals_1 = (_findFirst_1 != null);
      if (_tripleNotEquals_1) {
        return true;
      }
      _xblockexpression = false;
    }
    return _xblockexpression;
  }
  
  public boolean mustBeDefined(final FBroadcast it) {
    boolean _xblockexpression = false;
    {
      boolean _isMandatory = this.specHelper.isMandatory(FDBuiltInPropertyHost.BROADCASTS);
      if (_isMandatory) {
        return true;
      }
      boolean _isEmpty = it.getOutArgs().isEmpty();
      if (_isEmpty) {
        return false;
      }
      boolean _isMandatory_1 = this.specHelper.isMandatory(FDBuiltInPropertyHost.ARGUMENTS);
      if (_isMandatory_1) {
        return true;
      }
      final Function1<FArgument, Boolean> _function = new Function1<FArgument, Boolean>() {
        @Override
        public Boolean apply(final FArgument it) {
          return Boolean.valueOf(PropertyDefChecker.this.mustBeDefined(it));
        }
      };
      FArgument _findFirst = IterableExtensions.<FArgument>findFirst(it.getOutArgs(), _function);
      boolean _tripleNotEquals = (_findFirst != null);
      if (_tripleNotEquals) {
        return true;
      }
      _xblockexpression = false;
    }
    return _xblockexpression;
  }
  
  public boolean mustBeDefined(final FArrayType it) {
    return this.specHelper.isMandatory(FDBuiltInPropertyHost.ARRAYS);
  }
  
  public boolean mustBeDefined(final FStructType it) {
    boolean _xblockexpression = false;
    {
      boolean _isMandatory = this.specHelper.isMandatory(FDBuiltInPropertyHost.STRUCTS);
      if (_isMandatory) {
        return true;
      }
      _xblockexpression = this.mustBeDefined(it.getElements(), FDPropertyHost.builtIn(FDBuiltInPropertyHost.STRUCT_FIELDS));
    }
    return _xblockexpression;
  }
  
  public boolean mustBeDefined(final FUnionType it) {
    boolean _xblockexpression = false;
    {
      boolean _isMandatory = this.specHelper.isMandatory(FDBuiltInPropertyHost.UNIONS);
      if (_isMandatory) {
        return true;
      }
      _xblockexpression = this.mustBeDefined(it.getElements(), FDPropertyHost.builtIn(FDBuiltInPropertyHost.UNION_FIELDS));
    }
    return _xblockexpression;
  }
  
  public boolean mustBeDefined(final List<FField> it, final FDPropertyHost host) {
    boolean _xblockexpression = false;
    {
      boolean _isEmpty = it.isEmpty();
      if (_isEmpty) {
        return false;
      }
      boolean _isMandatory = this.specHelper.isMandatory(host);
      if (_isMandatory) {
        return true;
      }
      final Function1<FField, Boolean> _function = new Function1<FField, Boolean>() {
        @Override
        public Boolean apply(final FField it) {
          return Boolean.valueOf(PropertyDefChecker.this.mustBeDefined(it));
        }
      };
      FField _findFirst = IterableExtensions.<FField>findFirst(it, _function);
      boolean _tripleNotEquals = (_findFirst != null);
      if (_tripleNotEquals) {
        return true;
      }
      _xblockexpression = false;
    }
    return _xblockexpression;
  }
  
  public boolean mustBeDefined(final FEnumerationType it) {
    boolean _xblockexpression = false;
    {
      boolean _isMandatory = this.specHelper.isMandatory(FDBuiltInPropertyHost.ENUMERATIONS);
      if (_isMandatory) {
        return true;
      }
      boolean _isEmpty = it.getEnumerators().isEmpty();
      if (_isEmpty) {
        return false;
      }
      boolean _isMandatory_1 = this.specHelper.isMandatory(FDBuiltInPropertyHost.ENUMERATORS);
      if (_isMandatory_1) {
        return true;
      }
      _xblockexpression = false;
    }
    return _xblockexpression;
  }
  
  public boolean mustBeDefined(final FTypeDef it) {
    return this.specHelper.isMandatory(FDBuiltInPropertyHost.TYPEDEFS);
  }
  
  public boolean mustBeDefined(final FAttribute it) {
    return (this.specHelper.isMandatory(FDBuiltInPropertyHost.ATTRIBUTES) || this.mustBeDefined(it.getType(), it.isArray()));
  }
  
  public boolean mustBeDefined(final FArgument it) {
    return (this.specHelper.isMandatory(FDBuiltInPropertyHost.ARGUMENTS) || this.mustBeDefined(it.getType(), it.isArray()));
  }
  
  public boolean mustBeDefined(final FField it) {
    boolean _xblockexpression = false;
    {
      EObject _eContainer = it.eContainer();
      final boolean isStruct = (_eContainer instanceof FStructType);
      FDBuiltInPropertyHost _xifexpression = null;
      if (isStruct) {
        _xifexpression = FDBuiltInPropertyHost.STRUCT_FIELDS;
      } else {
        _xifexpression = FDBuiltInPropertyHost.UNION_FIELDS;
      }
      final FDBuiltInPropertyHost host = _xifexpression;
      _xblockexpression = ((this.specHelper.isMandatory(host) || this.specHelper.isMandatory(FDBuiltInPropertyHost.FIELDS)) || this.mustBeDefined(it.getType(), it.isArray()));
    }
    return _xblockexpression;
  }
  
  public boolean mustBeDefined(final FMapType it) {
    return ((this.specHelper.isMandatory(FDBuiltInPropertyHost.MAPS) || this.mustBeDefined(it.getKeyType())) || this.mustBeDefined(it.getValueType()));
  }
  
  private boolean mustBeDefined(final FTypeRef it, final boolean isInlineArray) {
    boolean _xblockexpression = false;
    {
      if (isInlineArray) {
        boolean _isMandatory = this.specHelper.isMandatory(FDBuiltInPropertyHost.ARRAYS);
        if (_isMandatory) {
          return true;
        }
      }
      _xblockexpression = this.mustBeDefined(it);
    }
    return _xblockexpression;
  }
  
  public boolean mustBeDefined(final FTypeRef it) {
    boolean _xblockexpression = false;
    {
      boolean _isString = FrancaHelpers.isString(it);
      if (_isString) {
        boolean _isMandatory = this.specHelper.isMandatory(FDBuiltInPropertyHost.STRINGS);
        if (_isMandatory) {
          return true;
        }
      } else {
        boolean _isInteger = FrancaHelpers.isInteger(it);
        if (_isInteger) {
          if ((this.specHelper.isMandatory(FDBuiltInPropertyHost.INTEGERS) || this.specHelper.isMandatory(FDBuiltInPropertyHost.NUMBERS))) {
            return true;
          }
        } else {
          boolean _isFloatingPoint = FrancaHelpers.isFloatingPoint(it);
          if (_isFloatingPoint) {
            if ((this.specHelper.isMandatory(FDBuiltInPropertyHost.FLOATS) || this.specHelper.isMandatory(FDBuiltInPropertyHost.NUMBERS))) {
              return true;
            }
          } else {
            boolean _isBoolean = FrancaHelpers.isBoolean(it);
            if (_isBoolean) {
              boolean _isMandatory_1 = this.specHelper.isMandatory(FDBuiltInPropertyHost.BOOLEANS);
              if (_isMandatory_1) {
                return true;
              }
            } else {
              boolean _isByteBuffer = FrancaHelpers.isByteBuffer(it);
              if (_isByteBuffer) {
                boolean _isMandatory_2 = this.specHelper.isMandatory(FDBuiltInPropertyHost.BYTE_BUFFERS);
                if (_isMandatory_2) {
                  return true;
                }
              }
            }
          }
        }
      }
      _xblockexpression = false;
    }
    return _xblockexpression;
  }
}
