/**
 * Copyright (c) 2018 itemis AG (http://www.itemis.de).
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.franca.deploymodel.extensions;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.common.collect.LinkedListMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.Platform;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.franca.deploymodel.dsl.fDeploy.FDAbstractExtensionElement;
import org.franca.deploymodel.dsl.fDeploy.FDExtensionElement;
import org.franca.deploymodel.dsl.fDeploy.FDExtensionRoot;
import org.franca.deploymodel.dsl.fDeploy.FDeployPackage;
import org.franca.deploymodel.extensions.ExtensionUtils;
import org.franca.deploymodel.extensions.IFDeployExtension;

/**
 * This is the registry for deployment extensions.</p>
 * 
 * It can be used in the IDE (with Eclipse's regular extension point mechanism) or
 * in standalone mode.</p>
 * 
 * @author Klaus Birken (itemis AG)
 */
@SuppressWarnings("all")
public class ExtensionRegistry {
  private final static String EXTENSION_POINT_ID = "org.franca.deploymodel.dsl.deploymentExtension";
  
  private static List<IFDeployExtension> extensions = null;
  
  private static Map<EClass, Iterable<IFDeployExtension.Host>> allAdditionalHosts = CollectionLiterals.<EClass, Iterable<IFDeployExtension.Host>>newHashMap();
  
  private static Map<IFDeployExtension.Host, Set<EClass>> hostingClasses = CollectionLiterals.<IFDeployExtension.Host, Set<EClass>>newHashMap();
  
  private static Map<EClass, IFDeployExtension> mixinExtensions = CollectionLiterals.<EClass, IFDeployExtension>newHashMap();
  
  private static Map<EClass, EClassifier> accessorArgumentType = CollectionLiterals.<EClass, EClassifier>newHashMap();
  
  private static Set<EClass> isNonFrancaMixin = CollectionLiterals.<EClass>newHashSet();
  
  private static Map<EClass, String> nonFrancaMixinRootPrefix = CollectionLiterals.<EClass, String>newHashMap();
  
  private static Multimap<EClass, EClass> mixinChildren = LinkedListMultimap.<EClass, EClass>create();
  
  /**
   * Add extension to registry.</p>
   * 
   * This should only be used in standalone mode. For the IDE,
   * use the extension point (see above) for registration.</p>
   * 
   * @param ^extension the Franca deployment extension to be registered
   */
  public static void addExtension(final IFDeployExtension extension) {
    if ((ExtensionRegistry.extensions == null)) {
      ExtensionRegistry.extensions = Lists.<IFDeployExtension>newArrayList();
    }
    ExtensionRegistry.register(extension);
  }
  
  /**
   * Reset the registry and remove all extensions.</p>
   * 
   * This should be used by standalone tests which call addExtension() to
   * register an extension and test it.</p>
   * 
   * Note that this method will remove <em>all</em> registered extensions.
   * Due to complex cached data structures in the registry it is not possible
   * to remove a single specific extension in the current implementation.</p>
   */
  public static void reset() {
    ExtensionRegistry.extensions.clear();
    ExtensionRegistry.allAdditionalHosts.clear();
    ExtensionRegistry.hostingClasses.clear();
    ExtensionRegistry.mixinExtensions.clear();
    ExtensionRegistry.accessorArgumentType.clear();
    ExtensionRegistry.isNonFrancaMixin.clear();
    ExtensionRegistry.mixinChildren.clear();
  }
  
  /**
   * Get all registered extensions.</p>
   * 
   * This will initialize the registry on demand.</p>
   * 
   * @return list of all registered extensions
   */
  public static Collection<IFDeployExtension> getExtensions() {
    if ((ExtensionRegistry.extensions == null)) {
      ExtensionRegistry.initializeValidators();
    }
    return ExtensionRegistry.extensions;
  }
  
  /**
   * Internal helper which initializes all deployment extensions
   * that have been registered as Eclipse extension points.</p>
   */
  private static void initializeValidators() {
    if ((ExtensionRegistry.extensions == null)) {
      ExtensionRegistry.extensions = Lists.<IFDeployExtension>newArrayList();
    }
    IExtensionRegistry reg = Platform.getExtensionRegistry();
    if ((reg == null)) {
      return;
    }
    IExtensionPoint ep = reg.getExtensionPoint(ExtensionRegistry.EXTENSION_POINT_ID);
    IExtension[] _extensions = ep.getExtensions();
    for (final IExtension ext : _extensions) {
      IConfigurationElement[] _configurationElements = ext.getConfigurationElements();
      for (final IConfigurationElement ce : _configurationElements) {
        boolean _equals = ce.getName().equals("validator");
        if (_equals) {
          try {
            Object o = ce.createExecutableExtension("class");
            if ((o instanceof IFDeployExtension)) {
              IFDeployExtension dExt = ((IFDeployExtension) o);
              ExtensionRegistry.register(dExt);
            }
          } catch (final Throwable _t) {
            if (_t instanceof CoreException) {
              final CoreException e = (CoreException)_t;
              e.printStackTrace();
            } else {
              throw Exceptions.sneakyThrow(_t);
            }
          }
        }
      }
    }
  }
  
  /**
   * This is the central registration method for new extensions.</p>
   * 
   * All extensions will be registered here, either via the IDE's extension point
   * infrastructure or via method addExtension in the standalone case.</p>
   */
  private static void register(final IFDeployExtension extension) {
    final String desc = extension.getShortDescription();
    final Function1<IFDeployExtension, String> _function = new Function1<IFDeployExtension, String>() {
      @Override
      public String apply(final IFDeployExtension it) {
        return it.getShortDescription();
      }
    };
    boolean _contains = ListExtensions.<IFDeployExtension, String>map(ExtensionRegistry.extensions, _function).contains(desc);
    if (_contains) {
      System.err.println((("ERROR: Deployment extension \'" + desc) + "\' has already been registered!"));
      return;
    }
    ExtensionRegistry.extensions.add(extension);
    Collection<IFDeployExtension.RootDef> _roots = extension.getRoots();
    for (final IFDeployExtension.RootDef root : _roots) {
      {
        final Consumer<IFDeployExtension.Host> _function_1 = new Consumer<IFDeployExtension.Host>() {
          @Override
          public void accept(final IFDeployExtension.Host it) {
            ExtensionRegistry.addHostingClass(it, FDeployPackage.eINSTANCE.getFDExtensionRoot());
          }
        };
        root.getHosts().forEach(_function_1);
        final Consumer<IFDeployExtension.Host> _function_2 = new Consumer<IFDeployExtension.Host>() {
          @Override
          public void accept(final IFDeployExtension.Host it) {
            ExtensionRegistry.addHostingClass(it, FDeployPackage.eINSTANCE.getFDExtensionElement());
          }
        };
        ExtensionUtils.getHostsOnlyInSubtree(root).forEach(_function_2);
      }
    }
    Collection<IFDeployExtension.HostMixinDef> _mixins = extension.getMixins();
    for (final IFDeployExtension.HostMixinDef mixin : _mixins) {
      {
        final EClass clazz = mixin.getHostingClass();
        final Collection<IFDeployExtension.Host> hosts = mixin.getHosts();
        boolean _containsKey = ExtensionRegistry.allAdditionalHosts.containsKey(clazz);
        if (_containsKey) {
          final Iterable<IFDeployExtension.Host> previousHosts = ExtensionRegistry.allAdditionalHosts.get(clazz);
          final Iterable<IFDeployExtension.Host> joined = Iterables.<IFDeployExtension.Host>unmodifiableIterable(Iterables.<IFDeployExtension.Host>concat(previousHosts, hosts));
          ExtensionRegistry.allAdditionalHosts.put(clazz, joined);
        } else {
          ExtensionRegistry.allAdditionalHosts.put(clazz, hosts);
        }
        final Consumer<IFDeployExtension.Host> _function_1 = new Consumer<IFDeployExtension.Host>() {
          @Override
          public void accept(final IFDeployExtension.Host it) {
            ExtensionRegistry.addHostingClass(it, clazz);
          }
        };
        hosts.forEach(_function_1);
      }
    }
    Collection<IFDeployExtension.HostMixinDef> _mixins_1 = extension.getMixins();
    for (final IFDeployExtension.HostMixinDef mixin_1 : _mixins_1) {
      {
        final EClass clazz = mixin_1.getHostingClass();
        ExtensionRegistry.mixinExtensions.put(clazz, extension);
        IFDeployExtension.HostMixinDef.AccessorArgumentStyle _accessorArgument = mixin_1.getAccessorArgument();
        boolean _tripleEquals = (_accessorArgument == IFDeployExtension.HostMixinDef.AccessorArgumentStyle.BY_TARGET_FEATURE);
        if (_tripleEquals) {
          final EClassifier targetType = ExtensionRegistry.getClassOfTargetFeature(clazz);
          boolean _containsKey = ExtensionRegistry.accessorArgumentType.containsKey(clazz);
          if (_containsKey) {
            final EClassifier targetType0 = ExtensionRegistry.accessorArgumentType.get(clazz);
            if ((targetType0 != targetType)) {
              String _name = clazz.getName();
              String _plus = ("ERROR: Duplicate argument type definition for class \'" + _name);
              String _plus_1 = (_plus + "\'");
              System.err.println(_plus_1);
            }
          } else {
            ExtensionRegistry.accessorArgumentType.put(clazz, targetType);
          }
        }
        boolean _isNonFrancaMixin = mixin_1.isNonFrancaMixin();
        if (_isNonFrancaMixin) {
          ExtensionRegistry.isNonFrancaMixin.add(clazz);
          String _accessorRootPrefix = mixin_1.getAccessorRootPrefix();
          boolean _tripleNotEquals = (_accessorRootPrefix != null);
          if (_tripleNotEquals) {
            final String prefix = mixin_1.getAccessorRootPrefix();
            ExtensionRegistry.nonFrancaMixinRootPrefix.put(clazz, prefix);
          }
        }
      }
    }
    final Function1<IFDeployExtension.HostMixinDef, Boolean> _function_1 = new Function1<IFDeployExtension.HostMixinDef, Boolean>() {
      @Override
      public Boolean apply(final IFDeployExtension.HostMixinDef it) {
        return Boolean.valueOf(it.isChildMixin());
      }
    };
    final Function1<IFDeployExtension.HostMixinDef, EClass> _function_2 = new Function1<IFDeployExtension.HostMixinDef, EClass>() {
      @Override
      public EClass apply(final IFDeployExtension.HostMixinDef it) {
        return it.getHostingClass();
      }
    };
    final Set<EClass> children = IterableExtensions.<EClass>toSet(IterableExtensions.<IFDeployExtension.HostMixinDef, EClass>map(IterableExtensions.<IFDeployExtension.HostMixinDef>filter(extension.getMixins(), _function_1), _function_2));
    final Function1<IFDeployExtension.HostMixinDef, Boolean> _function_3 = new Function1<IFDeployExtension.HostMixinDef, Boolean>() {
      @Override
      public Boolean apply(final IFDeployExtension.HostMixinDef it) {
        String _accessorRootPrefix = it.getAccessorRootPrefix();
        return Boolean.valueOf((_accessorRootPrefix != null));
      }
    };
    final Function1<IFDeployExtension.HostMixinDef, EClass> _function_4 = new Function1<IFDeployExtension.HostMixinDef, EClass>() {
      @Override
      public EClass apply(final IFDeployExtension.HostMixinDef it) {
        return it.getHostingClass();
      }
    };
    final Iterable<EClass> roots = IterableExtensions.<IFDeployExtension.HostMixinDef, EClass>map(IterableExtensions.<IFDeployExtension.HostMixinDef>filter(extension.getMixins(), _function_3), _function_4);
    for (final EClass rootMixin : roots) {
      {
        final Function1<EStructuralFeature, EClassifier> _function_5 = new Function1<EStructuralFeature, EClassifier>() {
          @Override
          public EClassifier apply(final EStructuralFeature it) {
            return it.getEType();
          }
        };
        final Set<EClassifier> usedClasses = IterableExtensions.<EClassifier>toSet(ListExtensions.<EStructuralFeature, EClassifier>map(rootMixin.getEStructuralFeatures(), _function_5));
        final Sets.SetView<EClass> both = Sets.<EClass>intersection(children, usedClasses);
        ExtensionRegistry.mixinChildren.putAll(rootMixin, both);
      }
    }
  }
  
  private static boolean addHostingClass(final IFDeployExtension.Host host, final EClass clazz) {
    boolean _xblockexpression = false;
    {
      boolean _containsKey = ExtensionRegistry.hostingClasses.containsKey(host);
      boolean _not = (!_containsKey);
      if (_not) {
        ExtensionRegistry.hostingClasses.put(host, CollectionLiterals.<EClass>newHashSet());
      }
      _xblockexpression = ExtensionRegistry.hostingClasses.get(host).add(clazz);
    }
    return _xblockexpression;
  }
  
  /**
   * Helper to get the classifier type from EMF feature "target".</p>
   */
  private static EClassifier getClassOfTargetFeature(final EClass clazz) {
    EClassifier _xblockexpression = null;
    {
      final Function1<EReference, Boolean> _function = new Function1<EReference, Boolean>() {
        @Override
        public Boolean apply(final EReference it) {
          String _name = it.getName();
          return Boolean.valueOf(Objects.equal(_name, "target"));
        }
      };
      final EReference targetFeature = IterableExtensions.<EReference>findFirst(clazz.getEAllReferences(), _function);
      EClassifier _xifexpression = null;
      if ((targetFeature != null)) {
        EClassifier _xifexpression_1 = null;
        EClassifier _eType = targetFeature.getEType();
        if ((_eType instanceof EClassifier)) {
          _xifexpression_1 = targetFeature.getEType();
        } else {
          _xifexpression_1 = null;
        }
        _xifexpression = _xifexpression_1;
      } else {
        _xifexpression = null;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  public static Map<IFDeployExtension.Host, IFDeployExtension> getHosts() {
    HashMap<IFDeployExtension.Host, IFDeployExtension> _xblockexpression = null;
    {
      final HashMap<IFDeployExtension.Host, IFDeployExtension> result = CollectionLiterals.<IFDeployExtension.Host, IFDeployExtension>newHashMap();
      Collection<IFDeployExtension> _extensions = ExtensionRegistry.getExtensions();
      for (final IFDeployExtension ext : _extensions) {
        {
          final Function1<IFDeployExtension.Host, IFDeployExtension> _function = new Function1<IFDeployExtension.Host, IFDeployExtension>() {
            @Override
            public IFDeployExtension apply(final IFDeployExtension.Host it) {
              return ext;
            }
          };
          final Map<IFDeployExtension.Host, IFDeployExtension> extHosts = IterableExtensions.<IFDeployExtension.Host, IFDeployExtension>toInvertedMap(ExtensionUtils.getAllHosts(ext), _function);
          result.putAll(extHosts);
        }
      }
      _xblockexpression = result;
    }
    return _xblockexpression;
  }
  
  public static IFDeployExtension.Host findHost(final String hostname) {
    IFDeployExtension.Host _xblockexpression = null;
    {
      final Set<IFDeployExtension.Host> hosts = ExtensionRegistry.getHosts().keySet();
      final Function1<IFDeployExtension.Host, Boolean> _function = new Function1<IFDeployExtension.Host, Boolean>() {
        @Override
        public Boolean apply(final IFDeployExtension.Host it) {
          String _name = it.getName();
          return Boolean.valueOf(Objects.equal(_name, hostname));
        }
      };
      _xblockexpression = IterableExtensions.<IFDeployExtension.Host>findFirst(hosts, _function);
    }
    return _xblockexpression;
  }
  
  public static Map<IFDeployExtension.RootDef, IFDeployExtension> getRoots() {
    HashMap<IFDeployExtension.RootDef, IFDeployExtension> _xblockexpression = null;
    {
      final HashMap<IFDeployExtension.RootDef, IFDeployExtension> result = CollectionLiterals.<IFDeployExtension.RootDef, IFDeployExtension>newHashMap();
      Collection<IFDeployExtension> _extensions = ExtensionRegistry.getExtensions();
      for (final IFDeployExtension ext : _extensions) {
        Collection<IFDeployExtension.RootDef> _roots = ext.getRoots();
        for (final IFDeployExtension.RootDef root : _roots) {
          result.put(root, ext);
        }
      }
      _xblockexpression = result;
    }
    return _xblockexpression;
  }
  
  public static IFDeployExtension.RootDef findRoot(final String rootTag) {
    IFDeployExtension.RootDef _xblockexpression = null;
    {
      final Function1<IFDeployExtension, Collection<IFDeployExtension.RootDef>> _function = new Function1<IFDeployExtension, Collection<IFDeployExtension.RootDef>>() {
        @Override
        public Collection<IFDeployExtension.RootDef> apply(final IFDeployExtension it) {
          return it.getRoots();
        }
      };
      final Iterable<IFDeployExtension.RootDef> roots = Iterables.<IFDeployExtension.RootDef>concat(IterableExtensions.<IFDeployExtension, Collection<IFDeployExtension.RootDef>>map(ExtensionRegistry.getExtensions(), _function));
      final Function1<IFDeployExtension.RootDef, Boolean> _function_1 = new Function1<IFDeployExtension.RootDef, Boolean>() {
        @Override
        public Boolean apply(final IFDeployExtension.RootDef it) {
          String _tag = it.getTag();
          return Boolean.valueOf(Objects.equal(_tag, rootTag));
        }
      };
      _xblockexpression = IterableExtensions.<IFDeployExtension.RootDef>findFirst(roots, _function_1);
    }
    return _xblockexpression;
  }
  
  public static boolean hasHostSubtree(final IFDeployExtension.AbstractElementDef elem, final IFDeployExtension.Host host) {
    return ExtensionUtils.getAllHosts(elem).contains(host);
  }
  
  /**
   * Get metamodel ElementDef from model FDAbstractExtensionElement (which is an EObject)</p>
   */
  public static IFDeployExtension.AbstractElementDef getElement(final FDAbstractExtensionElement elem) {
    IFDeployExtension.AbstractElementDef _switchResult = null;
    boolean _matched = false;
    if (elem instanceof FDExtensionRoot) {
      _matched=true;
      _switchResult = ExtensionRegistry.findRoot(((FDExtensionRoot)elem).getTag());
    }
    if (!_matched) {
      if (elem instanceof FDExtensionElement) {
        _matched=true;
        IFDeployExtension.ElementDef _xblockexpression = null;
        {
          EObject _eContainer = ((FDExtensionElement)elem).eContainer();
          final IFDeployExtension.AbstractElementDef parent = ExtensionRegistry.getElement(((FDAbstractExtensionElement) _eContainer));
          final Function1<IFDeployExtension.ElementDef, Boolean> _function = new Function1<IFDeployExtension.ElementDef, Boolean>() {
            @Override
            public Boolean apply(final IFDeployExtension.ElementDef it) {
              String _tag = it.getTag();
              String _tag_1 = ((FDExtensionElement)elem).getTag();
              return Boolean.valueOf(Objects.equal(_tag, _tag_1));
            }
          };
          _xblockexpression = IterableExtensions.<IFDeployExtension.ElementDef>findFirst(parent.getChildren(), _function);
        }
        _switchResult = _xblockexpression;
      }
    }
    if (!_matched) {
      return null;
    }
    return _switchResult;
  }
  
  public static Iterable<IFDeployExtension.Host> getAdditionalHosts(final EClass clazz) {
    Iterable<IFDeployExtension.Host> _xblockexpression = null;
    {
      if ((ExtensionRegistry.extensions == null)) {
        ExtensionRegistry.initializeValidators();
      }
      final Iterable<IFDeployExtension.Host> result = ExtensionRegistry.allAdditionalHosts.get(clazz);
      Iterable<IFDeployExtension.Host> _xifexpression = null;
      if ((result != null)) {
        _xifexpression = result;
      } else {
        _xifexpression = CollectionLiterals.<IFDeployExtension.Host>newArrayList();
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  public static Set<EClass> getHostingClasses(final IFDeployExtension.Host host) {
    Set<EClass> _xblockexpression = null;
    {
      if ((ExtensionRegistry.extensions == null)) {
        ExtensionRegistry.initializeValidators();
      }
      final Set<EClass> result = ExtensionRegistry.hostingClasses.get(host);
      Set<EClass> _xifexpression = null;
      if ((result != null)) {
        _xifexpression = result;
      } else {
        _xifexpression = CollectionLiterals.<EClass>newHashSet();
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  public static EClassifier getAccessorArgumentType(final EClass clazz) {
    EClassifier _xifexpression = null;
    boolean _containsKey = ExtensionRegistry.accessorArgumentType.containsKey(clazz);
    if (_containsKey) {
      _xifexpression = ExtensionRegistry.accessorArgumentType.get(clazz);
    } else {
      _xifexpression = clazz;
    }
    return _xifexpression;
  }
  
  public static IFDeployExtension getMixinExtension(final EClass clazz) {
    return ExtensionRegistry.mixinExtensions.get(clazz);
  }
  
  public static Set<EClass> getNonFrancaMixinRoots() {
    return ExtensionRegistry.nonFrancaMixinRootPrefix.keySet();
  }
  
  public static String getNonFrancaMixinPrefix(final EClass clazz) {
    return ExtensionRegistry.nonFrancaMixinRootPrefix.get(clazz);
  }
  
  public static Iterable<EClass> getMixinClasses(final EClass rootClass) {
    return Iterables.<EClass>concat(ExtensionRegistry.mixinChildren.get(rootClass), CollectionLiterals.<EClass>newArrayList(rootClass));
  }
  
  public static boolean isNonFrancaMixinHost(final EClass clazz) {
    return ExtensionRegistry.isNonFrancaMixin.contains(clazz);
  }
  
  public static Map<IFDeployExtension.TypeDef, IFDeployExtension> getTypes() {
    HashMap<IFDeployExtension.TypeDef, IFDeployExtension> _xblockexpression = null;
    {
      final HashMap<IFDeployExtension.TypeDef, IFDeployExtension> result = CollectionLiterals.<IFDeployExtension.TypeDef, IFDeployExtension>newHashMap();
      Collection<IFDeployExtension> _extensions = ExtensionRegistry.getExtensions();
      for (final IFDeployExtension ext : _extensions) {
        Collection<IFDeployExtension.TypeDef> _types = ext.getTypes();
        for (final IFDeployExtension.TypeDef type : _types) {
          result.put(type, ext);
        }
      }
      _xblockexpression = result;
    }
    return _xblockexpression;
  }
  
  public static IFDeployExtension.TypeDef findType(final String typeName) {
    IFDeployExtension.TypeDef _xblockexpression = null;
    {
      final Function1<IFDeployExtension, Collection<IFDeployExtension.TypeDef>> _function = new Function1<IFDeployExtension, Collection<IFDeployExtension.TypeDef>>() {
        @Override
        public Collection<IFDeployExtension.TypeDef> apply(final IFDeployExtension it) {
          return it.getTypes();
        }
      };
      final Iterable<IFDeployExtension.TypeDef> types = Iterables.<IFDeployExtension.TypeDef>concat(IterableExtensions.<IFDeployExtension, Collection<IFDeployExtension.TypeDef>>map(ExtensionRegistry.getExtensions(), _function));
      final Function1<IFDeployExtension.TypeDef, Boolean> _function_1 = new Function1<IFDeployExtension.TypeDef, Boolean>() {
        @Override
        public Boolean apply(final IFDeployExtension.TypeDef it) {
          String _name = it.getName();
          return Boolean.valueOf(Objects.equal(_name, typeName));
        }
      };
      _xblockexpression = IterableExtensions.<IFDeployExtension.TypeDef>findFirst(types, _function_1);
    }
    return _xblockexpression;
  }
}
