/**
 * Copyright (c) 2018 itemis AG (http://www.itemis.de).
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.franca.deploymodel.extensions;

import com.google.common.base.Objects;
import java.util.Collection;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.xtext.scoping.IScope;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.franca.deploymodel.dsl.fDeploy.FDElement;
import org.franca.deploymodel.dsl.fDeploy.FDValue;

/**
 * Extension point which can be used to specify deployment extensions.</p>
 * 
 * @author Klaus Birken (itemis AG)
 */
@SuppressWarnings("all")
public interface IFDeployExtension {
  /**
   * Definition of a new deployment host for this extension.</p>
   */
  public static class Host {
    private String name;
    
    public Host(final String name) {
      this.name = name;
    }
    
    public String getName() {
      return this.name;
    }
    
    @Override
    public String toString() {
      return (("" + this.name) + "");
    }
  }
  
  /**
   * A common base class for deployment definition roots and elements.</p>
   */
  public static abstract class AbstractElementDef {
    public enum Nameable {
      NO_NAME,
      
      OPTIONAL_NAME,
      
      MANDATORY_NAME;
    }
    
    private String tag;
    
    private EClass targetClass;
    
    private IFDeployExtension.AbstractElementDef.Nameable isNameable;
    
    private Collection<IFDeployExtension.Host> hosts;
    
    private Collection<IFDeployExtension.ElementDef> children;
    
    public AbstractElementDef(final String tag, final IFDeployExtension.AbstractElementDef.Nameable isNameable, final Collection<IFDeployExtension.Host> hosts) {
      this(tag, null, isNameable, hosts);
    }
    
    public AbstractElementDef(final String tag, final EClass targetClass, final IFDeployExtension.AbstractElementDef.Nameable isNameable, final Collection<IFDeployExtension.Host> hosts) {
      this.tag = tag;
      this.targetClass = targetClass;
      this.isNameable = isNameable;
      this.hosts = hosts;
      this.children = CollectionLiterals.<IFDeployExtension.ElementDef>newArrayList();
    }
    
    public void addChild(final IFDeployExtension.ElementDef child) {
      this.children.add(child);
      child.setParent(this);
    }
    
    public String getTag() {
      return this.tag;
    }
    
    public boolean mayHaveName() {
      return (this.isNameable != IFDeployExtension.AbstractElementDef.Nameable.NO_NAME);
    }
    
    public boolean mustHaveName() {
      return (this.isNameable == IFDeployExtension.AbstractElementDef.Nameable.MANDATORY_NAME);
    }
    
    public EClass getTargetClass() {
      return this.targetClass;
    }
    
    public Collection<IFDeployExtension.Host> getHosts() {
      return this.hosts;
    }
    
    public Collection<IFDeployExtension.ElementDef> getChildren() {
      return this.children;
    }
  }
  
  /**
   * Descriptor of a new root element for deployment definitions.</p>
   */
  public static class RootDef extends IFDeployExtension.AbstractElementDef {
    private IFDeployExtension extension;
    
    public RootDef(final IFDeployExtension extension, final String tag, final IFDeployExtension.AbstractElementDef.Nameable isNameable, final Collection<IFDeployExtension.Host> hosts) {
      this(extension, tag, null, isNameable, hosts);
    }
    
    public RootDef(final IFDeployExtension extension, final String tag, final EClass targetClass, final IFDeployExtension.AbstractElementDef.Nameable isNameable, final Collection<IFDeployExtension.Host> hosts) {
      super(tag, targetClass, isNameable, hosts);
      this.extension = extension;
    }
    
    public IFDeployExtension getExtension() {
      return this.extension;
    }
  }
  
  /**
   * Descriptor of a new child element for deployment definitions.</p>
   */
  public static class ElementDef extends IFDeployExtension.AbstractElementDef {
    private IFDeployExtension.AbstractElementDef parent;
    
    public ElementDef(final String tag, final IFDeployExtension.AbstractElementDef.Nameable isNameable, final Collection<IFDeployExtension.Host> hosts) {
      super(tag, isNameable, hosts);
    }
    
    public ElementDef(final String tag, final EClass targetClass, final IFDeployExtension.AbstractElementDef.Nameable isNameable, final Collection<IFDeployExtension.Host> hosts) {
      super(tag, targetClass, isNameable, hosts);
    }
    
    public void setParent(final IFDeployExtension.AbstractElementDef parent) {
      this.parent = parent;
    }
  }
  
  public static class HostMixinDef {
    public enum AccessorArgumentStyle {
      BY_RULE_CLASS,
      
      BY_TARGET_FEATURE;
    }
    
    public final static String CHILD_ELEMENT = "$CHILD_ELEMENT$";
    
    private EClass clazz;
    
    private IFDeployExtension.HostMixinDef.AccessorArgumentStyle argumentStyle;
    
    private String accessorPrefix;
    
    private Collection<IFDeployExtension.Host> hosts;
    
    /**
     * Constructor for mixin which provides additional hosts for Franca IDL concepts.</p>
     * 
     * Their properties will be added to existing IDataPropertyAccessor classes.</p>
     */
    public HostMixinDef(final EClass clazz, final IFDeployExtension.HostMixinDef.AccessorArgumentStyle argumentStyle, final Collection<IFDeployExtension.Host> hosts) {
      this(clazz, argumentStyle, null, hosts);
    }
    
    /**
     * Constructor for mixin which provides additional hosts for non-Franca concepts.</p>
     * 
     * These mixins will get own PropertyAccessor classes (named <em>accessorPrefix</em>PropertyAccessor).</p>
     * 
     * @param accessorPrefix prefix for property accessor class name, or CHILD_ELEMENT if this mixin
     *                       should be added to other property accessor based on class hierarchy
     */
    public HostMixinDef(final EClass clazz, final IFDeployExtension.HostMixinDef.AccessorArgumentStyle argumentStyle, final String accessorPrefix, final Collection<IFDeployExtension.Host> hosts) {
      this.clazz = clazz;
      this.argumentStyle = argumentStyle;
      this.accessorPrefix = accessorPrefix;
      this.hosts = hosts;
    }
    
    public EClass getHostingClass() {
      return this.clazz;
    }
    
    public IFDeployExtension.HostMixinDef.AccessorArgumentStyle getAccessorArgument() {
      return this.argumentStyle;
    }
    
    public Collection<IFDeployExtension.Host> getHosts() {
      return this.hosts;
    }
    
    public boolean isNonFrancaMixin() {
      return (this.accessorPrefix != null);
    }
    
    public String getAccessorRootPrefix() {
      String _xifexpression = null;
      if (((this.accessorPrefix == null) || Objects.equal(this.accessorPrefix, IFDeployExtension.HostMixinDef.CHILD_ELEMENT))) {
        _xifexpression = null;
      } else {
        _xifexpression = this.accessorPrefix;
      }
      return _xifexpression;
    }
    
    public boolean isChildMixin() {
      return Objects.equal(this.accessorPrefix, IFDeployExtension.HostMixinDef.CHILD_ELEMENT);
    }
  }
  
  /**
   * Descriptor for a new deployment property type.</p>
   * 
   * Use this to define additional types of properties, extending the built-in
   * type system of the deployment DSL (which supports Integer, String, Boolean, etc.).</p>
   */
  public static class TypeDef {
    private String name;
    
    private Function1<? super IScope, ? extends IScope> scopeFunc;
    
    private Function1<? super FDElement, ? extends FDValue> defaultCreator;
    
    private Class<? extends EObject> runtimeType;
    
    /**
     * Define a new property type for the deployment DSL.</p>
     * 
     * Use this to define additional types of properties, extending the built-in
     * type system of the deployment DSL (which supports Integer, String, Boolean, etc.).</p>
     * 
     * @param name the type name used by the concrete syntax of deployment specifications
     * @param scopeFunc implementation of scoping for the new type
     * @param defaultCreator a function which provides a default value (based on a given FDElement context)
     * @param runtimeType the Java class which will be used to represent this type in the
     *        generated PropertyAccessor class
     */
    public TypeDef(final String name, final Function1<? super IScope, ? extends IScope> scopeFunc, final Function1<? super FDElement, ? extends FDValue> defaultCreator, final Class<? extends EObject> runtimeType) {
      this.name = name;
      this.scopeFunc = scopeFunc;
      this.defaultCreator = defaultCreator;
      this.runtimeType = runtimeType;
    }
    
    public String getName() {
      return this.name;
    }
    
    public IScope getScope(final IScope all) {
      IScope _xifexpression = null;
      if ((this.scopeFunc == null)) {
        _xifexpression = IScope.NULLSCOPE;
      } else {
        _xifexpression = this.scopeFunc.apply(all);
      }
      return _xifexpression;
    }
    
    public FDValue createDefaultValue(final FDElement element) {
      FDValue _apply = null;
      if (this.defaultCreator!=null) {
        _apply=this.defaultCreator.apply(element);
      }
      return _apply;
    }
    
    public Class<? extends EObject> getRuntimeType() {
      return this.runtimeType;
    }
  }
  
  /**
   * Short description of this extension to be used by the IDE's user interface.</p>
   */
  public abstract String getShortDescription();
  
  /**
   * Called by the framework to get the list of new root elements
   * provided by this deployment definition extension.</p>
   */
  public abstract Collection<IFDeployExtension.RootDef> getRoots();
  
  /**
   * Called by the framework to get the list of new mixin elements
   * provided by this deployment extension.</p>
   */
  public abstract Collection<IFDeployExtension.HostMixinDef> getMixins();
  
  /**
   * Called by the framework to get the list of additional property types
   * provided by this deployment extension.</p>
   */
  public abstract Collection<IFDeployExtension.TypeDef> getTypes();
}
