/*******************************************************************************
* Copyright (c) 2015 itemis AG (http://www.itemis.de).
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
*******************************************************************************/
package org.franca.deploymodel.dsl.generator

import com.google.inject.Inject
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.resource.Resource
import org.eclipse.xtext.generator.IFileSystemAccess
import org.eclipse.xtext.generator.IFileSystemAccess2
import org.eclipse.xtext.generator.IGenerator2
import org.eclipse.xtext.generator.IGeneratorContext
import org.franca.deploymodel.core.FDPropertyHost
import org.franca.deploymodel.dsl.fDeploy.FDEnumType
import org.franca.deploymodel.dsl.fDeploy.FDExtensionRoot
import org.franca.deploymodel.dsl.fDeploy.FDPropertyDecl
import org.franca.deploymodel.dsl.fDeploy.FDSpecification
import org.franca.deploymodel.dsl.generator.internal.HelperGenerator
import org.franca.deploymodel.dsl.generator.internal.IDataGenerator
import org.franca.deploymodel.dsl.generator.internal.ImportManager
import org.franca.deploymodel.dsl.generator.internal.InterfaceAccessorGenerator
import org.franca.deploymodel.dsl.generator.internal.OverwriteAccessorGenerator
import org.franca.deploymodel.dsl.generator.internal.RootElementAccessorGenerator
import org.franca.deploymodel.dsl.generator.internal.TypeCollectionAccessorGenerator

import static org.franca.deploymodel.extensions.ExtensionRegistry.*

import static extension org.franca.deploymodel.dsl.generator.internal.GeneratorHelper.*
import static extension org.franca.deploymodel.dsl.generator.internal.HostLogic.*

/**
 * Generator for PropertyAccessor class from deployment specification.
 * The deployment specification is defined as fdepl model (spec part).
 * 
 * The accessor classes generated by this generator will be useful when
 * traversing a fidl model and getting the deployment properties for this
 * model.
 */
class FDeployGenerator implements IGenerator2 {
	
	@Inject extension ImportManager
	
	@Inject IDataGenerator genInterface
	@Inject HelperGenerator genHelper
	@Inject TypeCollectionAccessorGenerator genTCAcc
	@Inject InterfaceAccessorGenerator genInterfaceAcc
	@Inject RootElementAccessorGenerator genRootElementAcc
	@Inject OverwriteAccessorGenerator genOverwriteAcc
	
	// the types of PropertyAccessor classes we can generate
	final static String PA_INTERFACE = "Interface"
	final static String PA_TYPE_COLLECTION = "TypeCollection"
	
	override beforeGenerate(Resource input, IFileSystemAccess2 fsa, IGeneratorContext context) {
	}

	// the main function for this generator, will be called by Xtend framework
	override void doGenerate(Resource resource, IFileSystemAccess2 fsa, IGeneratorContext context) {
		for(m : resource.allContents.toIterable.filter(typeof(FDSpecification))) {
			// generate compound accessor class with several nested classes
			fsa.generateAll(m)
			
			// generate some legacy classes for backward-compatibility
			// (this is needed for Franca 0.9.1 and earlier)
			fsa.generateLegacy(m, PA_INTERFACE)
			fsa.generateLegacy(m, PA_TYPE_COLLECTION)
		}
	}

	override afterGenerate(Resource input, IFileSystemAccess2 fsa, IGeneratorContext context) {
	}
	
	
	// *****************************************************************************
	// top-level generation and analysis
	
	def generateAll (IFileSystemAccess fsa, FDSpecification spec) {
		initImportManager

		// generate class code and analyse for needed preliminaries
		val path = spec.getPackage().replace(".", "/")
		val code = spec.generateCombinedClass.toString
		var header = spec.generateHeader(null).toString
		
		fsa.generateFile(path + "/" + spec.classname + ".java", header + code)
	}

	
	def private generateCombinedClass(FDSpecification spec) '''
		/**
		 * This is a collection of all interfaces and classes needed for
		 * accessing deployment properties according to deployment specification
		 * 'spec.name'.
		 */
		public class spec.classname {

			genEnumInterface(spec)

			genInterface.generate(spec)

			genHelper.generate(spec)

			genTCAcc.generate(spec)

			genInterfaceAcc.generate(spec)

			FOR root : roots.keySet
				genRootElementAcc.generate(spec,
					FDExtensionRoot,
					root.tag,
					root.extension.shortDescription,
					[isHostFor(root)]
				)
				
			ENDFOR
			FOR clazz : nonFrancaMixinRoots
				val prefix = getNonFrancaMixinPrefix(clazz)
				genRootElementAcc.generate(spec,
					clazz.instanceClass as Class<? extends EObject>,
					prefix,
					getMixinExtension(clazz).shortDescription,
					[isHostFor(clazz)]
				)
				
			ENDFOR
			genOverwriteAcc.generate(spec)
		}
			
	'''


	def private genEnumInterface(FDSpecification spec) '''
		/**
		 * Enumerations for deployment specification spec.name.
		 */
		public interface Enums
			IF spec.base!==nullextends spec.base.qualifiedClassname.EnumsENDIF
		{
			FOR d : spec.declarations
				FOR p : d.properties
					p.genStaticEnum(d.host)
				ENDFOR
			ENDFOR
		}
	'''

	def private genStaticEnum(FDPropertyDecl it, FDPropertyHost host) {
		if (isEnum) {
			val enumType = name.toFirstUpper
			val enumerator = type.complex as FDEnumType
			'''
				public enum enumType {
					FOR e : enumerator.enumerators SEPARATOR ", "e.nameENDFOR
				}
				 
			'''
		} else
			''
	}
	

	/**
	 * Generate legacy classes needed for users of Franca 0.9.1 and earlier.
	 */
	def generateLegacy (IFileSystemAccess fsa, FDSpecification spec, String type) {
		// generate class code and analyse for needed preliminaries
		val path = spec.getPackage().replace(".", "/")
		val code = spec.generateLegacyClass(type).toString
		var header = spec.generateHeader(type).toString
		
		fsa.generateFile(path + "/" + spec.getLegacyClassname(type) + ".java", header + code)
	}
	
	
	def generateHeader (FDSpecification spec, String type) '''
		/*******************************************************************************
		* This file has been generated by Franca's FDeployGenerator.
		* Source: deployment specification 'spec.name'
		*******************************************************************************/
		IF ! spec.getPackage.empty
		package spec.getPackage;
		
		ENDIF
		IF type!==null
		import org.franca.deploymodel.core.getSupportingClass(type);
		ELSE
		genImports
		ENDIF

	'''


	def generateLegacyClass (FDSpecification spec, String type) '''
		/**
		 * Accessor for deployment properties for 'spec.name' specification
		 *
		 * @deprecated use class spec.qualifiedClassname.typePropertyAccessor instead
		 */
		public class spec.getLegacyClassname(type)
			extends spec.qualifiedClassname.typePropertyAccessor
			implements spec.qualifiedClassname.Enums
		{
			public spec.getLegacyClassname(type)(getSupportingClass(type) target) {
				super(target);
			}
		}
		   
	'''


	// *****************************************************************************
	// basic helpers
	
	def private getLegacyClassname (FDSpecification it, String type) {
		val sep = name.lastIndexOf(".")
		val basename = if (sep>0) name.substring(sep+1) else name
		basename.toFirstUpper + type + "PropertyAccessor"
	}

	def private getSupportingClass(String type) {
		switch (type) {
			case PA_TYPE_COLLECTION: "FDeployedTypeCollection"
			case PA_INTERFACE: "FDeployedInterface"
			default: ""
		}
	}
	
}
