package org.freedesktop.wayland.client;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
// 
//
//    Copyright © 2008-2011 Kristian Høgsberg
//    Copyright © 2010-2011 Intel Corporation
//    Copyright © 2012-2013 Collabora, Ltd.
//
//    Permission is hereby granted, free of charge, to any person
//    obtaining a copy of this software and associated documentation files
//    (the "Software"), to deal in the Software without restriction,
//    including without limitation the rights to use, copy, modify, merge,
//    publish, distribute, sublicense, and/or sell copies of the Software,
//    and to permit persons to whom the Software is furnished to do so,
//    subject to the following conditions:
//
//    The above copyright notice and this permission notice (including the
//    next paragraph) shall be included in all copies or substantial
//    portions of the Software.
//
//    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
//    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
//    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
//    NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
//    BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
//    ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
//    CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//    SOFTWARE.
//  
/**
 * data transfer device
 * <p>
 *
 *       There is one wl_data_device per seat which can be obtained
 *       from the global wl_data_device_manager singleton.
 * <p>
 *       A wl_data_device provides access to inter-client data transfer
 *       mechanisms such as copy-and-paste and drag-and-drop.
 *     
 */
public interface WlDataDeviceEventsV2 extends WlDataDeviceEvents {
  int VERSION = 2;

  /**
   * introduce a new wl_data_offer
   * <p>
   *
   * 	The data_offer event introduces a new wl_data_offer object,
   * 	which will subsequently be used in either the
   * 	data_device.enter event (for drag-and-drop) or the
   * 	data_device.selection event (for selections).  Immediately
   * 	following the data_device_data_offer event, the new data_offer
   * 	object will send out data_offer.offer events to describe the
   * 	mime types it offers.
   *       
   * @param emitter The protocol object that emitted the event.
   * @param id the new data_offer object
   */
  public void dataOffer(WlDataDeviceProxy emitter, @Nonnull WlDataOfferProxy id);

  /**
   * initiate drag-and-drop session
   * <p>
   *
   * 	This event is sent when an active drag-and-drop pointer enters
   * 	a surface owned by the client.  The position of the pointer at
   * 	enter time is provided by the x and y arguments, in surface-local
   * 	coordinates.
   *       
   * @param emitter The protocol object that emitted the event.
   * @param serial serial number of the enter event
   * @param surface client surface entered
   * @param x surface-local x coordinate
   * @param y surface-local y coordinate
   * @param id source data_offer object
   */
  public void enter(WlDataDeviceProxy emitter, int serial, @Nonnull WlSurfaceProxy surface, @Nonnull org.freedesktop.wayland.util.Fixed x, @Nonnull org.freedesktop.wayland.util.Fixed y, @Nullable WlDataOfferProxy id);

  /**
   * end drag-and-drop session
   * <p>
   *
   * 	This event is sent when the drag-and-drop pointer leaves the
   * 	surface and the session ends.  The client must destroy the
   * 	wl_data_offer introduced at enter time at this point.
   *       
   * @param emitter The protocol object that emitted the event.
   */
  public void leave(WlDataDeviceProxy emitter);

  /**
   * drag-and-drop session motion
   * <p>
   *
   * 	This event is sent when the drag-and-drop pointer moves within
   * 	the currently focused surface. The new position of the pointer
   * 	is provided by the x and y arguments, in surface-local
   * 	coordinates.
   *       
   * @param emitter The protocol object that emitted the event.
   * @param time timestamp with millisecond granularity
   * @param x surface-local x coordinate
   * @param y surface-local y coordinate
   */
  public void motion(WlDataDeviceProxy emitter, int time, @Nonnull org.freedesktop.wayland.util.Fixed x, @Nonnull org.freedesktop.wayland.util.Fixed y);

  /**
   * end drag-and-drop session successfully
   * <p>
   *
   * 	The event is sent when a drag-and-drop operation is ended
   * 	because the implicit grab is removed.
   * <p>
   * 	The drag-and-drop destination is expected to honor the last action
   * 	received through wl_data_offer.action, if the resulting action is
   * 	"copy" or "move", the destination can still perform
   * 	wl_data_offer.receive requests, and is expected to end all
   * 	transfers with a wl_data_offer.finish request.
   * <p>
   * 	If the resulting action is "ask", the action will not be considered
   * 	final. The drag-and-drop destination is expected to perform one last
   * 	wl_data_offer.set_actions request, or wl_data_offer.destroy in order
   * 	to cancel the operation.
   *       
   * @param emitter The protocol object that emitted the event.
   */
  public void drop(WlDataDeviceProxy emitter);

  /**
   * advertise new selection
   * <p>
   *
   * 	The selection event is sent out to notify the client of a new
   * 	wl_data_offer for the selection for this device.  The
   * 	data_device.data_offer and the data_offer.offer events are
   * 	sent out immediately before this event to introduce the data
   * 	offer object.  The selection event is sent to a client
   * 	immediately before receiving keyboard focus and when a new
   * 	selection is set while the client has keyboard focus.  The
   * 	data_offer is valid until a new data_offer or NULL is received
   * 	or until the client loses keyboard focus.  The client must
   * 	destroy the previous selection data_offer, if any, upon receiving
   * 	this event.
   *       
   * @param emitter The protocol object that emitted the event.
   * @param id selection data_offer object
   */
  public void selection(WlDataDeviceProxy emitter, @Nullable WlDataOfferProxy id);
}
