package org.freedesktop.wayland.client;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.freedesktop.wayland.util.Arguments;
import org.freedesktop.wayland.util.Interface;
import org.freedesktop.wayland.util.Message;
// 
//
//    Copyright © 2008-2011 Kristian Høgsberg
//    Copyright © 2010-2011 Intel Corporation
//    Copyright © 2012-2013 Collabora, Ltd.
//
//    Permission is hereby granted, free of charge, to any person
//    obtaining a copy of this software and associated documentation files
//    (the "Software"), to deal in the Software without restriction,
//    including without limitation the rights to use, copy, modify, merge,
//    publish, distribute, sublicense, and/or sell copies of the Software,
//    and to permit persons to whom the Software is furnished to do so,
//    subject to the following conditions:
//
//    The above copyright notice and this permission notice (including the
//    next paragraph) shall be included in all copies or substantial
//    portions of the Software.
//
//    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
//    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
//    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
//    NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
//    BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
//    ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
//    CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//    SOFTWARE.
//  
/**
 * data transfer device
 * <p>
 *
 *       There is one wl_data_device per seat which can be obtained
 *       from the global wl_data_device_manager singleton.
 * <p>
 *       A wl_data_device provides access to inter-client data transfer
 *       mechanisms such as copy-and-paste and drag-and-drop.
 *     
 */
@Interface(
  methods = {
    @Message(
  types = {
    org.freedesktop.wayland.client.WlDataSourceProxy.class,
    org.freedesktop.wayland.client.WlSurfaceProxy.class,
    org.freedesktop.wayland.client.WlSurfaceProxy.class,
    int.class
  },
  signature = "?oo?ou",
  functionName = "startDrag",
  name = "start_drag"
)
,
    @Message(
  types = {
    org.freedesktop.wayland.client.WlDataSourceProxy.class,
    int.class
  },
  signature = "?ou",
  functionName = "setSelection",
  name = "set_selection"
)
,
    @Message(
  types = {
  },
  signature = "2",
  functionName = "release",
  name = "release"
)

  },
  name = "wl_data_device",
  version = 3,
  events = {
    @Message(
  types = {
    org.freedesktop.wayland.client.WlDataOfferProxy.class
  },
  signature = "n",
  functionName = "dataOffer",
  name = "data_offer"
)
,
    @Message(
  types = {
    int.class,
    org.freedesktop.wayland.client.WlSurfaceProxy.class,
    org.freedesktop.wayland.util.Fixed.class,
    org.freedesktop.wayland.util.Fixed.class,
    org.freedesktop.wayland.client.WlDataOfferProxy.class
  },
  signature = "uoff?o",
  functionName = "enter",
  name = "enter"
)
,
    @Message(
  types = {
  },
  signature = "",
  functionName = "leave",
  name = "leave"
)
,
    @Message(
  types = {
    int.class,
    org.freedesktop.wayland.util.Fixed.class,
    org.freedesktop.wayland.util.Fixed.class
  },
  signature = "uff",
  functionName = "motion",
  name = "motion"
)
,
    @Message(
  types = {
  },
  signature = "",
  functionName = "drop",
  name = "drop"
)
,
    @Message(
  types = {
    org.freedesktop.wayland.client.WlDataOfferProxy.class
  },
  signature = "?o",
  functionName = "selection",
  name = "selection"
)

  }
)
public class WlDataDeviceProxy extends Proxy<WlDataDeviceEvents> {

  public static final String INTERFACE_NAME = "wl_data_device";

  public WlDataDeviceProxy(long pointer, WlDataDeviceEvents implementation, int version) {
    super(pointer, implementation, version);
  }

  public WlDataDeviceProxy(long pointer) {
    super(pointer);
  }

  /**
   * start drag-and-drop operation
   * <p>
   *
   * 	This request asks the compositor to start a drag-and-drop
   * 	operation on behalf of the client.
   * <p>
   * 	The source argument is the data source that provides the data
   * 	for the eventual data transfer. If source is NULL, enter, leave
   * 	and motion events are sent only to the client that initiated the
   * 	drag and the client is expected to handle the data passing
   * 	internally.
   * <p>
   * 	The origin surface is the surface where the drag originates and
   * 	the client must have an active implicit grab that matches the
   * 	serial.
   * <p>
   * 	The icon surface is an optional (can be NULL) surface that
   * 	provides an icon to be moved around with the cursor.  Initially,
   * 	the top-left corner of the icon surface is placed at the cursor
   * 	hotspot, but subsequent wl_surface.attach request can move the
   * 	relative position. Attach requests must be confirmed with
   * 	wl_surface.commit as usual. The icon surface is given the role of
   * 	a drag-and-drop icon. If the icon surface already has another role,
   * 	it raises a protocol error.
   * <p>
   * 	The current and pending input regions of the icon wl_surface are
   * 	cleared, and wl_surface.set_input_region is ignored until the
   * 	wl_surface is no longer used as the icon surface. When the use
   * 	as an icon ends, the current and pending input regions become
   * 	undefined, and the wl_surface is unmapped.
   *       
   * @param source data source for the eventual transfer
   * @param origin surface where the drag originates
   * @param icon drag-and-drop icon surface
   * @param serial serial number of the implicit grab on the origin
   */
  public void startDrag(@Nullable WlDataSourceProxy source, @Nonnull WlSurfaceProxy origin, @Nullable WlSurfaceProxy icon, int serial) {
    marshal(0, Arguments.create(4).set(0, source).set(1, origin).set(2, icon).set(3, serial));
  }

  /**
   * copy data to the selection
   * <p>
   *
   * 	This request asks the compositor to set the selection
   * 	to the data from the source on behalf of the client.
   * <p>
   * 	To unset the selection, set the source to NULL.
   *       
   * @param source data source for the selection
   * @param serial serial number of the event that triggered this request
   */
  public void setSelection(@Nullable WlDataSourceProxy source, int serial) {
    marshal(1, Arguments.create(2).set(0, source).set(1, serial));
  }

  /**
   * destroy data device
   * <p>
   *
   * 	This request destroys the data device.
   *       
   */
  public void release() {
    if (getVersion() < 2) {
      throw new UnsupportedOperationException("This object is version "+getVersion()+" while version 2 is required for this operation.");
    }
    marshal(2);
  }
}
