package org.freedesktop.wayland.client;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
// 
//
//    Copyright © 2008-2011 Kristian Høgsberg
//    Copyright © 2010-2011 Intel Corporation
//    Copyright © 2012-2013 Collabora, Ltd.
//
//    Permission is hereby granted, free of charge, to any person
//    obtaining a copy of this software and associated documentation files
//    (the "Software"), to deal in the Software without restriction,
//    including without limitation the rights to use, copy, modify, merge,
//    publish, distribute, sublicense, and/or sell copies of the Software,
//    and to permit persons to whom the Software is furnished to do so,
//    subject to the following conditions:
//
//    The above copyright notice and this permission notice (including the
//    next paragraph) shall be included in all copies or substantial
//    portions of the Software.
//
//    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
//    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
//    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
//    NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
//    BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
//    ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
//    CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//    SOFTWARE.
//  
/**
 * offer to transfer data
 * <p>
 *
 *       A wl_data_offer represents a piece of data offered for transfer
 *       by another client (the source client).  It is used by the
 *       copy-and-paste and drag-and-drop mechanisms.  The offer
 *       describes the different mime types that the data can be
 *       converted to and provides the mechanism for transferring the
 *       data directly from the source client.
 *     
 */
public interface WlDataOfferEventsV3 extends WlDataOfferEventsV2 {
  int VERSION = 3;

  /**
   * advertise offered mime type
   * <p>
   *
   * 	Sent immediately after creating the wl_data_offer object.  One
   * 	event per offered mime type.
   *       
   * @param emitter The protocol object that emitted the event.
   * @param mimeType offered mime type
   */
  public void offer(WlDataOfferProxy emitter, @Nonnull String mimeType);

  /**
   * notify the source-side available actions
   * <p>
   *
   * 	This event indicates the actions offered by the data source. It
   * 	will be sent right after wl_data_device.enter, or anytime the source
   * 	side changes its offered actions through wl_data_source.set_actions.
   *       
   * @param emitter The protocol object that emitted the event.
   * @param sourceActions actions offered by the data source
   */
  public void sourceActions(WlDataOfferProxy emitter, int sourceActions);

  /**
   * notify the selected action
   * <p>
   *
   * 	This event indicates the action selected by the compositor after
   * 	matching the source/destination side actions. Only one action (or
   * 	none) will be offered here.
   * <p>
   * 	This event can be emitted multiple times during the drag-and-drop
   * 	operation in response to destination side action changes through
   * 	wl_data_offer.set_actions.
   * <p>
   * 	This event will no longer be emitted after wl_data_device.drop
   * 	happened on the drag-and-drop destination, the client must
   * 	honor the last action received, or the last preferred one set
   * 	through wl_data_offer.set_actions when handling an "ask" action.
   * <p>
   * 	Compositors may also change the selected action on the fly, mainly
   * 	in response to keyboard modifier changes during the drag-and-drop
   * 	operation.
   * <p>
   * 	The most recent action received is always the valid one. Prior to
   * 	receiving wl_data_device.drop, the chosen action may change (e.g.
   * 	due to keyboard modifiers being pressed). At the time of receiving
   * 	wl_data_device.drop the drag-and-drop destination must honor the
   * 	last action received.
   * <p>
   * 	Action changes may still happen after wl_data_device.drop,
   * 	especially on "ask" actions, where the drag-and-drop destination
   * 	may choose another action afterwards. Action changes happening
   * 	at this stage are always the result of inter-client negotiation, the
   * 	compositor shall no longer be able to induce a different action.
   * <p>
   * 	Upon "ask" actions, it is expected that the drag-and-drop destination
   * 	may potentially choose a different action and/or mime type,
   * 	based on wl_data_offer.source_actions and finally chosen by the
   * 	user (e.g. popping up a menu with the available options). The
   * 	final wl_data_offer.set_actions and wl_data_offer.accept requests
   * 	must happen before the call to wl_data_offer.finish.
   *       
   * @param emitter The protocol object that emitted the event.
   * @param dndAction action selected by the compositor
   */
  public void action(WlDataOfferProxy emitter, int dndAction);
}
