package org.freedesktop.wayland.client;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.freedesktop.wayland.util.Arguments;
import org.freedesktop.wayland.util.Interface;
import org.freedesktop.wayland.util.Message;
// 
//
//    Copyright © 2008-2011 Kristian Høgsberg
//    Copyright © 2010-2011 Intel Corporation
//    Copyright © 2012-2013 Collabora, Ltd.
//
//    Permission is hereby granted, free of charge, to any person
//    obtaining a copy of this software and associated documentation files
//    (the "Software"), to deal in the Software without restriction,
//    including without limitation the rights to use, copy, modify, merge,
//    publish, distribute, sublicense, and/or sell copies of the Software,
//    and to permit persons to whom the Software is furnished to do so,
//    subject to the following conditions:
//
//    The above copyright notice and this permission notice (including the
//    next paragraph) shall be included in all copies or substantial
//    portions of the Software.
//
//    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
//    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
//    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
//    NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
//    BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
//    ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
//    CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//    SOFTWARE.
//  
/**
 * offer to transfer data
 * <p>
 *
 *       The wl_data_source object is the source side of a wl_data_offer.
 *       It is created by the source client in a data transfer and
 *       provides a way to describe the offered data and a way to respond
 *       to requests to transfer the data.
 *     
 */
@Interface(
  methods = {
    @Message(
  types = {
    java.lang.String.class
  },
  signature = "s",
  functionName = "offer",
  name = "offer"
)
,
    @Message(
  types = {
  },
  signature = "",
  functionName = "destroy",
  name = "destroy"
)
,
    @Message(
  types = {
    int.class
  },
  signature = "3u",
  functionName = "setActions",
  name = "set_actions"
)

  },
  name = "wl_data_source",
  version = 3,
  events = {
    @Message(
  types = {
    java.lang.String.class
  },
  signature = "?s",
  functionName = "target",
  name = "target"
)
,
    @Message(
  types = {
    java.lang.String.class,
    int.class
  },
  signature = "sh",
  functionName = "send",
  name = "send"
)
,
    @Message(
  types = {
  },
  signature = "",
  functionName = "cancelled",
  name = "cancelled"
)
,
    @Message(
  types = {
  },
  signature = "3",
  functionName = "dndDropPerformed",
  name = "dnd_drop_performed"
)
,
    @Message(
  types = {
  },
  signature = "3",
  functionName = "dndFinished",
  name = "dnd_finished"
)
,
    @Message(
  types = {
    int.class
  },
  signature = "3u",
  functionName = "action",
  name = "action"
)

  }
)
public class WlDataSourceProxy extends Proxy<WlDataSourceEvents> {

  public static final String INTERFACE_NAME = "wl_data_source";

  public WlDataSourceProxy(long pointer, WlDataSourceEvents implementation, int version) {
    super(pointer, implementation, version);
  }

  public WlDataSourceProxy(long pointer) {
    super(pointer);
  }

  /**
   * add an offered mime type
   * <p>
   *
   * 	This request adds a mime type to the set of mime types
   * 	advertised to targets.  Can be called several times to offer
   * 	multiple types.
   *       
   * @param mimeType mime type offered by the data source
   */
  public void offer(@Nonnull String mimeType) {
    marshal(0, Arguments.create(1).set(0, mimeType));
  }

  /**
   * destroy the data source
   * <p>
   *
   * 	Destroy the data source.
   *       
   */
  public void destroy() {
    marshal(1);
  }

  /**
   * set the available drag-and-drop actions
   * <p>
   *
   * 	Sets the actions that the source side client supports for this
   * 	operation. This request may trigger wl_data_source.action and
   * 	wl_data_offer.action events if the compositor needs to change the
   * 	selected action.
   * <p>
   * 	The dnd_actions argument must contain only values expressed in the
   * 	wl_data_device_manager.dnd_actions enum, otherwise it will result
   * 	in a protocol error.
   * <p>
   * 	This request must be made once only, and can only be made on sources
   * 	used in drag-and-drop, so it must be performed before
   * 	wl_data_device.start_drag. Attempting to use the source other than
   * 	for drag-and-drop will raise a protocol error.
   *       
   * @param dndActions actions supported by the data source
   */
  public void setActions(int dndActions) {
    if (getVersion() < 3) {
      throw new UnsupportedOperationException("This object is version "+getVersion()+" while version 3 is required for this operation.");
    }
    marshal(2, Arguments.create(1).set(0, dndActions));
  }
}
