package org.freedesktop.wayland.client;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
// 
//
//    Copyright © 2008-2011 Kristian Høgsberg
//    Copyright © 2010-2011 Intel Corporation
//    Copyright © 2012-2013 Collabora, Ltd.
//
//    Permission is hereby granted, free of charge, to any person
//    obtaining a copy of this software and associated documentation files
//    (the "Software"), to deal in the Software without restriction,
//    including without limitation the rights to use, copy, modify, merge,
//    publish, distribute, sublicense, and/or sell copies of the Software,
//    and to permit persons to whom the Software is furnished to do so,
//    subject to the following conditions:
//
//    The above copyright notice and this permission notice (including the
//    next paragraph) shall be included in all copies or substantial
//    portions of the Software.
//
//    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
//    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
//    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
//    NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
//    BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
//    ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
//    CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//    SOFTWARE.
//  
/**
 * pointer input device
 * <p>
 *
 *       The wl_pointer interface represents one or more input devices,
 *       such as mice, which control the pointer location and pointer_focus
 *       of a seat.
 * <p>
 *       The wl_pointer interface generates motion, enter and leave
 *       events for the surfaces that the pointer is located over,
 *       and button and axis events for button presses, button releases
 *       and scrolling.
 *     
 */
public interface WlPointerEventsV4 extends WlPointerEventsV3 {
  int VERSION = 4;

  /**
   * enter event
   * <p>
   *
   * 	Notification that this seat's pointer is focused on a certain
   * 	surface.
   * <p>
   * 	When a seat's focus enters a surface, the pointer image
   * 	is undefined and a client should respond to this event by setting
   * 	an appropriate pointer image with the set_cursor request.
   *       
   * @param emitter The protocol object that emitted the event.
   * @param serial serial number of the enter event
   * @param surface surface entered by the pointer
   * @param surfaceX surface-local x coordinate
   * @param surfaceY surface-local y coordinate
   */
  public void enter(WlPointerProxy emitter, int serial, @Nonnull WlSurfaceProxy surface, @Nonnull org.freedesktop.wayland.util.Fixed surfaceX, @Nonnull org.freedesktop.wayland.util.Fixed surfaceY);

  /**
   * leave event
   * <p>
   *
   * 	Notification that this seat's pointer is no longer focused on
   * 	a certain surface.
   * <p>
   * 	The leave notification is sent before the enter notification
   * 	for the new focus.
   *       
   * @param emitter The protocol object that emitted the event.
   * @param serial serial number of the leave event
   * @param surface surface left by the pointer
   */
  public void leave(WlPointerProxy emitter, int serial, @Nonnull WlSurfaceProxy surface);

  /**
   * pointer motion event
   * <p>
   *
   * 	Notification of pointer location change. The arguments
   * 	surface_x and surface_y are the location relative to the
   * 	focused surface.
   *       
   * @param emitter The protocol object that emitted the event.
   * @param time timestamp with millisecond granularity
   * @param surfaceX surface-local x coordinate
   * @param surfaceY surface-local y coordinate
   */
  public void motion(WlPointerProxy emitter, int time, @Nonnull org.freedesktop.wayland.util.Fixed surfaceX, @Nonnull org.freedesktop.wayland.util.Fixed surfaceY);

  /**
   * pointer button event
   * <p>
   *
   * 	Mouse button click and release notifications.
   * <p>
   * 	The location of the click is given by the last motion or
   * 	enter event.
   *         The time argument is a timestamp with millisecond
   *         granularity, with an undefined base.
   *       
   * @param emitter The protocol object that emitted the event.
   * @param serial serial number of the button event
   * @param time timestamp with millisecond granularity
   * @param button button that produced the event
   * @param state physical state of the button
   */
  public void button(WlPointerProxy emitter, int serial, int time, int button, int state);

  /**
   * axis event
   * <p>
   *
   * 	Scroll and other axis notifications.
   * <p>
   * 	For scroll events (vertical and horizontal scroll axes), the
   * 	value parameter is the length of a vector along the specified
   * 	axis in a coordinate space identical to those of motion events,
   * 	representing a relative movement along the specified axis.
   * <p>
   * 	For devices that support movements non-parallel to axes multiple
   * 	axis events will be emitted.
   * <p>
   * 	When applicable, for example for touch pads, the server can
   * 	choose to emit scroll events where the motion vector is
   * 	equivalent to a motion event vector.
   * <p>
   * 	When applicable, a client can transform its content relative to the
   * 	scroll distance.
   *       
   * @param emitter The protocol object that emitted the event.
   * @param time timestamp with millisecond granularity
   * @param axis axis type
   * @param value length of vector in surface-local coordinate space
   */
  public void axis(WlPointerProxy emitter, int time, int axis, @Nonnull org.freedesktop.wayland.util.Fixed value);
}
