package org.freedesktop.wayland.client;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.freedesktop.wayland.util.Arguments;
import org.freedesktop.wayland.util.Interface;
import org.freedesktop.wayland.util.Message;
// 
//
//    Copyright © 2008-2011 Kristian Høgsberg
//    Copyright © 2010-2011 Intel Corporation
//    Copyright © 2012-2013 Collabora, Ltd.
//
//    Permission is hereby granted, free of charge, to any person
//    obtaining a copy of this software and associated documentation files
//    (the "Software"), to deal in the Software without restriction,
//    including without limitation the rights to use, copy, modify, merge,
//    publish, distribute, sublicense, and/or sell copies of the Software,
//    and to permit persons to whom the Software is furnished to do so,
//    subject to the following conditions:
//
//    The above copyright notice and this permission notice (including the
//    next paragraph) shall be included in all copies or substantial
//    portions of the Software.
//
//    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
//    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
//    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
//    NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
//    BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
//    ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
//    CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//    SOFTWARE.
//  
/**
 * pointer input device
 * <p>
 *
 *       The wl_pointer interface represents one or more input devices,
 *       such as mice, which control the pointer location and pointer_focus
 *       of a seat.
 * <p>
 *       The wl_pointer interface generates motion, enter and leave
 *       events for the surfaces that the pointer is located over,
 *       and button and axis events for button presses, button releases
 *       and scrolling.
 *     
 */
@Interface(
  methods = {
    @Message(
  types = {
    int.class,
    org.freedesktop.wayland.client.WlSurfaceProxy.class,
    int.class,
    int.class
  },
  signature = "u?oii",
  functionName = "setCursor",
  name = "set_cursor"
)
,
    @Message(
  types = {
  },
  signature = "3",
  functionName = "release",
  name = "release"
)

  },
  name = "wl_pointer",
  version = 5,
  events = {
    @Message(
  types = {
    int.class,
    org.freedesktop.wayland.client.WlSurfaceProxy.class,
    org.freedesktop.wayland.util.Fixed.class,
    org.freedesktop.wayland.util.Fixed.class
  },
  signature = "uoff",
  functionName = "enter",
  name = "enter"
)
,
    @Message(
  types = {
    int.class,
    org.freedesktop.wayland.client.WlSurfaceProxy.class
  },
  signature = "uo",
  functionName = "leave",
  name = "leave"
)
,
    @Message(
  types = {
    int.class,
    org.freedesktop.wayland.util.Fixed.class,
    org.freedesktop.wayland.util.Fixed.class
  },
  signature = "uff",
  functionName = "motion",
  name = "motion"
)
,
    @Message(
  types = {
    int.class,
    int.class,
    int.class,
    int.class
  },
  signature = "uuuu",
  functionName = "button",
  name = "button"
)
,
    @Message(
  types = {
    int.class,
    int.class,
    org.freedesktop.wayland.util.Fixed.class
  },
  signature = "uuf",
  functionName = "axis",
  name = "axis"
)
,
    @Message(
  types = {
  },
  signature = "5",
  functionName = "frame",
  name = "frame"
)
,
    @Message(
  types = {
    int.class
  },
  signature = "5u",
  functionName = "axisSource",
  name = "axis_source"
)
,
    @Message(
  types = {
    int.class,
    int.class
  },
  signature = "5uu",
  functionName = "axisStop",
  name = "axis_stop"
)
,
    @Message(
  types = {
    int.class,
    int.class
  },
  signature = "5ui",
  functionName = "axisDiscrete",
  name = "axis_discrete"
)

  }
)
public class WlPointerProxy extends Proxy<WlPointerEvents> {

  public static final String INTERFACE_NAME = "wl_pointer";

  public WlPointerProxy(long pointer, WlPointerEvents implementation, int version) {
    super(pointer, implementation, version);
  }

  public WlPointerProxy(long pointer) {
    super(pointer);
  }

  /**
   * set the pointer surface
   * <p>
   *
   * 	Set the pointer surface, i.e., the surface that contains the
   * 	pointer image (cursor). This request gives the surface the role
   * 	of a cursor. If the surface already has another role, it raises
   * 	a protocol error.
   * <p>
   * 	The cursor actually changes only if the pointer
   * 	focus for this device is one of the requesting client's surfaces
   * 	or the surface parameter is the current pointer surface. If
   * 	there was a previous surface set with this request it is
   * 	replaced. If surface is NULL, the pointer image is hidden.
   * <p>
   * 	The parameters hotspot_x and hotspot_y define the position of
   * 	the pointer surface relative to the pointer location. Its
   * 	top-left corner is always at (x, y) - (hotspot_x, hotspot_y),
   * 	where (x, y) are the coordinates of the pointer location, in
   * 	surface-local coordinates.
   * <p>
   * 	On surface.attach requests to the pointer surface, hotspot_x
   * 	and hotspot_y are decremented by the x and y parameters
   * 	passed to the request. Attach must be confirmed by
   * 	wl_surface.commit as usual.
   * <p>
   * 	The hotspot can also be updated by passing the currently set
   * 	pointer surface to this request with new values for hotspot_x
   * 	and hotspot_y.
   * <p>
   * 	The current and pending input regions of the wl_surface are
   * 	cleared, and wl_surface.set_input_region is ignored until the
   * 	wl_surface is no longer used as the cursor. When the use as a
   * 	cursor ends, the current and pending input regions become
   * 	undefined, and the wl_surface is unmapped.
   *       
   * @param serial serial number of the enter event
   * @param surface pointer surface
   * @param hotspotX surface-local x coordinate
   * @param hotspotY surface-local y coordinate
   */
  public void setCursor(int serial, @Nullable WlSurfaceProxy surface, int hotspotX, int hotspotY) {
    marshal(0, Arguments.create(4).set(0, serial).set(1, surface).set(2, hotspotX).set(3, hotspotY));
  }

  /**
   * release the pointer object
   * <p>
   *
   * 	Using this request a client can tell the server that it is not going to
   * 	use the pointer object anymore.
   * <p>
   * 	This request destroys the pointer proxy object, so clients must not call
   * 	wl_pointer_destroy() after using this request.
   *       
   */
  public void release() {
    if (getVersion() < 3) {
      throw new UnsupportedOperationException("This object is version "+getVersion()+" while version 3 is required for this operation.");
    }
    marshal(1);
  }
}
