package org.freedesktop.wayland.client;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.freedesktop.wayland.util.Arguments;
import org.freedesktop.wayland.util.Interface;
import org.freedesktop.wayland.util.Message;
// 
//
//    Copyright © 2008-2011 Kristian Høgsberg
//    Copyright © 2010-2011 Intel Corporation
//    Copyright © 2012-2013 Collabora, Ltd.
//
//    Permission is hereby granted, free of charge, to any person
//    obtaining a copy of this software and associated documentation files
//    (the "Software"), to deal in the Software without restriction,
//    including without limitation the rights to use, copy, modify, merge,
//    publish, distribute, sublicense, and/or sell copies of the Software,
//    and to permit persons to whom the Software is furnished to do so,
//    subject to the following conditions:
//
//    The above copyright notice and this permission notice (including the
//    next paragraph) shall be included in all copies or substantial
//    portions of the Software.
//
//    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
//    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
//    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
//    NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
//    BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
//    ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
//    CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//    SOFTWARE.
//  
/**
 * group of input devices
 * <p>
 *
 *       A seat is a group of keyboards, pointer and touch devices. This
 *       object is published as a global during start up, or when such a
 *       device is hot plugged.  A seat typically has a pointer and
 *       maintains a keyboard focus and a pointer focus.
 *     
 */
@Interface(
  methods = {
    @Message(
  types = {
    org.freedesktop.wayland.client.WlPointerProxy.class
  },
  signature = "n",
  functionName = "getPointer",
  name = "get_pointer"
)
,
    @Message(
  types = {
    org.freedesktop.wayland.client.WlKeyboardProxy.class
  },
  signature = "n",
  functionName = "getKeyboard",
  name = "get_keyboard"
)
,
    @Message(
  types = {
    org.freedesktop.wayland.client.WlTouchProxy.class
  },
  signature = "n",
  functionName = "getTouch",
  name = "get_touch"
)
,
    @Message(
  types = {
  },
  signature = "5",
  functionName = "release",
  name = "release"
)

  },
  name = "wl_seat",
  version = 5,
  events = {
    @Message(
  types = {
    int.class
  },
  signature = "u",
  functionName = "capabilities",
  name = "capabilities"
)
,
    @Message(
  types = {
    java.lang.String.class
  },
  signature = "2s",
  functionName = "name",
  name = "name"
)

  }
)
public class WlSeatProxy extends Proxy<WlSeatEvents> {

  public static final String INTERFACE_NAME = "wl_seat";

  public WlSeatProxy(long pointer, WlSeatEvents implementation, int version) {
    super(pointer, implementation, version);
  }

  public WlSeatProxy(long pointer) {
    super(pointer);
  }

  /**
   * return pointer object
   * <p>
   *
   * 	The ID provided will be initialized to the wl_pointer interface
   * 	for this seat.
   * <p>
   * 	This request only takes effect if the seat has the pointer
   * 	capability, or has had the pointer capability in the past.
   * 	It is a protocol violation to issue this request on a seat that has
   * 	never had the pointer capability.
   *       
   * @param implementation A protocol event listener for the newly created proxy.
   */
  public WlPointerProxy getPointer(WlPointerEvents implementation) {
    return marshalConstructor(0, implementation, getVersion(), org.freedesktop.wayland.client.WlPointerProxy.class, Arguments.create(1).set(0, 0));
  }

  /**
   * return keyboard object
   * <p>
   *
   * 	The ID provided will be initialized to the wl_keyboard interface
   * 	for this seat.
   * <p>
   * 	This request only takes effect if the seat has the keyboard
   * 	capability, or has had the keyboard capability in the past.
   * 	It is a protocol violation to issue this request on a seat that has
   * 	never had the keyboard capability.
   *       
   * @param implementation A protocol event listener for the newly created proxy.
   */
  public WlKeyboardProxy getKeyboard(WlKeyboardEvents implementation) {
    return marshalConstructor(1, implementation, getVersion(), org.freedesktop.wayland.client.WlKeyboardProxy.class, Arguments.create(1).set(0, 0));
  }

  /**
   * return touch object
   * <p>
   *
   * 	The ID provided will be initialized to the wl_touch interface
   * 	for this seat.
   * <p>
   * 	This request only takes effect if the seat has the touch
   * 	capability, or has had the touch capability in the past.
   * 	It is a protocol violation to issue this request on a seat that has
   * 	never had the touch capability.
   *       
   * @param implementation A protocol event listener for the newly created proxy.
   */
  public WlTouchProxy getTouch(WlTouchEvents implementation) {
    return marshalConstructor(2, implementation, getVersion(), org.freedesktop.wayland.client.WlTouchProxy.class, Arguments.create(1).set(0, 0));
  }

  /**
   * release the seat object
   * <p>
   *
   * 	Using this request a client can tell the server that it is not going to
   * 	use the seat object anymore.
   *       
   */
  public void release() {
    if (getVersion() < 5) {
      throw new UnsupportedOperationException("This object is version "+getVersion()+" while version 5 is required for this operation.");
    }
    marshal(3);
  }
}
