package org.freedesktop.wayland.client;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
// 
//
//    Copyright © 2008-2011 Kristian Høgsberg
//    Copyright © 2010-2011 Intel Corporation
//    Copyright © 2012-2013 Collabora, Ltd.
//
//    Permission is hereby granted, free of charge, to any person
//    obtaining a copy of this software and associated documentation files
//    (the "Software"), to deal in the Software without restriction,
//    including without limitation the rights to use, copy, modify, merge,
//    publish, distribute, sublicense, and/or sell copies of the Software,
//    and to permit persons to whom the Software is furnished to do so,
//    subject to the following conditions:
//
//    The above copyright notice and this permission notice (including the
//    next paragraph) shall be included in all copies or substantial
//    portions of the Software.
//
//    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
//    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
//    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
//    NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
//    BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
//    ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
//    CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//    SOFTWARE.
//  
/**
 * desktop-style metadata interface
 * <p>
 *
 *       An interface that may be implemented by a wl_surface, for
 *       implementations that provide a desktop-style user interface.
 * <p>
 *       It provides requests to treat surfaces like toplevel, fullscreen
 *       or popup windows, move, resize or maximize them, associate
 *       metadata like title and class, etc.
 * <p>
 *       On the server side the object is automatically destroyed when
 *       the related wl_surface is destroyed. On the client side,
 *       wl_shell_surface_destroy() must be called before destroying
 *       the wl_surface object.
 *     
 */
public interface WlShellSurfaceEvents {
  int VERSION = 1;

  /**
   * ping client
   * <p>
   *
   * 	Ping a client to check if it is receiving events and sending
   * 	requests. A client is expected to reply with a pong request.
   *       
   * @param emitter The protocol object that emitted the event.
   * @param serial serial number of the ping
   */
  public void ping(WlShellSurfaceProxy emitter, int serial);

  /**
   * suggest resize
   * <p>
   *
   * 	The configure event asks the client to resize its surface.
   * <p>
   * 	The size is a hint, in the sense that the client is free to
   * 	ignore it if it doesn't resize, pick a smaller size (to
   * 	satisfy aspect ratio or resize in steps of NxM pixels).
   * <p>
   * 	The edges parameter provides a hint about how the surface
   * 	was resized. The client may use this information to decide
   * 	how to adjust its content to the new size (e.g. a scrolling
   * 	area might adjust its content position to leave the viewable
   * 	content unmoved).
   * <p>
   * 	The client is free to dismiss all but the last configure
   * 	event it received.
   * <p>
   * 	The width and height arguments specify the size of the window
   * 	in surface-local coordinates.
   *       
   * @param emitter The protocol object that emitted the event.
   * @param edges how the surface was resized
   * @param width new width of the surface
   * @param height new height of the surface
   */
  public void configure(WlShellSurfaceProxy emitter, int edges, int width, int height);

  /**
   * popup interaction is done
   * <p>
   *
   * 	The popup_done event is sent out when a popup grab is broken,
   * 	that is, when the user clicks a surface that doesn't belong
   * 	to the client owning the popup surface.
   *       
   * @param emitter The protocol object that emitted the event.
   */
  public void popupDone(WlShellSurfaceProxy emitter);
}
