package org.freedesktop.wayland.client;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.freedesktop.wayland.util.Arguments;
import org.freedesktop.wayland.util.Interface;
import org.freedesktop.wayland.util.Message;
// 
//
//    Copyright © 2008-2011 Kristian Høgsberg
//    Copyright © 2010-2011 Intel Corporation
//    Copyright © 2012-2013 Collabora, Ltd.
//
//    Permission is hereby granted, free of charge, to any person
//    obtaining a copy of this software and associated documentation files
//    (the "Software"), to deal in the Software without restriction,
//    including without limitation the rights to use, copy, modify, merge,
//    publish, distribute, sublicense, and/or sell copies of the Software,
//    and to permit persons to whom the Software is furnished to do so,
//    subject to the following conditions:
//
//    The above copyright notice and this permission notice (including the
//    next paragraph) shall be included in all copies or substantial
//    portions of the Software.
//
//    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
//    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
//    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
//    NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
//    BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
//    ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
//    CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//    SOFTWARE.
//  
/**
 * a shared memory pool
 * <p>
 *
 *       The wl_shm_pool object encapsulates a piece of memory shared
 *       between the compositor and client.  Through the wl_shm_pool
 *       object, the client can allocate shared memory wl_buffer objects.
 *       All objects created through the same pool share the same
 *       underlying mapped memory. Reusing the mapped memory avoids the
 *       setup/teardown overhead and is useful when interactively resizing
 *       a surface or for many small buffers.
 *     
 */
@Interface(
  methods = {
    @Message(
  types = {
    org.freedesktop.wayland.client.WlBufferProxy.class,
    int.class,
    int.class,
    int.class,
    int.class,
    int.class
  },
  signature = "niiiiu",
  functionName = "createBuffer",
  name = "create_buffer"
)
,
    @Message(
  types = {
  },
  signature = "",
  functionName = "destroy",
  name = "destroy"
)
,
    @Message(
  types = {
    int.class
  },
  signature = "i",
  functionName = "resize",
  name = "resize"
)

  },
  name = "wl_shm_pool",
  version = 1,
  events = {
  }
)
public class WlShmPoolProxy extends Proxy<WlShmPoolEvents> {

  public static final String INTERFACE_NAME = "wl_shm_pool";

  public WlShmPoolProxy(long pointer, WlShmPoolEvents implementation, int version) {
    super(pointer, implementation, version);
  }

  public WlShmPoolProxy(long pointer) {
    super(pointer);
  }

  /**
   * create a buffer from the pool
   * <p>
   *
   * 	Create a wl_buffer object from the pool.
   * <p>
   * 	The buffer is created offset bytes into the pool and has
   * 	width and height as specified.  The stride argument specifies
   * 	the number of bytes from the beginning of one row to the beginning
   * 	of the next.  The format is the pixel format of the buffer and
   * 	must be one of those advertised through the wl_shm.format event.
   * <p>
   * 	A buffer will keep a reference to the pool it was created from
   * 	so it is valid to destroy the pool immediately after creating
   * 	a buffer from it.
   *       
   * @param implementation A protocol event listener for the newly created proxy.
   * @param offset buffer byte offset within the pool
   * @param width buffer width, in pixels
   * @param height buffer height, in pixels
   * @param stride number of bytes from the beginning of one row to the beginning of the next row
   * @param format buffer pixel format
   */
  public WlBufferProxy createBuffer(WlBufferEvents implementation, int offset, int width, int height, int stride, int format) {
    return marshalConstructor(0, implementation, getVersion(), org.freedesktop.wayland.client.WlBufferProxy.class, Arguments.create(6).set(0, 0).set(1, offset).set(2, width).set(3, height).set(4, stride).set(5, format));
  }

  /**
   * destroy the pool
   * <p>
   *
   * 	Destroy the shared memory pool.
   * <p>
   * 	The mmapped memory will be released when all
   * 	buffers that have been created from this pool
   * 	are gone.
   *       
   */
  public void destroy() {
    marshal(1);
  }

  /**
   * change the size of the pool mapping
   * <p>
   *
   * 	This request will cause the server to remap the backing memory
   * 	for the pool from the file descriptor passed when the pool was
   * 	created, but using the new size.  This request can only be
   * 	used to make the pool bigger.
   *       
   * @param size new size of the pool, in bytes
   */
  public void resize(int size) {
    marshal(2, Arguments.create(1).set(0, size));
  }
}
