package org.freedesktop.wayland.client;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.freedesktop.wayland.util.Arguments;
import org.freedesktop.wayland.util.Interface;
import org.freedesktop.wayland.util.Message;
// 
//
//    Copyright © 2008-2011 Kristian Høgsberg
//    Copyright © 2010-2011 Intel Corporation
//    Copyright © 2012-2013 Collabora, Ltd.
//
//    Permission is hereby granted, free of charge, to any person
//    obtaining a copy of this software and associated documentation files
//    (the "Software"), to deal in the Software without restriction,
//    including without limitation the rights to use, copy, modify, merge,
//    publish, distribute, sublicense, and/or sell copies of the Software,
//    and to permit persons to whom the Software is furnished to do so,
//    subject to the following conditions:
//
//    The above copyright notice and this permission notice (including the
//    next paragraph) shall be included in all copies or substantial
//    portions of the Software.
//
//    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
//    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
//    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
//    NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
//    BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
//    ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
//    CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//    SOFTWARE.
//  
/**
 * sub-surface compositing
 * <p>
 *
 *       The global interface exposing sub-surface compositing capabilities.
 *       A wl_surface, that has sub-surfaces associated, is called the
 *       parent surface. Sub-surfaces can be arbitrarily nested and create
 *       a tree of sub-surfaces.
 * <p>
 *       The root surface in a tree of sub-surfaces is the main
 *       surface. The main surface cannot be a sub-surface, because
 *       sub-surfaces must always have a parent.
 * <p>
 *       A main surface with its sub-surfaces forms a (compound) window.
 *       For window management purposes, this set of wl_surface objects is
 *       to be considered as a single window, and it should also behave as
 *       such.
 * <p>
 *       The aim of sub-surfaces is to offload some of the compositing work
 *       within a window from clients to the compositor. A prime example is
 *       a video player with decorations and video in separate wl_surface
 *       objects. This should allow the compositor to pass YUV video buffer
 *       processing to dedicated overlay hardware when possible.
 *     
 */
@Interface(
  methods = {
    @Message(
  types = {
  },
  signature = "",
  functionName = "destroy",
  name = "destroy"
)
,
    @Message(
  types = {
    org.freedesktop.wayland.client.WlSubsurfaceProxy.class,
    org.freedesktop.wayland.client.WlSurfaceProxy.class,
    org.freedesktop.wayland.client.WlSurfaceProxy.class
  },
  signature = "noo",
  functionName = "getSubsurface",
  name = "get_subsurface"
)

  },
  name = "wl_subcompositor",
  version = 1,
  events = {
  }
)
public class WlSubcompositorProxy extends Proxy<WlSubcompositorEvents> {

  public static final String INTERFACE_NAME = "wl_subcompositor";

  public WlSubcompositorProxy(long pointer, WlSubcompositorEvents implementation, int version) {
    super(pointer, implementation, version);
  }

  public WlSubcompositorProxy(long pointer) {
    super(pointer);
  }

  /**
   * unbind from the subcompositor interface
   * <p>
   *
   * 	Informs the server that the client will not be using this
   * 	protocol object anymore. This does not affect any other
   * 	objects, wl_subsurface objects included.
   *       
   */
  public void destroy() {
    marshal(0);
  }

  /**
   * give a surface the role sub-surface
   * <p>
   *
   * 	Create a sub-surface interface for the given surface, and
   * 	associate it with the given parent surface. This turns a
   * 	plain wl_surface into a sub-surface.
   * <p>
   * 	The to-be sub-surface must not already have another role, and it
   * 	must not have an existing wl_subsurface object. Otherwise a protocol
   * 	error is raised.
   *       
   * @param implementation A protocol event listener for the newly created proxy.
   * @param surface the surface to be turned into a sub-surface
   * @param parent the parent surface
   */
  public WlSubsurfaceProxy getSubsurface(WlSubsurfaceEvents implementation, @Nonnull WlSurfaceProxy surface, @Nonnull WlSurfaceProxy parent) {
    return marshalConstructor(1, implementation, getVersion(), org.freedesktop.wayland.client.WlSubsurfaceProxy.class, Arguments.create(3).set(0, 0).set(1, surface).set(2, parent));
  }
}
