/*
 * Decompiled with CFR 0.152.
 */
package org.apache.shindig.gadgets.http;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.Arrays;
import junitx.framework.Assert;
import org.apache.commons.io.IOUtils;
import org.apache.shindig.common.util.DateUtil;
import org.apache.shindig.gadgets.http.HttpResponse;
import org.apache.shindig.gadgets.http.HttpResponseBuilder;
import org.junit.Test;

public class HttpResponseTest {
    private static final byte[] UTF8_DATA = new byte[]{-17, -69, -65, 104, 101, 108, 108, 111};
    private static final String UTF8_STRING = "hello";
    private static final byte[] LATIN1_DATA = new byte[]{71, 97, 109, 101, 115, 44, 32, 72, 81, 44, 32, 77, 97, 110, 103, -31, 44, 32, 65, 110, 105, 109, 101, 32, 101, 32, 116, 117, 100, 111, 32, 113, 117, 101, 32, 117, 109, 32, 98, 111, 109, 32, 110, 101, 114, 100, 32, 97, 109, 97};
    private static final String LATIN1_STRING = "Games, HQ, Mang\u00e1, Anime e tudo que um bom nerd ama";
    private static final byte[] BIG5_DATA = new byte[]{-89, 65, -90, 110};
    private static final String BIG5_STRING = "\u4f60\u597d";

    private static int roundToSeconds(long ts) {
        return (int)(ts / 1000L);
    }

    @Test
    public void testEncodingDetectionUtf8WithBom() throws Exception {
        HttpResponse response = new HttpResponseBuilder().addHeader("Content-Type", "text/plain; charset=UTF-8").setResponse(UTF8_DATA).create();
        Assert.assertEquals((String)UTF8_STRING, (String)response.getResponseAsString());
        Assert.assertEquals((String)"UTF-8", (String)response.getEncoding());
    }

    @Test
    public void testEncodingDetectionLatin1() throws Exception {
        HttpResponse response = new HttpResponseBuilder().addHeader("Content-Type", "text/plain; charset=iso-8859-1").setResponse(LATIN1_DATA).create();
        Assert.assertEquals((String)LATIN1_STRING, (String)response.getResponseAsString());
    }

    @Test
    public void testEncodingDetectionLatin1withIncorrectCharset() throws Exception {
        HttpResponse response = new HttpResponseBuilder().addHeader("Content-Type", "text/plain; charset=iso-88859-1").setResponse(LATIN1_DATA).create();
        Assert.assertEquals((String)LATIN1_STRING, (String)response.getResponseAsString());
        Assert.assertEquals((String)"ISO-8859-1", (String)response.getEncoding());
    }

    @Test
    public void testEncodingDetectionUtf8WithBomAndIncorrectCharset() throws Exception {
        HttpResponse response = new HttpResponseBuilder().addHeader("Content-Type", "text/plain; charset=UTTFF-88").setResponse(UTF8_DATA).create();
        Assert.assertEquals((String)UTF8_STRING, (String)response.getResponseAsString());
        Assert.assertEquals((String)"UTF-8", (String)response.getEncoding());
    }

    @Test
    public void testEncodingDetectionUtf8WithBomAndInvalidCharset() throws Exception {
        HttpResponse response = new HttpResponseBuilder().addHeader("Content-Type", "text/plain; charset=.UTF-8").setResponse(UTF8_DATA).create();
        Assert.assertEquals((String)UTF8_STRING, (String)response.getResponseAsString());
        Assert.assertEquals((String)"UTF-8", (String)response.getEncoding());
    }

    @Test
    public void testEncodingDetectionBig5() throws Exception {
        HttpResponse response = new HttpResponseBuilder().addHeader("Content-Type", "text/plain; charset=BIG5").setResponse(BIG5_DATA).create();
        Assert.assertEquals((String)BIG5_STRING, (String)response.getResponseAsString());
    }

    @Test
    public void testEncodingDetectionBig5WithQuotes() throws Exception {
        HttpResponse response = new HttpResponseBuilder().addHeader("Content-Type", "text/plain; charset=\"BIG5\"").setResponse(BIG5_DATA).create();
        Assert.assertEquals((String)BIG5_STRING, (String)response.getResponseAsString());
    }

    @Test
    public void testEncodingDetectionUtf8WithBomNoCharsetSpecified() throws Exception {
        HttpResponse response = new HttpResponseBuilder().addHeader("Content-Type", "text/plain").setResponse(UTF8_DATA).create();
        Assert.assertEquals((String)"UTF-8", (String)response.getEncoding().toUpperCase());
        Assert.assertEquals((String)UTF8_STRING, (String)response.getResponseAsString());
    }

    @Test
    public void testEncodingDetectionLatin1NoCharsetSpecified() throws Exception {
        HttpResponse response = new HttpResponseBuilder().addHeader("Content-Type", "text/plain;").setResponse(LATIN1_DATA).create();
        Assert.assertEquals((String)"ISO-8859-1", (String)response.getEncoding().toUpperCase());
        Assert.assertEquals((String)LATIN1_STRING, (String)response.getResponseAsString());
    }

    @Test
    public void testEncodingDetectionUtf8WithBomNoContentHeader() throws Exception {
        HttpResponse response = new HttpResponseBuilder().setResponse(UTF8_DATA).create();
        Assert.assertEquals((String)"UTF-8", (String)response.getEncoding().toUpperCase());
        Assert.assertEquals((String)UTF8_STRING, (String)response.getResponseAsString());
    }

    @Test
    public void testEncodingDetectionLatin1NoContentHeader() throws Exception {
        HttpResponse response = new HttpResponseBuilder().setResponse(LATIN1_DATA).create();
        Assert.assertEquals((String)HttpResponse.DEFAULT_ENCODING.name(), (String)response.getEncoding());
    }

    @Test
    public void testGetEncodingForImageContentType() throws Exception {
        HttpResponse response = new HttpResponseBuilder().setResponse(LATIN1_DATA).addHeader("Content-Type", "image/png; charset=iso-8859-1").create();
        Assert.assertEquals((String)HttpResponse.DEFAULT_ENCODING.name(), (String)response.getEncoding().toUpperCase());
    }

    @Test
    public void testGetEncodingForFlashContentType() throws Exception {
        HttpResponse response = new HttpResponseBuilder().setResponse(LATIN1_DATA).addHeader("Content-Type", "application/x-shockwave-flash; charset=iso-8859-1").create();
        Assert.assertEquals((String)HttpResponse.DEFAULT_ENCODING.name(), (String)response.getEncoding().toUpperCase());
    }

    @Test
    public void testPreserveBinaryData() throws Exception {
        byte[] data = new byte[]{0, -34, -22, -37, -18, -16};
        HttpResponse response = new HttpResponseBuilder().addHeader("Content-Type", "application/octet-stream").setResponse(data).create();
        byte[] out = IOUtils.toByteArray((InputStream)response.getResponse());
        Assert.assertEquals((int)data.length, (int)response.getContentLength());
        junit.framework.Assert.assertTrue((boolean)Arrays.equals(data, out));
        out = IOUtils.toByteArray((InputStream)response.getResponse());
        junit.framework.Assert.assertTrue((boolean)Arrays.equals(data, out));
    }

    @Test
    public void testStrictCacheControlNoCache() throws Exception {
        HttpResponse response = new HttpResponseBuilder().addHeader("Cache-Control", "no-cache").create();
        junit.framework.Assert.assertTrue((boolean)response.isStrictNoCache());
        Assert.assertEquals((long)-1L, (long)response.getCacheExpiration());
        Assert.assertEquals((long)-1L, (long)response.getCacheTtl());
    }

    @Test
    public void testStrictPragmaNoCache() throws Exception {
        HttpResponse response = new HttpResponseBuilder().addHeader("Pragma", "no-cache").create();
        junit.framework.Assert.assertTrue((boolean)response.isStrictNoCache());
        Assert.assertEquals((long)-1L, (long)response.getCacheExpiration());
        Assert.assertEquals((long)-1L, (long)response.getCacheTtl());
    }

    @Test
    public void testStrictPragmaJunk() throws Exception {
        HttpResponse response = new HttpResponseBuilder().addHeader("Pragma", "junk").create();
        Assert.assertFalse((boolean)response.isStrictNoCache());
        int expected = HttpResponseTest.roundToSeconds(System.currentTimeMillis() + 300000L);
        int expires = HttpResponseTest.roundToSeconds(response.getCacheExpiration());
        Assert.assertEquals((int)expected, (int)expires);
        junit.framework.Assert.assertTrue((response.getCacheTtl() <= 300000L && response.getCacheTtl() > 0L ? 1 : 0) != 0);
    }

    @Test
    public void testCachingHeadersIgnoredOnError() throws Exception {
        HttpResponse response = new HttpResponseBuilder().addHeader("Cache-Control", "no-cache").setHttpStatusCode(404).create();
        Assert.assertFalse((boolean)response.isStrictNoCache());
        response = new HttpResponseBuilder().addHeader("Cache-Control", "no-cache").setHttpStatusCode(403).create();
        junit.framework.Assert.assertTrue((boolean)response.isStrictNoCache());
        response = new HttpResponseBuilder().addHeader("Cache-Control", "no-cache").setHttpStatusCode(401).create();
        junit.framework.Assert.assertTrue((boolean)response.isStrictNoCache());
    }

    private static void assertTtlOk(int maxAge, HttpResponse response) {
        Assert.assertEquals((int)(maxAge - 1), (int)HttpResponseTest.roundToSeconds(response.getCacheTtl() - 1L));
    }

    @Test
    public void testExpires() throws Exception {
        int maxAge = 10;
        int time = HttpResponseTest.roundToSeconds(System.currentTimeMillis()) + maxAge;
        HttpResponse response = new HttpResponseBuilder().addHeader("Expires", DateUtil.formatRfc1123Date((long)(1000L * (long)time))).create();
        Assert.assertEquals((int)time, (int)HttpResponseTest.roundToSeconds(response.getCacheExpiration()));
        HttpResponseTest.assertTtlOk(maxAge, response);
    }

    @Test
    public void testMaxAgeNoDate() throws Exception {
        int maxAge = 10;
        int expected = HttpResponseTest.roundToSeconds(System.currentTimeMillis()) + maxAge;
        HttpResponse response = new HttpResponseBuilder().addHeader("Cache-Control", "public, max-age=" + maxAge).create();
        int expiration = HttpResponseTest.roundToSeconds(response.getCacheExpiration());
        Assert.assertEquals((int)expected, (int)expiration);
        HttpResponseTest.assertTtlOk(maxAge, response);
    }

    @Test
    public void testMaxAgeInvalidDate() throws Exception {
        int maxAge = 10;
        int expected = HttpResponseTest.roundToSeconds(System.currentTimeMillis()) + maxAge;
        HttpResponse response = new HttpResponseBuilder().addHeader("Date", "Wed, 09 Jul 2008 19:18:33 EDT").addHeader("Cache-Control", "public, max-age=" + maxAge).create();
        int expiration = HttpResponseTest.roundToSeconds(response.getCacheExpiration());
        Assert.assertEquals((int)expected, (int)expiration);
        HttpResponseTest.assertTtlOk(maxAge, response);
    }

    @Test
    public void testMaxAgeWithDate() throws Exception {
        int maxAge = 10;
        int now = HttpResponseTest.roundToSeconds(System.currentTimeMillis());
        HttpResponse response = new HttpResponseBuilder().addHeader("Date", DateUtil.formatRfc1123Date((long)(1000L * (long)now))).addHeader("Cache-Control", "public, max-age=" + maxAge).create();
        Assert.assertEquals((int)(now + maxAge), (int)HttpResponseTest.roundToSeconds(response.getCacheExpiration()));
        HttpResponseTest.assertTtlOk(maxAge, response);
    }

    @Test
    public void testFixedDate() throws Exception {
        int time = HttpResponseTest.roundToSeconds(System.currentTimeMillis());
        HttpResponse response = new HttpResponseBuilder().addHeader("Date", DateUtil.formatRfc1123Date((long)(1000L * (long)time))).create();
        Assert.assertEquals((int)(time + HttpResponseTest.roundToSeconds(300000L)), (int)HttpResponseTest.roundToSeconds(response.getCacheExpiration()));
        HttpResponseTest.assertTtlOk(HttpResponseTest.roundToSeconds(300000L), response);
    }

    @Test
    public void testNegativeCaching() {
        junit.framework.Assert.assertTrue((String)"Bad HTTP responses must be cacheable!", (HttpResponse.error().getCacheExpiration() > System.currentTimeMillis() ? 1 : 0) != 0);
        junit.framework.Assert.assertTrue((String)"Bad HTTP responses must be cacheable!", (HttpResponse.notFound().getCacheExpiration() > System.currentTimeMillis() ? 1 : 0) != 0);
        junit.framework.Assert.assertTrue((String)"Bad HTTP responses must be cacheable!", (HttpResponse.timeout().getCacheExpiration() > System.currentTimeMillis() ? 1 : 0) != 0);
        long ttl = HttpResponse.error().getCacheTtl();
        junit.framework.Assert.assertTrue((ttl <= 300000L && ttl > 0L ? 1 : 0) != 0);
    }

    private static void assertDoesNotAllowNegativeCaching(int status) {
        HttpResponse response = new HttpResponseBuilder().setHttpStatusCode(status).setResponse(UTF8_DATA).setStrictNoCache().create();
        Assert.assertEquals((long)-1L, (long)response.getCacheTtl());
    }

    private static void assertAllowsNegativeCaching(int status) {
        HttpResponse response = new HttpResponseBuilder().setHttpStatusCode(status).setResponse(UTF8_DATA).setStrictNoCache().create();
        long ttl = response.getCacheTtl();
        junit.framework.Assert.assertTrue((ttl <= 300000L && ttl > 0L ? 1 : 0) != 0);
    }

    @Test
    public void testStrictNoCacheAndNegativeCaching() {
        HttpResponseTest.assertDoesNotAllowNegativeCaching(401);
        HttpResponseTest.assertDoesNotAllowNegativeCaching(403);
        HttpResponseTest.assertDoesNotAllowNegativeCaching(200);
        HttpResponseTest.assertAllowsNegativeCaching(404);
        HttpResponseTest.assertAllowsNegativeCaching(500);
        HttpResponseTest.assertAllowsNegativeCaching(504);
    }

    @Test
    public void testSetNoCache() {
        int time = HttpResponseTest.roundToSeconds(System.currentTimeMillis());
        HttpResponse response = new HttpResponseBuilder().addHeader("Expires", DateUtil.formatRfc1123Date((long)(1000L * (long)time))).setStrictNoCache().create();
        junit.framework.Assert.assertNull((Object)response.getHeader("Expires"));
        Assert.assertEquals((String)"no-cache", (String)response.getHeader("Pragma"));
        Assert.assertEquals((String)"no-cache", (String)response.getHeader("Cache-Control"));
    }

    @Test
    public void testNullHeaderNamesStripped() {
        HttpResponse response = new HttpResponseBuilder().addHeader(null, "dummy").create();
        for (String key : response.getHeaders().keySet()) {
            junit.framework.Assert.assertNotNull((String)"Null header not removed.", (Object)key);
        }
    }

    @Test
    public void testIsError() {
        HttpResponse response;
        int j;
        int i;
        for (i = 100; i < 400; i += 100) {
            for (j = 0; j < 10; ++j) {
                response = new HttpResponseBuilder().setHttpStatusCode(i).create();
                Assert.assertFalse((String)"Status below 400 considered to be an error", (boolean)response.isError());
            }
        }
        for (i = 400; i < 600; i += 100) {
            for (j = 0; j < 10; ++j) {
                response = new HttpResponseBuilder().setHttpStatusCode(i).create();
                junit.framework.Assert.assertTrue((String)"Status above 400 considered to be an error", (boolean)response.isError());
            }
        }
    }

    @Test
    public void testSerialization() throws Exception {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        ObjectOutputStream out = new ObjectOutputStream(baos);
        HttpResponse response = new HttpResponseBuilder().addHeader("Foo", "bar").addHeader("Foo", "baz").addHeader("Blah", "blah").setHttpStatusCode(204).setResponseString("This is the response string").create();
        out.writeObject(response);
        ByteArrayInputStream bais = new ByteArrayInputStream(baos.toByteArray());
        ObjectInputStream in = new ObjectInputStream(bais);
        HttpResponse deserialized = (HttpResponse)in.readObject();
        Assert.assertEquals((Object)response, (Object)deserialized);
    }

    @Test
    public void testSerializationWithTransientFields() throws Exception {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        ObjectOutputStream out = new ObjectOutputStream(baos);
        long now = System.currentTimeMillis();
        HttpResponse response = new HttpResponseBuilder().addHeader("Foo", "bar").addHeader("Foo", "baz").addHeader("Blah", "blah").addHeader("Date", DateUtil.formatRfc1123Date((long)now)).setHttpStatusCode(204).setResponseString("This is the response string").setMetadata("foo", "bar").create();
        out.writeObject(response);
        ByteArrayInputStream bais = new ByteArrayInputStream(baos.toByteArray());
        ObjectInputStream in = new ObjectInputStream(bais);
        HttpResponse deserialized = (HttpResponse)in.readObject();
        HttpResponse expectedResponse = new HttpResponseBuilder().addHeader("Foo", "bar").addHeader("Foo", "baz").addHeader("Blah", "blah").addHeader("Date", DateUtil.formatRfc1123Date((long)now)).setHttpStatusCode(204).setResponseString("This is the response string").create();
        Assert.assertEquals((Object)expectedResponse, (Object)deserialized);
    }
}

