/*
 * Decompiled with CFR 0.152.
 */
package org.apache.shindig.gadgets.oauth;

import com.google.common.collect.Lists;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.Logger;
import net.oauth.OAuth;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang.ArrayUtils;
import org.apache.shindig.auth.BasicSecurityToken;
import org.apache.shindig.auth.SecurityToken;
import org.apache.shindig.common.crypto.BasicBlobCrypter;
import org.apache.shindig.common.crypto.BlobCrypter;
import org.apache.shindig.common.uri.Uri;
import org.apache.shindig.common.uri.UriBuilder;
import org.apache.shindig.common.util.CharsetUtil;
import org.apache.shindig.common.util.FakeTimeSource;
import org.apache.shindig.common.util.TimeSource;
import org.apache.shindig.gadgets.FakeGadgetSpecFactory;
import org.apache.shindig.gadgets.GadgetException;
import org.apache.shindig.gadgets.GadgetSpecFactory;
import org.apache.shindig.gadgets.http.HttpFetcher;
import org.apache.shindig.gadgets.http.HttpRequest;
import org.apache.shindig.gadgets.http.HttpResponse;
import org.apache.shindig.gadgets.oauth.AccessorInfo;
import org.apache.shindig.gadgets.oauth.BasicOAuthStore;
import org.apache.shindig.gadgets.oauth.BasicOAuthStoreConsumerIndex;
import org.apache.shindig.gadgets.oauth.BasicOAuthStoreConsumerKeyAndSecret;
import org.apache.shindig.gadgets.oauth.GadgetOAuthTokenStore;
import org.apache.shindig.gadgets.oauth.OAuthArguments;
import org.apache.shindig.gadgets.oauth.OAuthCallbackGenerator;
import org.apache.shindig.gadgets.oauth.OAuthError;
import org.apache.shindig.gadgets.oauth.OAuthFetcherConfig;
import org.apache.shindig.gadgets.oauth.OAuthResponseParams;
import org.apache.shindig.gadgets.oauth.OAuthStore;
import org.apache.shindig.gadgets.oauth.testing.FakeOAuthServiceProvider;
import org.apache.shindig.gadgets.oauth.testing.MakeRequestClient;
import org.json.JSONObject;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class OAuthRequestTest {
    private OAuthFetcherConfig fetcherConfig;
    private FakeOAuthServiceProvider serviceProvider;
    private OAuthCallbackGenerator callbackGenerator;
    private BasicOAuthStore base;
    private Logger logger;
    protected final List<LogRecord> logRecords = Lists.newArrayList();
    private final FakeTimeSource clock = new FakeTimeSource();
    public static final String GADGET_URL = "http://www.example.com/gadget.xml";
    public static final String GADGET_URL_NO_KEY = "http://www.example.com/nokey.xml";
    public static final String GADGET_URL_HEADER = "http://www.example.com/header.xml";
    public static final String GADGET_URL_BODY = "http://www.example.com/body.xml";
    public static final String GADGET_URL_BAD_OAUTH_URL = "http://www.example.com/badoauthurl.xml";
    public static final String GADGET_URL_APPROVAL_PARAMS = "http://www.example.com/approvalparams.xml";
    public static final String GADGET_MAKE_REQUEST_URL = "http://127.0.0.1/gadgets/makeRequest?params=foo";

    @Before
    public void setUp() throws Exception {
        this.base = new BasicOAuthStore();
        this.base.setDefaultCallbackUrl("http://www.example.com/oauthcallback");
        this.serviceProvider = new FakeOAuthServiceProvider((TimeSource)this.clock);
        this.callbackGenerator = this.createNullCallbackGenerator();
        this.fetcherConfig = new OAuthFetcherConfig((BlobCrypter)new BasicBlobCrypter("abcdefghijklmnop".getBytes()), this.getOAuthStore(this.base), (TimeSource)this.clock, this.callbackGenerator, false);
        this.logger = Logger.getLogger(OAuthResponseParams.class.getName());
        this.logger.addHandler(new Handler(){

            public void close() throws SecurityException {
            }

            public void flush() {
            }

            public void publish(LogRecord arg0) {
                OAuthRequestTest.this.logRecords.add(arg0);
            }
        });
    }

    private OAuthCallbackGenerator createNullCallbackGenerator() {
        return new OAuthCallbackGenerator(){

            public String generateCallback(OAuthFetcherConfig fetcherConfig, String baseCallback, HttpRequest request, OAuthResponseParams responseParams) {
                return null;
            }
        };
    }

    private OAuthCallbackGenerator createRealCallbackGenerator() {
        return new OAuthCallbackGenerator(){

            public String generateCallback(OAuthFetcherConfig fetcherConfig, String baseCallback, HttpRequest request, OAuthResponseParams responseParams) {
                SecurityToken st = request.getSecurityToken();
                Uri activeUrl = Uri.parse((String)st.getActiveUrl());
                Assert.assertEquals((Object)OAuthRequestTest.GADGET_MAKE_REQUEST_URL, (Object)activeUrl.toString());
                Assert.assertEquals((Object)"http://www.example.com/oauthcallback", (Object)baseCallback);
                return new UriBuilder().setScheme("http").setAuthority(activeUrl.getAuthority()).setPath("/realcallback").toString();
            }
        };
    }

    public GadgetOAuthTokenStore getOAuthStore(BasicOAuthStore base) {
        return this.getOAuthStore(base, new FakeGadgetSpecFactory());
    }

    private GadgetOAuthTokenStore getOAuthStore(BasicOAuthStore base, GadgetSpecFactory specFactory) {
        if (base == null) {
            base = new BasicOAuthStore();
            base.setDefaultCallbackUrl("http://www.example.com/oauthcallback");
        }
        OAuthRequestTest.addValidConsumer(base);
        OAuthRequestTest.addInvalidConsumer(base);
        OAuthRequestTest.addAuthHeaderConsumer(base);
        OAuthRequestTest.addBodyConsumer(base);
        OAuthRequestTest.addBadOAuthUrlConsumer(base);
        OAuthRequestTest.addApprovalParamsConsumer(base);
        OAuthRequestTest.addDefaultKey(base);
        return new GadgetOAuthTokenStore((OAuthStore)base, specFactory);
    }

    private static void addValidConsumer(BasicOAuthStore base) {
        OAuthRequestTest.addConsumer(base, GADGET_URL, "testservice", "consumer", "secret");
    }

    private static void addInvalidConsumer(BasicOAuthStore base) {
        OAuthRequestTest.addConsumer(base, GADGET_URL_NO_KEY, "nokey", "garbage_key", "garbage_secret");
    }

    private static void addAuthHeaderConsumer(BasicOAuthStore base) {
        OAuthRequestTest.addConsumer(base, GADGET_URL_HEADER, "testservice", "consumer", "secret");
    }

    private static void addBodyConsumer(BasicOAuthStore base) {
        OAuthRequestTest.addConsumer(base, GADGET_URL_BODY, "testservice", "consumer", "secret");
    }

    private static void addBadOAuthUrlConsumer(BasicOAuthStore base) {
        OAuthRequestTest.addConsumer(base, GADGET_URL_BAD_OAUTH_URL, "testservice", "consumer", "secret");
    }

    private static void addApprovalParamsConsumer(BasicOAuthStore base) {
        OAuthRequestTest.addConsumer(base, GADGET_URL_APPROVAL_PARAMS, "testservice", "consumer", "secret");
    }

    private static void addConsumer(BasicOAuthStore base, String gadgetUrl, String serviceName, String consumerKey, String consumerSecret) {
        BasicOAuthStoreConsumerIndex providerKey = new BasicOAuthStoreConsumerIndex();
        providerKey.setGadgetUri(gadgetUrl);
        providerKey.setServiceName(serviceName);
        BasicOAuthStoreConsumerKeyAndSecret kas = new BasicOAuthStoreConsumerKeyAndSecret(consumerKey, consumerSecret, BasicOAuthStoreConsumerKeyAndSecret.KeyType.HMAC_SYMMETRIC, null, null);
        base.setConsumerKeyAndSecret(providerKey, kas);
    }

    private static void addDefaultKey(BasicOAuthStore base) {
        BasicOAuthStoreConsumerKeyAndSecret defaultKey = new BasicOAuthStoreConsumerKeyAndSecret("signedfetch", "MIICdgIBADANBgkqhkiG9w0BAQEFAASCAmAwggJcAgEAAoGBALRiMLAh9iimur8VA7qVvdqxevEuUkW4K+2KdMXmnQbG9Aa7k7eBjK1S+0LYmVjPKlJGNXHDGuy5Fw/d7rjVJ0BLB+ubPK8iA/Tw3hLQgXMRRGRXXCn8ikfuQfjUS1uZSatdLB81mydBETlJhI6GH4twrbDJCR2Bwy/XWXgqgGRzAgMBAAECgYBYWVtleUzavkbrPjy0T5FMou8HX9u2AC2ry8vD/l7cqedtwMPp9k7TubgNFo+NGvKsl2ynyprOZR1xjQ7WgrgVB+mmuScOM/5HVceFuGRDhYTCObE+y1kxRloNYXnx3ei1zbeYLPCHdhxRYW7T0qcynNmwrn05/KO2RLjgQNalsQJBANeA3Q4Nugqy4QBUCEC09SqylT2K9FrrItqL2QKc9v0ZzO2uwllCbg0dwpVuYPYXYvikNHHg+aCWF+VXsb9rpPsCQQDWR9TT4ORdzoj+NccnqkMsDmzt0EfNaAOwHOmVJ2RVBspPcxt5iN4HI7HNeG6U5YsFBb+/GZbgfBT3kpNGWPTpAkBI+gFhjfJvRw38n3g/+UeAkwMI2TJQS4n8+hid0uus3/zOjDySH3XHCUnocn1xOJAyZODBo47E+67R4jV1/gzbAkEAklJaspRPXP877NssM5nAZMU0/O/NGCZ+3jPgDUno6WbJn5cqm8MqWhW1xGkImgRk+fkDBquiq4gPiT898jusgQJAd5Zrr6Q8AO/0isr/3aa6O6NLQxISLKcPDk2NOccAfS/xOtfOz4sJYM3+Bs4Io9+dZGSDCA54Lw03eHTNQghS0A==", BasicOAuthStoreConsumerKeyAndSecret.KeyType.RSA_PRIVATE, "foo", null);
        base.setDefaultKey(defaultKey);
    }

    public static SecurityToken getNormalSecurityToken(String owner, String viewer) throws Exception {
        return OAuthRequestTest.getSecurityToken(owner, viewer, GADGET_URL);
    }

    public static SecurityToken getNokeySecurityToken(String owner, String viewer) throws Exception {
        return OAuthRequestTest.getSecurityToken(owner, viewer, GADGET_URL_NO_KEY);
    }

    public static SecurityToken getHeaderSecurityToken(String owner, String viewer) throws Exception {
        return OAuthRequestTest.getSecurityToken(owner, viewer, GADGET_URL_HEADER);
    }

    public static SecurityToken getBodySecurityToken(String owner, String viewer) throws Exception {
        return OAuthRequestTest.getSecurityToken(owner, viewer, GADGET_URL_BODY);
    }

    public static SecurityToken getSecurityToken(String owner, String viewer, String gadget) throws Exception {
        return new BasicSecurityToken(owner, viewer, "app", "container.com", gadget, "0", "default", GADGET_MAKE_REQUEST_URL, null);
    }

    @After
    public void tearDown() throws Exception {
    }

    private MakeRequestClient makeNonSocialClient(String owner, String viewer, String gadget) throws Exception {
        SecurityToken securityToken = OAuthRequestTest.getSecurityToken(owner, viewer, gadget);
        MakeRequestClient client = new MakeRequestClient(securityToken, this.fetcherConfig, this.serviceProvider, "testservice");
        client.getBaseArgs().setSignOwner(true);
        client.getBaseArgs().setSignViewer(true);
        return client;
    }

    private MakeRequestClient makeStrictNonSocialClient(String owner, String viewer, String gadget) throws Exception {
        SecurityToken securityToken = OAuthRequestTest.getSecurityToken(owner, viewer, gadget);
        return new MakeRequestClient(securityToken, this.fetcherConfig, this.serviceProvider, "testservice");
    }

    private MakeRequestClient makeSocialOAuthClient(String owner, String viewer, String gadget) throws Exception {
        SecurityToken securityToken = OAuthRequestTest.getSecurityToken(owner, viewer, gadget);
        MakeRequestClient client = new MakeRequestClient(securityToken, this.fetcherConfig, this.serviceProvider, "testservice");
        client.getBaseArgs().setUseToken(OAuthArguments.UseToken.IF_AVAILABLE);
        return client;
    }

    private MakeRequestClient makeSignedFetchClient(String owner, String viewer, String gadget) throws Exception {
        SecurityToken securityToken = OAuthRequestTest.getSecurityToken(owner, viewer, gadget);
        MakeRequestClient client = new MakeRequestClient(securityToken, this.fetcherConfig, this.serviceProvider, null);
        client.setBaseArgs(client.makeSignedFetchArguments());
        return client;
    }

    @Test
    public void testOAuthFlow() throws Exception {
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        this.checkEmptyLog();
    }

    @Test
    public void testOAuthFlow_withCallbackVerifier() throws Exception {
        this.fetcherConfig = new OAuthFetcherConfig((BlobCrypter)new BasicBlobCrypter("abcdefghijklmnop".getBytes()), this.getOAuthStore(this.base), (TimeSource)this.clock, this.createRealCallbackGenerator(), false);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        this.checkEmptyLog();
    }

    @Test
    public void testOAuthFlow_badCallbackVerifier() throws Exception {
        this.fetcherConfig = new OAuthFetcherConfig((BlobCrypter)new BasicBlobCrypter("abcdefghijklmnop".getBytes()), this.getOAuthStore(this.base), (TimeSource)this.clock, this.createRealCallbackGenerator(), false);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        client.setReceivedCallbackUrl("nonsense");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Assert.assertNotNull(response.getMetadata().get("oauthErrorText"));
        client.approveToken("user_data=try-again");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is try-again", (Object)response.getResponseAsString());
    }

    @Test
    public void testOAuthFlow_tokenReused() throws Exception {
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        MakeRequestClient client2 = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        response = client2.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
    }

    @Test
    public void testOAuthFlow_unauthUser() throws Exception {
        MakeRequestClient client = this.makeNonSocialClient(null, null, GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Assert.assertEquals((long)403L, (long)response.getHttpStatusCode());
        Assert.assertEquals((long)-1L, (long)response.getCacheTtl());
        Assert.assertEquals((Object)OAuthError.UNAUTHENTICATED.name(), response.getMetadata().get("oauthError"));
    }

    @Test
    public void testOAuthFlow_noViewer() throws Exception {
        for (boolean secureOwner : Arrays.asList(true, false)) {
            this.fetcherConfig = new OAuthFetcherConfig((BlobCrypter)new BasicBlobCrypter("abcdefghijklmnop".getBytes()), this.getOAuthStore(this.base), (TimeSource)this.clock, this.callbackGenerator, secureOwner);
            MakeRequestClient client = this.makeNonSocialClient("owner", null, GADGET_URL);
            HttpResponse response = client.sendGet("http://www.example.com/data");
            Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
            Assert.assertEquals((long)403L, (long)response.getHttpStatusCode());
            Assert.assertEquals((long)-1L, (long)response.getCacheTtl());
            Assert.assertEquals((Object)OAuthError.UNAUTHENTICATED.name(), response.getMetadata().get("oauthError"));
        }
    }

    @Test
    public void testOAuthFlow_noSpec() throws Exception {
        this.fetcherConfig = new OAuthFetcherConfig((BlobCrypter)new BasicBlobCrypter("abcdefghijklmnop".getBytes()), this.getOAuthStore(this.base, null), (TimeSource)this.clock, this.callbackGenerator, false);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        this.setNoSpecOptions(client);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        this.checkEmptyLog();
    }

    private void setNoSpecOptions(MakeRequestClient client) {
        client.getBaseArgs().setRequestOption("OAUTH_PROGRAMMATIC_CONFIG", "true");
        client.getBaseArgs().setRequestOption("OAUTH_PARAM_LOCATION", "uri-query");
        client.getBaseArgs().setRequestOption("OAUTH_REQUEST_METHOD", "GET");
        client.getBaseArgs().setRequestOption("OAUTH_REQUEST_TOKEN_URL", "http://www.example.com/request?param=foo");
        client.getBaseArgs().setRequestOption("OAUTH_ACCESS_TOKEN_URL", "http://www.example.com/access");
        client.getBaseArgs().setRequestOption("OAUTH_AUTHORIZATION_URL", "http://www.example.com/authorize");
    }

    @Test
    public void testOAuthFlow_noSpecNoRequestTokenUrl() throws Exception {
        this.fetcherConfig = new OAuthFetcherConfig((BlobCrypter)new BasicBlobCrypter("abcdefghijklmnop".getBytes()), this.getOAuthStore(this.base, null), (TimeSource)this.clock, null, false);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        this.setNoSpecOptions(client);
        client.getBaseArgs().removeRequestOption("OAUTH_REQUEST_TOKEN_URL");
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Assert.assertEquals((long)403L, (long)response.getHttpStatusCode());
        Assert.assertEquals((Object)OAuthError.BAD_OAUTH_TOKEN_URL.name(), response.getMetadata().get("oauthError"));
        String errorText = (String)response.getMetadata().get("oauthErrorText");
        Assert.assertNotNull((Object)errorText);
        this.checkStringContains("should report no request token url", errorText, "No request token URL specified");
    }

    @Test
    public void testOAuthFlow_noSpecNoAccessTokenUrl() throws Exception {
        this.fetcherConfig = new OAuthFetcherConfig((BlobCrypter)new BasicBlobCrypter("abcdefghijklmnop".getBytes()), this.getOAuthStore(this.base, null), (TimeSource)this.clock, this.callbackGenerator, false);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        this.setNoSpecOptions(client);
        client.getBaseArgs().removeRequestOption("OAUTH_ACCESS_TOKEN_URL");
        HttpResponse response = client.sendGet("http://www.example.com/data");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Assert.assertEquals((long)403L, (long)response.getHttpStatusCode());
        Assert.assertEquals((Object)OAuthError.BAD_OAUTH_TOKEN_URL.name(), response.getMetadata().get("oauthError"));
        String errorText = (String)response.getMetadata().get("oauthErrorText");
        Assert.assertNotNull((Object)errorText);
        this.checkStringContains("should report no access token url", errorText, "No access token URL specified");
    }

    @Test
    public void testOAuthFlow_noSpecNoApprovalUrl() throws Exception {
        this.fetcherConfig = new OAuthFetcherConfig((BlobCrypter)new BasicBlobCrypter("abcdefghijklmnop".getBytes()), this.getOAuthStore(this.base, null), (TimeSource)this.clock, this.callbackGenerator, false);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        this.setNoSpecOptions(client);
        client.getBaseArgs().removeRequestOption("OAUTH_AUTHORIZATION_URL");
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Assert.assertEquals((long)403L, (long)response.getHttpStatusCode());
        Assert.assertEquals((Object)OAuthError.BAD_OAUTH_TOKEN_URL.name(), response.getMetadata().get("oauthError"));
        String errorText = (String)response.getMetadata().get("oauthErrorText");
        Assert.assertNotNull((Object)errorText);
        this.checkStringContains("should report no authorization url", errorText, "No authorization URL specified");
    }

    @Test
    public void testOAuthFlow_noSpecAuthHeader() throws Exception {
        this.serviceProvider.setParamLocation(AccessorInfo.OAuthParamLocation.AUTH_HEADER);
        this.fetcherConfig = new OAuthFetcherConfig((BlobCrypter)new BasicBlobCrypter("abcdefghijklmnop".getBytes()), this.getOAuthStore(this.base, null), (TimeSource)this.clock, this.callbackGenerator, false);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        this.setNoSpecOptions(client);
        client.getBaseArgs().setRequestOption("OAUTH_PARAM_LOCATION", "auth-header");
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        this.checkEmptyLog();
    }

    @Test
    public void testOAuthFlow_noSpecPostBody() throws Exception {
        this.serviceProvider.setParamLocation(AccessorInfo.OAuthParamLocation.POST_BODY);
        this.fetcherConfig = new OAuthFetcherConfig((BlobCrypter)new BasicBlobCrypter("abcdefghijklmnop".getBytes()), this.getOAuthStore(this.base, null), (TimeSource)this.clock, this.callbackGenerator, false);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        this.setNoSpecOptions(client);
        client.getBaseArgs().setRequestOption("OAUTH_REQUEST_METHOD", "POST");
        client.getBaseArgs().setRequestOption("OAUTH_PARAM_LOCATION", "post-body");
        HttpResponse response = client.sendFormPost("http://www.example.com/data", "");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendFormPost("http://www.example.com/data", "");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        this.checkEmptyLog();
    }

    @Test
    public void testOAuthFlow_noSpecPostBodyAndHeader() throws Exception {
        this.serviceProvider.setParamLocation(AccessorInfo.OAuthParamLocation.POST_BODY);
        this.serviceProvider.addParamLocation(AccessorInfo.OAuthParamLocation.AUTH_HEADER);
        this.fetcherConfig = new OAuthFetcherConfig((BlobCrypter)new BasicBlobCrypter("abcdefghijklmnop".getBytes()), this.getOAuthStore(this.base, null), (TimeSource)this.clock, this.callbackGenerator, false);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        this.setNoSpecOptions(client);
        client.getBaseArgs().setRequestOption("OAUTH_REQUEST_METHOD", "POST");
        client.getBaseArgs().setRequestOption("OAUTH_PARAM_LOCATION", "post-body");
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        this.checkEmptyLog();
    }

    @Test
    public void testOAuthFlow_noSpecInvalidUrl() throws Exception {
        this.fetcherConfig = new OAuthFetcherConfig((BlobCrypter)new BasicBlobCrypter("abcdefghijklmnop".getBytes()), this.getOAuthStore(this.base, null), (TimeSource)this.clock, null, false);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        this.setNoSpecOptions(client);
        client.getBaseArgs().setRequestOption("OAUTH_REQUEST_TOKEN_URL", "foo");
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Assert.assertEquals((long)403L, (long)response.getHttpStatusCode());
        Assert.assertEquals((Object)OAuthError.INVALID_URL.name(), response.getMetadata().get("oauthError"));
        String errorText = (String)response.getMetadata().get("oauthErrorText");
        Assert.assertNotNull((Object)errorText);
        this.checkStringContains("should report invalid url", errorText, "Invalid URL: foo");
    }

    @Test
    public void testOAuthFlow_noSpecBlankUrl() throws Exception {
        this.fetcherConfig = new OAuthFetcherConfig((BlobCrypter)new BasicBlobCrypter("abcdefghijklmnop".getBytes()), this.getOAuthStore(this.base, null), (TimeSource)this.clock, null, false);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        this.setNoSpecOptions(client);
        client.getBaseArgs().setRequestOption("OAUTH_REQUEST_TOKEN_URL", "");
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Assert.assertEquals((long)403L, (long)response.getHttpStatusCode());
        Assert.assertEquals((Object)OAuthError.INVALID_URL.name(), response.getMetadata().get("oauthError"));
        String errorText = (String)response.getMetadata().get("oauthErrorText");
        Assert.assertNotNull((Object)errorText);
        this.checkStringContains("should report invalid url", errorText, "Invalid URL: ");
    }

    @Test
    public void testAccessTokenNotUsedForSocialPage() throws Exception {
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        MakeRequestClient friend = this.makeNonSocialClient("owner", "friend", GADGET_URL);
        response = friend.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Assert.assertEquals((long)403L, (long)response.getHttpStatusCode());
        Assert.assertEquals((Object)OAuthError.NOT_OWNER.name(), response.getMetadata().get("oauthError"));
    }

    @Test
    public void testAccessTokenOkForSecureOwnerPage() throws Exception {
        this.fetcherConfig = new OAuthFetcherConfig((BlobCrypter)new BasicBlobCrypter("abcdefghijklmnop".getBytes()), this.getOAuthStore(this.base), (TimeSource)this.clock, this.callbackGenerator, true);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        MakeRequestClient friend = this.makeNonSocialClient("owner", "friend", GADGET_URL);
        response = friend.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Assert.assertEquals((long)200L, (long)response.getHttpStatusCode());
    }

    @Test
    public void testParamsInHeader() throws Exception {
        this.serviceProvider.setParamLocation(AccessorInfo.OAuthParamLocation.AUTH_HEADER);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL_HEADER);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        String aznHeader = response.getHeader("X-Echoed-Authz");
        Assert.assertNotNull((Object)aznHeader);
        Assert.assertNotSame((String)("azn header: " + aznHeader), (Object)aznHeader.indexOf("OAuth"), (Object)-1);
    }

    @Test
    public void testParamsInBody() throws Exception {
        this.serviceProvider.setParamLocation(AccessorInfo.OAuthParamLocation.POST_BODY);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL_BODY);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendFormPost("http://www.example.com/data", "");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        String echoedBody = response.getHeader("X-Echoed-Body");
        Assert.assertNotNull((Object)echoedBody);
        Assert.assertNotSame((String)("body: " + echoedBody), (Object)echoedBody.indexOf("oauth_consumer_key="), (Object)-1);
    }

    @Test
    public void testParamsInBody_withExtraParams() throws Exception {
        this.serviceProvider.setParamLocation(AccessorInfo.OAuthParamLocation.POST_BODY);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL_BODY);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendFormPost("http://www.example.com/data", "foo=bar&foo=baz");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        String echoedBody = response.getHeader("X-Echoed-Body");
        Assert.assertNotNull((Object)echoedBody);
        Assert.assertNotSame((String)("body: " + echoedBody), (Object)echoedBody.indexOf("oauth_consumer_key="), (Object)-1);
        Assert.assertNotSame((String)("body: " + echoedBody), (Object)echoedBody.indexOf("foo=bar&foo=baz"), (Object)-1);
    }

    @Test
    public void testParamsInBody_forGetRequest() throws Exception {
        this.serviceProvider.setParamLocation(AccessorInfo.OAuthParamLocation.POST_BODY);
        this.serviceProvider.addParamLocation(AccessorInfo.OAuthParamLocation.AUTH_HEADER);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL_BODY);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        String aznHeader = response.getHeader("X-Echoed-Authz");
        Assert.assertNotNull((Object)aznHeader);
        Assert.assertNotSame((String)("azn header: " + aznHeader), (Object)aznHeader.indexOf("OAuth"), (Object)-1);
    }

    @Test
    public void testParamsInBody_forGetRequestStrictSp() throws Exception {
        this.serviceProvider.setParamLocation(AccessorInfo.OAuthParamLocation.POST_BODY);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL_BODY);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Assert.assertEquals((long)403L, (long)response.getHttpStatusCode());
        Assert.assertEquals((Object)"parameter_absent", response.getMetadata().get("oauthError"));
        Assert.assertNull(response.getMetadata().get("oauthApprovalUrl"));
    }

    @Test
    public void testRevokedAccessToken() throws Exception {
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data?cachebust=1");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        this.serviceProvider.revokeAllAccessTokens();
        response = client.sendGet("http://www.example.com/data?cachebust=2");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Assert.assertNotNull(response.getMetadata().get("oauthApprovalUrl"));
        Assert.assertNull((String)"Should not return oauthError for revoked token", response.getMetadata().get("oauthError"));
        String errorText = (String)response.getMetadata().get("oauthErrorText");
        Assert.assertNotNull((Object)errorText);
        this.checkStringContains("should return original request", errorText, "GET /data?cachebust=2\n");
        this.checkStringContains("should return signed request", errorText, "GET /data?cachebust=2&");
        this.checkStringContains("should remove secret", errorText, "oauth_token_secret=REMOVED");
        this.checkStringContains("should return response", errorText, "HTTP/1.1 401");
        this.checkStringContains("should return response", errorText, "oauth_problem=\"token_revoked\"");
        client.approveToken("user_data=reapproved");
        response = client.sendGet("http://www.example.com/data?cachebust=3");
        Assert.assertEquals((Object)"User data is reapproved", (Object)response.getResponseAsString());
    }

    @Test
    public void testError401() throws Exception {
        this.serviceProvider.setVagueErrors(true);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data?cachebust=1");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        this.serviceProvider.revokeAllAccessTokens();
        response = client.sendGet("http://www.example.com/data?cachebust=2");
        this.checkLogContains("GET /data?cachebust=2");
        this.checkLogContains("HTTP/1.1 401");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Assert.assertNotNull(response.getMetadata().get("oauthApprovalUrl"));
        client.approveToken("user_data=reapproved");
        response = client.sendGet("http://www.example.com/data?cachebust=3");
        Assert.assertEquals((Object)"User data is reapproved", (Object)response.getResponseAsString());
    }

    @Test
    public void testUnknownConsumerKey() throws Exception {
        SecurityToken securityToken = OAuthRequestTest.getSecurityToken("owner", "owner", GADGET_URL_NO_KEY);
        MakeRequestClient client = new MakeRequestClient(securityToken, this.fetcherConfig, this.serviceProvider, "nokey");
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Map metadata = response.getMetadata();
        Assert.assertNotNull((Object)metadata);
        Assert.assertEquals((Object)"consumer_key_unknown", metadata.get("oauthError"));
        String errorText = (String)response.getMetadata().get("oauthErrorText");
        this.checkStringContains("oauthErrorText mismatch", errorText, "Service provider rejected request");
        this.checkStringContains("oauthErrorText mismatch", errorText, "oauth_problem_advice=\"invalid%20consumer%3A%20garbage_key\"");
        this.checkStringContains("should return original request", errorText, "GET /data\n");
        this.checkStringContains("should return request token request", errorText, "GET /request?param=foo&");
    }

    @Test
    public void testBrokenRequestTokenResponse() throws Exception {
        SecurityToken securityToken = OAuthRequestTest.getSecurityToken("owner", "owner", GADGET_URL_BAD_OAUTH_URL);
        MakeRequestClient client = new MakeRequestClient(securityToken, this.fetcherConfig, this.serviceProvider, "testservice");
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((long)403L, (long)response.getHttpStatusCode());
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Map metadata = response.getMetadata();
        Assert.assertNotNull((Object)metadata);
        Assert.assertEquals((Object)"MISSING_OAUTH_PARAMETER", metadata.get("oauthError"));
        String errorText = (String)response.getMetadata().get("oauthErrorText");
        this.checkStringContains("oauthErrorText mismatch", errorText, "No oauth_token returned from service provider");
        this.checkStringContains("oauthErrorText mismatch", errorText, "GET /echo?mary_had_a_little_lamb");
    }

    @Test
    public void testBrokenAccessTokenResponse() throws Exception {
        SecurityToken securityToken = OAuthRequestTest.getSecurityToken("owner", "owner", GADGET_URL_BAD_OAUTH_URL);
        MakeRequestClient client = new MakeRequestClient(securityToken, this.fetcherConfig, this.serviceProvider, "testservice");
        client.getBaseArgs().setRequestToken("reqtoken");
        client.getBaseArgs().setRequestTokenSecret("reqtokensecret");
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((long)403L, (long)response.getHttpStatusCode());
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Map metadata = response.getMetadata();
        Assert.assertNotNull((Object)metadata);
        Assert.assertEquals((Object)"MISSING_OAUTH_PARAMETER", metadata.get("oauthError"));
        String errorText = (String)response.getMetadata().get("oauthErrorText");
        this.checkStringContains("oauthErrorText mismatch", errorText, "No oauth_token_secret returned from service provider");
        this.checkStringContains("oauthErrorText mismatch", errorText, "with_fleece_as_white_as_snow");
    }

    @Test
    public void testExtraApprovalParams() throws Exception {
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL_APPROVAL_PARAMS);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        String approvalUrl = (String)response.getMetadata().get("oauthApprovalUrl");
        Assert.assertSame((String)approvalUrl, (Object)0, (Object)approvalUrl.indexOf("http://www.example.com/authorize?oauth_callback=foo&oauth_token="));
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        this.checkEmptyLog();
    }

    @Test
    public void testError403() throws Exception {
        this.serviceProvider.setVagueErrors(true);
        SecurityToken securityToken = OAuthRequestTest.getSecurityToken("owner", "owner", GADGET_URL_NO_KEY);
        MakeRequestClient client = new MakeRequestClient(securityToken, this.fetcherConfig, this.serviceProvider, "nokey");
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Map metadata = response.getMetadata();
        Assert.assertNotNull((Object)metadata);
        Assert.assertEquals((Object)"MISSING_OAUTH_PARAMETER", metadata.get("oauthError"));
        this.checkStringContains("oauthErrorText mismatch", (String)metadata.get("oauthErrorText"), "some vague error");
        this.checkStringContains("oauthErrorText mismatch", (String)metadata.get("oauthErrorText"), "HTTP/1.1 403");
        this.checkLogContains("HTTP/1.1 403");
        this.checkLogContains("GET /request");
        this.checkLogContains("some vague error");
    }

    @Test
    public void testError404() throws Exception {
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data?cachebust=1");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        response = client.sendGet("http://www.example.com/404");
        Assert.assertEquals((Object)"not found", (Object)response.getResponseAsString());
        Assert.assertEquals((long)404L, (long)response.getHttpStatusCode());
        response = client.sendGet("http://www.example.com/data?cachebust=3");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
    }

    @Test
    public void testError400() throws Exception {
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data?cachebust=1");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        response = client.sendGet("http://www.example.com/400");
        Assert.assertEquals((Object)"bad request", (Object)response.getResponseAsString());
        Assert.assertEquals((long)400L, (long)response.getHttpStatusCode());
        response = client.sendGet("http://www.example.com/data?cachebust=3");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
    }

    @Test
    public void testConsumerThrottled() throws Exception {
        Assert.assertEquals((long)0L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)0L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)0L, (long)this.serviceProvider.getResourceAccessCount());
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)0L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)0L, (long)this.serviceProvider.getResourceAccessCount());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getResourceAccessCount());
        response = client.sendGet("http://www.example.com/data?cachebust=1");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getResourceAccessCount());
        this.serviceProvider.setConsumersThrottled(true);
        response = client.sendGet("http://www.example.com/data?cachebust=2");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Map metadata = response.getMetadata();
        Assert.assertNotNull((Object)metadata);
        Assert.assertEquals((Object)"consumer_key_refused", metadata.get("oauthError"));
        this.checkStringContains("oauthErrorText mismatch", (String)metadata.get("oauthErrorText"), "Service provider rejected request");
        this.checkStringContains("oauthErrorText missing request entry", (String)metadata.get("oauthErrorText"), "GET /data?cachebust=2\n");
        this.checkStringContains("oauthErrorText missing request entry", (String)metadata.get("oauthErrorText"), "GET /data?cachebust=2&oauth_body_hash=2jm");
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)3L, (long)this.serviceProvider.getResourceAccessCount());
        this.serviceProvider.setConsumersThrottled(false);
        client.clearState();
        response = client.sendGet("http://www.example.com/data?cachebust=3");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)4L, (long)this.serviceProvider.getResourceAccessCount());
    }

    @Test
    public void testConsumerThrottled_vagueErrors() throws Exception {
        this.serviceProvider.setVagueErrors(true);
        Assert.assertEquals((long)0L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)0L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)0L, (long)this.serviceProvider.getResourceAccessCount());
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)0L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)0L, (long)this.serviceProvider.getResourceAccessCount());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getResourceAccessCount());
        response = client.sendGet("http://www.example.com/data?cachebust=1");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getResourceAccessCount());
        this.serviceProvider.setConsumersThrottled(true);
        response = client.sendGet("http://www.example.com/data?cachebust=2");
        Assert.assertEquals((long)403L, (long)response.getHttpStatusCode());
        Assert.assertEquals((Object)"some vague error", (Object)response.getResponseAsString());
        Map metadata = response.getMetadata();
        Assert.assertNotNull((Object)metadata);
        Assert.assertNull(metadata.get("oauthError"));
        this.checkStringContains("oauthErrorText missing request entry", (String)metadata.get("oauthErrorText"), "GET /data?cachebust=2\n");
        this.checkStringContains("oauthErrorText missing request entry", (String)metadata.get("oauthErrorText"), "GET /data?cachebust=2&oauth_body_hash=2jm");
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)3L, (long)this.serviceProvider.getResourceAccessCount());
        this.serviceProvider.setConsumersThrottled(false);
        client.clearState();
        response = client.sendGet("http://www.example.com/data?cachebust=3");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)4L, (long)this.serviceProvider.getResourceAccessCount());
    }

    @Test
    public void testSocialOAuth_tokenRevoked() throws Exception {
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        this.serviceProvider.revokeAllAccessTokens();
        Assert.assertEquals((long)0L, (long)this.base.getAccessTokenRemoveCount());
        client = this.makeSocialOAuthClient("owner", "owner", GADGET_URL);
        response = client.sendGet("http://www.example.com/data?cb=1");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.base.getAccessTokenRemoveCount());
    }

    @Test
    public void testWrongServiceName() throws Exception {
        SecurityToken securityToken = OAuthRequestTest.getSecurityToken("owner", "owner", GADGET_URL);
        MakeRequestClient client = new MakeRequestClient(securityToken, this.fetcherConfig, this.serviceProvider, "nosuchservice");
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Map metadata = response.getMetadata();
        Assert.assertNull(metadata.get("oauthApprovalUrl"));
        Assert.assertEquals((Object)"BAD_OAUTH_CONFIGURATION", metadata.get("oauthError"));
        String errorText = (String)metadata.get("oauthErrorText");
        Assert.assertTrue((String)errorText, (boolean)errorText.startsWith("Failed to retrieve OAuth URLs, spec for gadget does not contain OAuth service nosuchservice.  Known services: testservice"));
    }

    @Test
    public void testPreapprovedToken() throws Exception {
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        FakeOAuthServiceProvider.TokenPair reqToken = this.serviceProvider.getPreapprovedToken("preapproved");
        client.getBaseArgs().setRequestToken(reqToken.token);
        client.getBaseArgs().setRequestTokenSecret(reqToken.secret);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is preapproved", (Object)response.getResponseAsString());
        Assert.assertEquals((long)0L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getResourceAccessCount());
        response = client.sendGet("http://www.example.com/data?cachebust=1");
        Assert.assertEquals((Object)"User data is preapproved", (Object)response.getResponseAsString());
        Assert.assertEquals((long)0L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getResourceAccessCount());
        response = client.sendGet("http://www.example.com/data?cachebust=2");
        Assert.assertEquals((Object)"User data is preapproved", (Object)response.getResponseAsString());
        Assert.assertEquals((long)0L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)3L, (long)this.serviceProvider.getResourceAccessCount());
    }

    @Test
    public void testPreapprovedToken_invalid() throws Exception {
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        client.getBaseArgs().setRequestToken("garbage");
        client.getBaseArgs().setRequestTokenSecret("garbage");
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)0L, (long)this.serviceProvider.getResourceAccessCount());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getResourceAccessCount());
    }

    @Test
    public void testPreapprovedToken_notUsedIfAccessTokenExists() throws Exception {
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        FakeOAuthServiceProvider.TokenPair reqToken = this.serviceProvider.getPreapprovedToken("preapproved");
        client.getBaseArgs().setRequestToken(reqToken.token);
        client.getBaseArgs().setRequestTokenSecret(reqToken.secret);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is preapproved", (Object)response.getResponseAsString());
        Assert.assertEquals((long)0L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getResourceAccessCount());
        MakeRequestClient client2 = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        response = client2.sendGet("http://www.example.com/data?cachebust=1");
        Assert.assertEquals((Object)"User data is preapproved", (Object)response.getResponseAsString());
        Assert.assertEquals((long)0L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getResourceAccessCount());
    }

    @Test
    public void testSignedFetchParametersSet() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        HttpResponse resp = client.sendGet("http://www.example.com/data");
        List queryParams = OAuth.decodeForm((String)resp.getResponseAsString());
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_owner_id", "o"));
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_viewer_id", "v"));
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_app_id", "app"));
        Assert.assertTrue((boolean)this.contains(queryParams, "oauth_consumer_key", "signedfetch"));
        Assert.assertTrue((boolean)this.contains(queryParams, "xoauth_signature_publickey", "foo"));
        Assert.assertTrue((boolean)this.contains(queryParams, "xoauth_public_key", "foo"));
        Assert.assertFalse((boolean)this.contains(queryParams, "opensocial_proxied_content", "1"));
    }

    @Test
    public void testSignedFetch_authHeader() throws Exception {
        this.serviceProvider.setParamLocation(AccessorInfo.OAuthParamLocation.AUTH_HEADER);
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        client.getBaseArgs().setRequestOption("OAUTH_PROGRAMMATIC_CONFIG", "true");
        client.getBaseArgs().setRequestOption("OAUTH_PARAM_LOCATION", "auth-header");
        HttpResponse resp = client.sendGet("http://www.example.com/data");
        String auth = resp.getHeader("X-Echoed-Authz");
        Assert.assertNotNull((String)"Should have echoed authz header", (Object)auth);
        this.checkStringContains("should have opensocial params in header", auth, "opensocial_owner_id=\"o\"");
    }

    @Test
    public void testSignedFetchParametersSetProxiedContent() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        client.getBaseArgs().setProxiedContentRequest(true);
        HttpResponse resp = client.sendGet("http://www.example.com/data");
        List queryParams = OAuth.decodeForm((String)resp.getResponseAsString());
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_owner_id", "o"));
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_viewer_id", "v"));
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_app_id", "app"));
        Assert.assertTrue((boolean)this.contains(queryParams, "oauth_consumer_key", "signedfetch"));
        Assert.assertTrue((boolean)this.contains(queryParams, "xoauth_signature_publickey", "foo"));
        Assert.assertTrue((boolean)this.contains(queryParams, "xoauth_public_key", "foo"));
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_proxied_content", "1"));
    }

    @Test
    public void testPostBinaryData() throws Exception {
        byte[] raw = new byte[]{0, 1, 2, 3, 4, 5};
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        HttpResponse resp = client.sendRawPost("http://www.example.com/data", null, raw);
        List queryParams = OAuth.decodeForm((String)resp.getResponseAsString());
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_owner_id", "o"));
        Assert.assertTrue((boolean)this.contains(queryParams, "oauth_consumer_key", "signedfetch"));
        String echoed = resp.getHeader("X-Echoed-Raw-Body");
        byte[] echoedBytes = Base64.decodeBase64((byte[])CharsetUtil.getUtf8Bytes((String)echoed));
        Assert.assertTrue((boolean)Arrays.equals(raw, echoedBytes));
    }

    @Test
    public void testPostWeirdContentType() throws Exception {
        byte[] raw = new byte[]{0, 1, 2, 3, 4, 5};
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        HttpResponse resp = client.sendRawPost("http://www.example.com/data", "funky-content", raw);
        List queryParams = OAuth.decodeForm((String)resp.getResponseAsString());
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_owner_id", "o"));
        Assert.assertTrue((boolean)this.contains(queryParams, "oauth_consumer_key", "signedfetch"));
        String echoed = resp.getHeader("X-Echoed-Raw-Body");
        byte[] echoedBytes = Base64.decodeBase64((byte[])CharsetUtil.getUtf8Bytes((String)echoed));
        Assert.assertTrue((boolean)Arrays.equals(raw, echoedBytes));
    }

    @Test
    public void testGetWithFormEncodedBody() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        HttpResponse resp = client.sendGetWithBody("http://www.example.com/data", "application/x-www-form-urlencoded", "war=peace&yes=no".getBytes());
        Assert.assertEquals((Object)"war=peace&yes=no", (Object)resp.getHeader("X-Echoed-Body"));
    }

    @Test
    public void testGetWithRawBody() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        HttpResponse resp = client.sendGetWithBody("http://www.example.com/data", "application/json", "war=peace&yes=no".getBytes());
        Assert.assertEquals((Object)"war=peace&yes=no", (Object)resp.getHeader("X-Echoed-Body"));
        List queryParams = OAuth.decodeForm((String)resp.getResponseAsString());
        this.checkContains(queryParams, "oauth_body_hash", "MfhwxPN6ns5CwQAZN9OcJXu3Jv4=");
    }

    @Test
    public void testGetTamperedRawContent() throws Exception {
        byte[] raw = new byte[]{0, 1, 2, 3, 4, 5};
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        client.setNextFetcher(new HttpFetcher(){

            public HttpResponse fetch(HttpRequest request) throws GadgetException {
                request.setPostBody("yo momma".getBytes());
                return OAuthRequestTest.this.serviceProvider.fetch(request);
            }
        });
        try {
            client.sendGetWithBody("http://www.example.com/data", "funky-content", raw);
            Assert.fail((String)"Should have thrown with oauth_body_hash mismatch");
        }
        catch (RuntimeException runtimeException) {
            // empty catch block
        }
    }

    @Test(expected=RuntimeException.class)
    public void testGetTamperedFormContent() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        client.setNextFetcher(new HttpFetcher(){

            public HttpResponse fetch(HttpRequest request) throws GadgetException {
                request.setPostBody("foo=quux".getBytes());
                return OAuthRequestTest.this.serviceProvider.fetch(request);
            }
        });
        client.sendGetWithBody("http://www.example.com/data", "application/x-www-form-urlencoded", "foo=bar".getBytes());
        Assert.fail((String)"Should have thrown with oauth signature mismatch");
    }

    @Test(expected=RuntimeException.class)
    public void testGetTamperedRemoveRawContent() throws Exception {
        byte[] raw = new byte[]{0, 1, 2, 3, 4, 5};
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        client.setNextFetcher(new HttpFetcher(){

            public HttpResponse fetch(HttpRequest request) throws GadgetException {
                request.setPostBody(ArrayUtils.EMPTY_BYTE_ARRAY);
                request.setHeader("Content-Type", "application/x-www-form-urlencoded");
                return OAuthRequestTest.this.serviceProvider.fetch(request);
            }
        });
        client.sendGetWithBody("http://www.example.com/data", "funky-content", raw);
        Assert.fail((String)"Should have thrown with body hash in form encoded request");
    }

    @Test(expected=RuntimeException.class)
    public void testPostTamperedRawContent() throws Exception {
        byte[] raw = new byte[]{0, 1, 2, 3, 4, 5};
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        client.setNextFetcher(new HttpFetcher(){

            public HttpResponse fetch(HttpRequest request) throws GadgetException {
                request.setPostBody("yo momma".getBytes());
                return OAuthRequestTest.this.serviceProvider.fetch(request);
            }
        });
        client.sendRawPost("http://www.example.com/data", "funky-content", raw);
        Assert.fail((String)"Should have thrown with oauth_body_hash mismatch");
    }

    @Test(expected=RuntimeException.class)
    public void testPostTamperedFormContent() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        client.setNextFetcher(new HttpFetcher(){

            public HttpResponse fetch(HttpRequest request) throws GadgetException {
                request.setPostBody("foo=quux".getBytes());
                return OAuthRequestTest.this.serviceProvider.fetch(request);
            }
        });
        client.sendFormPost("http://www.example.com/data", "foo=bar");
        Assert.fail((String)"Should have thrown with oauth signature mismatch");
    }

    @Test(expected=RuntimeException.class)
    public void testPostTamperedRemoveRawContent() throws Exception {
        byte[] raw = new byte[]{0, 1, 2, 3, 4, 5};
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        client.setNextFetcher(new HttpFetcher(){

            public HttpResponse fetch(HttpRequest request) throws GadgetException {
                request.setPostBody(ArrayUtils.EMPTY_BYTE_ARRAY);
                request.setHeader("Content-Type", "application/x-www-form-urlencoded");
                return OAuthRequestTest.this.serviceProvider.fetch(request);
            }
        });
        client.sendRawPost("http://www.example.com/data", "funky-content", raw);
        Assert.fail((String)"Should have thrown with body hash in form encoded request");
    }

    @Test
    public void testSignedFetch_error401() throws Exception {
        Assert.assertEquals((long)0L, (long)this.base.getAccessTokenRemoveCount());
        this.serviceProvider.setConsumerUnauthorized(true);
        this.serviceProvider.setVagueErrors(true);
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertNull(response.getMetadata().get("oauthError"));
        String errorText = (String)response.getMetadata().get("oauthErrorText");
        this.checkStringContains("Should return sent request", errorText, "GET /data");
        this.checkStringContains("Should return response", errorText, "HTTP/1.1 401");
        this.checkStringContains("Should return response", errorText, "some vague error");
        Assert.assertEquals((long)0L, (long)this.base.getAccessTokenRemoveCount());
    }

    @Test
    public void testSignedFetch_error403() throws Exception {
        Assert.assertEquals((long)0L, (long)this.base.getAccessTokenRemoveCount());
        this.serviceProvider.setConsumersThrottled(true);
        this.serviceProvider.setVagueErrors(true);
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertNull(response.getMetadata().get("oauthError"));
        String errorText = (String)response.getMetadata().get("oauthErrorText");
        this.checkStringContains("Should return sent request", errorText, "GET /data");
        this.checkStringContains("Should return response", errorText, "HTTP/1.1 403");
        this.checkStringContains("Should return response", errorText, "some vague error");
        Assert.assertEquals((long)0L, (long)this.base.getAccessTokenRemoveCount());
    }

    @Test
    public void testSignedFetch_unnamedConsumerKey() throws Exception {
        BasicOAuthStoreConsumerKeyAndSecret defaultKey = new BasicOAuthStoreConsumerKeyAndSecret(null, "MIICdgIBADANBgkqhkiG9w0BAQEFAASCAmAwggJcAgEAAoGBALRiMLAh9iimur8VA7qVvdqxevEuUkW4K+2KdMXmnQbG9Aa7k7eBjK1S+0LYmVjPKlJGNXHDGuy5Fw/d7rjVJ0BLB+ubPK8iA/Tw3hLQgXMRRGRXXCn8ikfuQfjUS1uZSatdLB81mydBETlJhI6GH4twrbDJCR2Bwy/XWXgqgGRzAgMBAAECgYBYWVtleUzavkbrPjy0T5FMou8HX9u2AC2ry8vD/l7cqedtwMPp9k7TubgNFo+NGvKsl2ynyprOZR1xjQ7WgrgVB+mmuScOM/5HVceFuGRDhYTCObE+y1kxRloNYXnx3ei1zbeYLPCHdhxRYW7T0qcynNmwrn05/KO2RLjgQNalsQJBANeA3Q4Nugqy4QBUCEC09SqylT2K9FrrItqL2QKc9v0ZzO2uwllCbg0dwpVuYPYXYvikNHHg+aCWF+VXsb9rpPsCQQDWR9TT4ORdzoj+NccnqkMsDmzt0EfNaAOwHOmVJ2RVBspPcxt5iN4HI7HNeG6U5YsFBb+/GZbgfBT3kpNGWPTpAkBI+gFhjfJvRw38n3g/+UeAkwMI2TJQS4n8+hid0uus3/zOjDySH3XHCUnocn1xOJAyZODBo47E+67R4jV1/gzbAkEAklJaspRPXP877NssM5nAZMU0/O/NGCZ+3jPgDUno6WbJn5cqm8MqWhW1xGkImgRk+fkDBquiq4gPiT898jusgQJAd5Zrr6Q8AO/0isr/3aa6O6NLQxISLKcPDk2NOccAfS/xOtfOz4sJYM3+Bs4Io9+dZGSDCA54Lw03eHTNQghS0A==", BasicOAuthStoreConsumerKeyAndSecret.KeyType.RSA_PRIVATE, "foo", null);
        this.base.setDefaultKey(defaultKey);
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        HttpResponse resp = client.sendGet("http://www.example.com/data");
        List queryParams = OAuth.decodeForm((String)resp.getResponseAsString());
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_owner_id", "o"));
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_viewer_id", "v"));
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_app_id", "app"));
        Assert.assertTrue((boolean)this.contains(queryParams, "oauth_consumer_key", "container.com"));
        Assert.assertTrue((boolean)this.contains(queryParams, "xoauth_signature_publickey", "foo"));
        Assert.assertTrue((boolean)this.contains(queryParams, "xoauth_public_key", "foo"));
    }

    @Test
    public void testSignedFetch_extraQueryParameters() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        HttpResponse resp = client.sendGet("http://www.example.com/data?foo=bar&foo=baz");
        List queryParams = OAuth.decodeForm((String)resp.getResponseAsString());
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_owner_id", "o"));
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_viewer_id", "v"));
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_app_id", "app"));
        Assert.assertTrue((boolean)this.contains(queryParams, "oauth_consumer_key", "signedfetch"));
        Assert.assertTrue((boolean)this.contains(queryParams, "xoauth_signature_publickey", "foo"));
        Assert.assertTrue((boolean)this.contains(queryParams, "xoauth_public_key", "foo"));
    }

    @Test
    public void testNoSignViewer() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        client.getBaseArgs().setSignViewer(false);
        HttpResponse resp = client.sendGet("http://www.example.com/data");
        List queryParams = OAuth.decodeForm((String)resp.getResponseAsString());
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_owner_id", "o"));
        Assert.assertFalse((boolean)this.contains(queryParams, "opensocial_viewer_id", "v"));
    }

    @Test
    public void testNoSignOwner() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        client.getBaseArgs().setSignOwner(false);
        HttpResponse resp = client.sendGet("http://www.example.com/data");
        List queryParams = OAuth.decodeForm((String)resp.getResponseAsString());
        Assert.assertFalse((boolean)this.contains(queryParams, "opensocial_owner_id", "o"));
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_viewer_id", "v"));
    }

    @Test
    public void testTrickyParametersInQuery() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        String tricky = "%6fpensocial_owner_id=gotcha";
        HttpResponse resp = client.sendGet("http://www.example.com/data?" + tricky);
        Assert.assertEquals((Object)OAuthError.INVALID_PARAMETER.name(), resp.getMetadata().get("oauthError"));
        this.checkStringContains("Wrong error text", (String)resp.getMetadata().get("oauthErrorText"), "Invalid parameter name opensocial_owner_id, applications may not override oauth, xoauth, or opensocial parameters");
    }

    @Test
    public void testTrickyParametersInBody() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        String tricky = "%6fpensocial_owner_id=gotcha";
        HttpResponse resp = client.sendFormPost("http://www.example.com/data", tricky);
        Assert.assertEquals((Object)OAuthError.INVALID_PARAMETER.name(), resp.getMetadata().get("oauthError"));
        this.checkStringContains("Wrong error text", (String)resp.getMetadata().get("oauthErrorText"), "Invalid parameter name opensocial_owner_id, applications may not override oauth, xoauth, or opensocial parameters");
    }

    @Test
    public void testGetNoQuery() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        HttpResponse resp = client.sendGet("http://www.example.com/data");
        List queryParams = OAuth.decodeForm((String)resp.getResponseAsString());
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_owner_id", "o"));
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_viewer_id", "v"));
    }

    @Test
    public void testGetWithQuery() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        HttpResponse resp = client.sendGet("http://www.example.com/data?a=b");
        List queryParams = OAuth.decodeForm((String)resp.getResponseAsString());
        Assert.assertTrue((boolean)this.contains(queryParams, "a", "b"));
    }

    @Test
    public void testGetWithQueryMultiParam() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        HttpResponse resp = client.sendGet("http://www.example.com/data?a=b&a=c");
        List queryParams = OAuth.decodeForm((String)resp.getResponseAsString());
        Assert.assertTrue((boolean)this.contains(queryParams, "a", "b"));
        Assert.assertTrue((boolean)this.contains(queryParams, "a", "c"));
    }

    @Test
    public void testValidParameterCharacters() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        String weird = "~!@$*()-_[]:,./";
        HttpResponse resp = client.sendGet("http://www.example.com/data?" + weird + "=foo");
        List queryParams = OAuth.decodeForm((String)resp.getResponseAsString());
        Assert.assertTrue((boolean)this.contains(queryParams, weird, "foo"));
    }

    @Test
    public void testPostNoQueryNoData() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        HttpResponse resp = client.sendFormPost("http://www.example.com/data", null);
        List queryParams = OAuth.decodeForm((String)resp.getResponseAsString());
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_owner_id", "o"));
        Assert.assertEquals((Object)"", (Object)resp.getHeader("X-Echoed-Body"));
    }

    @Test
    public void testPostWithQueryNoData() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        HttpResponse resp = client.sendFormPost("http://www.example.com/data?name=value", null);
        List queryParams = OAuth.decodeForm((String)resp.getResponseAsString());
        Assert.assertTrue((boolean)this.contains(queryParams, "name", "value"));
        Assert.assertEquals((Object)"", (Object)resp.getHeader("X-Echoed-Body"));
    }

    @Test
    public void testPostNoQueryWithData() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        HttpResponse resp = client.sendFormPost("http://www.example.com/data", "name=value");
        List queryParams = OAuth.decodeForm((String)resp.getResponseAsString());
        Assert.assertFalse((boolean)this.contains(queryParams, "name", "value"));
        Assert.assertEquals((Object)"name=value", (Object)resp.getHeader("X-Echoed-Body"));
    }

    @Test
    public void testPostWithQueryWithData() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        HttpResponse resp = client.sendFormPost("http://www.example.com/data?queryName=queryValue", "name=value");
        List queryParams = OAuth.decodeForm((String)resp.getResponseAsString());
        Assert.assertTrue((boolean)this.contains(queryParams, "queryName", "queryValue"));
        Assert.assertEquals((Object)"name=value", (Object)resp.getHeader("X-Echoed-Body"));
    }

    @Test
    public void testStripOpenSocialParamsFromQuery() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        HttpResponse resp = client.sendFormPost("http://www.example.com/data?opensocial_foo=bar", null);
        Assert.assertEquals((Object)OAuthError.INVALID_PARAMETER.name(), resp.getMetadata().get("oauthError"));
        this.checkStringContains("Wrong error text", (String)resp.getMetadata().get("oauthErrorText"), "Invalid parameter name opensocial_foo");
    }

    @Test
    public void testStripOAuthParamsFromQuery() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        HttpResponse resp = client.sendFormPost("http://www.example.com/data?oauth_foo=bar", "name=value");
        Assert.assertEquals((Object)OAuthError.INVALID_PARAMETER.name(), resp.getMetadata().get("oauthError"));
        this.checkStringContains("Wrong error text", (String)resp.getMetadata().get("oauthErrorText"), "Invalid parameter name oauth_foo");
    }

    @Test
    public void testStripOpenSocialParamsFromBody() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        HttpResponse resp = client.sendFormPost("http://www.example.com/data", "opensocial_foo=bar");
        Assert.assertEquals((Object)OAuthError.INVALID_PARAMETER.name(), resp.getMetadata().get("oauthError"));
        this.checkStringContains("Wrong error text", (String)resp.getMetadata().get("oauthErrorText"), "Invalid parameter name opensocial_foo");
    }

    @Test
    public void testStripOAuthParamsFromBody() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        HttpResponse resp = client.sendFormPost("http://www.example.com/data", "oauth_foo=bar");
        Assert.assertEquals((Object)OAuthError.INVALID_PARAMETER.name(), resp.getMetadata().get("oauthError"));
        this.checkStringContains("Wrong error text", (String)resp.getMetadata().get("oauthErrorText"), "Invalid parameter name oauth_foo");
    }

    @Test
    public void testAccessTokenExpires_onClient() throws Exception {
        this.serviceProvider.setSessionExtension(true);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getResourceAccessCount());
        this.clock.incrementSeconds(61);
        response = client.sendGet("http://www.example.com/data?cb=1");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getResourceAccessCount());
        response = client.sendGet("http://www.example.com/data?cb=3");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)3L, (long)this.serviceProvider.getResourceAccessCount());
        this.clock.incrementSeconds(61);
        response = client.sendGet("http://www.example.com/data?cb=4");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)3L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)4L, (long)this.serviceProvider.getResourceAccessCount());
        this.checkEmptyLog();
    }

    @Test
    public void testAccessTokenExpires_onClientNoPredictedExpiration() throws Exception {
        this.serviceProvider.setSessionExtension(true);
        this.serviceProvider.setReportExpirationTimes(false);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getResourceAccessCount());
        this.clock.incrementSeconds(61);
        response = client.sendGet("http://www.example.com/data?cb=1");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)3L, (long)this.serviceProvider.getResourceAccessCount());
        response = client.sendGet("http://www.example.com/data?cb=3");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)4L, (long)this.serviceProvider.getResourceAccessCount());
        this.clock.incrementSeconds(61);
        response = client.sendGet("http://www.example.com/data?cb=4");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)3L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)6L, (long)this.serviceProvider.getResourceAccessCount());
    }

    @Test
    public void testAccessTokenExpires_onServer() throws Exception {
        this.serviceProvider.setSessionExtension(true);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getResourceAccessCount());
        client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        this.clock.incrementSeconds(61);
        response = client.sendGet("http://www.example.com/data?cb=1");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getResourceAccessCount());
    }

    @Test
    public void testAccessTokenExpired_andRevoked() throws Exception {
        this.serviceProvider.setSessionExtension(true);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getResourceAccessCount());
        this.clock.incrementSeconds(61);
        this.serviceProvider.revokeAllAccessTokens();
        response = client.sendGet("http://www.example.com/data?cb=1");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getResourceAccessCount());
        client.approveToken("user_data=renewed");
        response = client.sendGet("http://www.example.com/data?cb=1");
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)3L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getResourceAccessCount());
        Assert.assertEquals((Object)"User data is renewed", (Object)response.getResponseAsString());
        this.checkLogContains("oauth_token_secret=REMOVED");
    }

    @Test
    public void testBadSessionHandle() throws Exception {
        this.serviceProvider.setSessionExtension(true);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getResourceAccessCount());
        this.clock.incrementSeconds(61);
        this.serviceProvider.changeAllSessionHandles();
        response = client.sendGet("http://www.example.com/data?cb=1");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getResourceAccessCount());
        client.approveToken("user_data=renewed");
        response = client.sendGet("http://www.example.com/data?cb=1");
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)3L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getResourceAccessCount());
        Assert.assertEquals((Object)"User data is renewed", (Object)response.getResponseAsString());
        this.checkLogContains("oauth_session_handle=REMOVED");
    }

    @Test
    public void testExtraParamsRejected() throws Exception {
        this.serviceProvider.setRejectExtraParams(true);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"parameter_rejected", response.getMetadata().get("oauthError"));
    }

    @Test
    public void testExtraParamsSuppressed() throws Exception {
        this.serviceProvider.setRejectExtraParams(true);
        MakeRequestClient client = this.makeStrictNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
    }

    @Test
    public void testCanRetrieveAccessTokenData() throws Exception {
        this.serviceProvider.setReturnAccessTokenData(true);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/access");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/access");
        Assert.assertEquals((Object)"application/json; charset=UTF-8", (Object)response.getHeader("Content-Type"));
        JSONObject json = new JSONObject(response.getResponseAsString());
        Assert.assertEquals((Object)"userid value", (Object)json.get("userid"));
        Assert.assertEquals((Object)"xoauth_stuff value", (Object)json.get("xoauth_stuff"));
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
    }

    @Test
    public void testAccessTokenData_noOAuthParams() throws Exception {
        this.serviceProvider.setReturnAccessTokenData(true);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/access");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/access");
        JSONObject json = new JSONObject(response.getResponseAsString());
        Assert.assertEquals((Object)"userid value", (Object)json.get("userid"));
        Assert.assertEquals((Object)"xoauth_stuff value", (Object)json.get("xoauth_stuff"));
        Assert.assertEquals((long)2L, (long)json.length());
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
    }

    @Test(expected=RuntimeException.class)
    public void testAccessTokenData_noDirectRequest() throws Exception {
        this.serviceProvider.setReturnAccessTokenData(true);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/access");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        client.sendGet("http://www.example.com/access");
        Assert.fail((String)"Service provider should have rejected bogus request to access token URL");
    }

    @Test
    public void testNextFetchReturnsNull() throws Exception {
        this.serviceProvider.setReturnNull(true);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/access");
        Assert.assertEquals((Object)"MISSING_SERVER_RESPONSE", response.getMetadata().get("oauthError"));
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        String oauthErrorText = (String)response.getMetadata().get("oauthErrorText");
        this.checkStringContains("should say no response", oauthErrorText, "No response from server");
        this.checkStringContains("should show request", oauthErrorText, "GET /request?param=foo&opensocial_owner_id=owner");
        this.checkStringContains("should log empty response", oauthErrorText, "Received response 1:\n\n");
        this.checkLogContains("No response from server");
        this.checkLogContains("GET /request?param=foo&opensocial_owner_id=owner");
        this.checkLogContains("OAuth error [MISSING_SERVER_RESPONSE, No response from server] for application http://www.example.com/gadget.xml");
    }

    @Test
    public void testNextFetchThrowsGadgetException() throws Exception {
        this.serviceProvider.setThrow(new GadgetException(GadgetException.Code.FAILED_TO_RETRIEVE_CONTENT, "mildly wrong"));
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/access");
        Assert.assertEquals((Object)"MISSING_SERVER_RESPONSE", response.getMetadata().get("oauthError"));
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        String oauthErrorText = (String)response.getMetadata().get("oauthErrorText");
        this.checkStringContains("should say no response", oauthErrorText, "No response from server");
        this.checkStringContains("should show request", oauthErrorText, "GET /request?param=foo&opensocial_owner_id=owner");
        this.checkStringContains("should log empty response", oauthErrorText, "Received response 1:\n\n");
        this.checkLogContains("No response from server");
        this.checkLogContains("GET /request?param=foo&opensocial_owner_id=owner");
        this.checkLogContains("OAuth error [MISSING_SERVER_RESPONSE, No response from server] for application http://www.example.com/gadget.xml");
        this.checkLogContains("GadgetException");
        this.checkLogContains("mildly wrong");
    }

    @Test
    public void testNextFetchThrowsRuntimeException() throws Exception {
        this.serviceProvider.setThrow(new RuntimeException("very, very wrong"));
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        try {
            client.sendGet("http://www.example.com/access");
            Assert.fail((String)"Should have thrown");
        }
        catch (RuntimeException runtimeException) {
            // empty catch block
        }
        this.checkLogContains("OAuth fetch unexpected fatal erro");
        this.checkLogContains("GET /request?param=foo&opensocial_owner_id=owner");
        this.checkLogContains("OAuth error [very, very wrong] for application http://www.example.com/gadget.xml");
        this.checkLogContains("RuntimeException");
        this.checkLogContains("very, very wrong");
    }

    @Test
    public void testTrustedParams() throws Exception {
        this.serviceProvider.setCheckTrustedParams(true);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        client.setTrustedParam("oauth_magic", "foo");
        client.setTrustedParam("opensocial_magic", "bar");
        client.setTrustedParam("xoauth_magic", "quux");
        client.setTrustedParam("opensocial_owner_id", "overridden_opensocial_owner_id");
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)12L, (long)this.serviceProvider.getTrustedParamCount());
    }

    @Test
    public void testTrustedParamsMisc() throws Exception {
        this.serviceProvider.setCheckTrustedParams(true);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        client.setTrustedParam("oauth_magic", "foo");
        client.setTrustedParam("opensocial_magic", "bar");
        client.setTrustedParam("xoauth_magic", "quux_overridden");
        client.setTrustedParam("xoauth_magic", "quux");
        client.setTrustedParam("opensocial_owner_id", "overridden_opensocial_owner_id");
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)12L, (long)this.serviceProvider.getTrustedParamCount());
    }

    @Test
    public void testAlwaysAppendTrustedParams() throws Exception {
        this.serviceProvider.setCheckTrustedParams(true);
        MakeRequestClient client = this.makeStrictNonSocialClient("owner", "owner", GADGET_URL);
        client.setTrustedParam("oauth_magic", "foo");
        client.setTrustedParam("opensocial_magic", "bar");
        client.setTrustedParam("xoauth_magic", "quux");
        client.setTrustedParam("opensocial_owner_id", "overridden_opensocial_owner_id");
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)12L, (long)this.serviceProvider.getTrustedParamCount());
    }

    @Test
    public void testTrustedParamsInvalidParameter() throws Exception {
        this.serviceProvider.setCheckTrustedParams(true);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        client.setTrustedParam("oauth_magic", "foo");
        client.setTrustedParam("opensocial_magic", "bar");
        client.setTrustedParam("xoauth_magic", "quux");
        client.setTrustedParam("opensocial_owner_id", "overridden_opensocial_owner_id");
        client.setTrustedParam("invalid_trusted_parameter", "invalid");
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((long)403L, (long)response.getHttpStatusCode());
    }

    private boolean contains(List<OAuth.Parameter> params, String key, String value) {
        for (OAuth.Parameter p : params) {
            if (!p.getKey().equals(key) || !p.getValue().equals(value)) continue;
            return true;
        }
        return false;
    }

    private void checkContains(List<OAuth.Parameter> params, String key, String value) {
        for (OAuth.Parameter p : params) {
            if (!p.getKey().equals(key)) continue;
            Assert.assertEquals((Object)value, (Object)p.getValue());
            return;
        }
        Assert.fail((String)("List did not contain " + key + '=' + value + "; instead was " + params));
    }

    private String getLogText() {
        StringBuilder logText = new StringBuilder();
        for (LogRecord record : this.logRecords) {
            logText.append(record.getMessage());
            if (record.getThrown() == null) continue;
            StringWriter sw = new StringWriter();
            PrintWriter pw = new PrintWriter(sw);
            record.getThrown().printStackTrace(pw);
            pw.flush();
            logText.append(sw.toString());
        }
        return logText.toString();
    }

    private void checkLogContains(String text) {
        if (this.logger.getLevel() != null && this.logger.getLevel().equals(Level.OFF)) {
            return;
        }
        String logText = this.getLogText();
        if (!logText.contains(text)) {
            Assert.fail((String)("Should have logged '" + text + "', instead got " + logText));
        }
    }

    private void checkEmptyLog() {
        Assert.assertEquals((Object)"", (Object)this.getLogText());
    }

    private void checkStringContains(String message, String text, String expected) {
        if (!text.contains(expected)) {
            Assert.fail((String)(message + ", expected [" + expected + "], got + [" + text + ']'));
        }
    }
}

