package org.geneontology.owl.differ

import java.io.{File, FileWriter}
import java.util.Collections

import org.geneontology.owl.differ.render.{BasicDiffRenderer, HTMLDiffRenderer, MarkdownGroupedDiffRenderer}
import org.semanticweb.owlapi.apibinding.OWLManager
import org.semanticweb.owlapi.model.{IRI, OWLOntology, OWLOntologyIRIMapper}

import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.duration.Duration
import scala.concurrent.{Await, Future, blocking}

object Run extends App {

  private object DummyMapper extends OWLOntologyIRIMapper {

    def getDocumentIRI(ontIRI: IRI): IRI = {
      val manager = OWLManager.createOWLOntologyManager()
      val ont = manager.createOntology(ontIRI)
      ont.getOntologyID.getDefaultDocumentIRI.orNull
    }

  }

  def loadOntologies(left: IRI, right: IRI, loadImports: Boolean): Future[(OWLOntology, OWLOntology)] = {
    val leftManager = OWLManager.createOWLOntologyManager()
    val rightManager = OWLManager.createOWLOntologyManager()
    if (!loadImports) {
      leftManager.setIRIMappers(Collections.singleton(DummyMapper))
      rightManager.setIRIMappers(Collections.singleton(DummyMapper))
    }
    val leftOntologyFut = Future(blocking(leftManager.loadOntology(left)))
    val rightOntologyFut = Future(blocking(rightManager.loadOntology(right)))
    for {
      leftOnt <- leftOntologyFut
      rightOnt <- rightOntologyFut
    } yield (leftOnt, rightOnt)
  }

  val task = for {
    (left, right) <- loadOntologies(IRI.create(new File(args(0))), IRI.create(new File(args(1))), true)
  } yield {
    val writer = new FileWriter(new File(args(2)))
    val diff = Differ.diff(left, right)
    writer.write(ROBOTDiff.render(diff, right))
    writer.close()
    val plainWriter = new FileWriter(new File(args(3)))
    plainWriter.write(BasicDiffRenderer.renderPlain(diff))
    plainWriter.close()
    val grouped = Differ.groupedDiff(diff)
    val htmlWriter = new FileWriter(new File(args(4)))
    htmlWriter.write(HTMLDiffRenderer.render(grouped, right.getOWLOntologyManager))
    htmlWriter.close()
    val markdownWriter = new FileWriter(new File(args(5)))
    markdownWriter.write(MarkdownGroupedDiffRenderer.render(grouped, right.getOWLOntologyManager))
    markdownWriter.close()
  }
  println(Await.ready(task, Duration.Inf))

}