/*
 * Copyright 2020 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.taxonomy.gringlobal.model;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang3.StringUtils;

/**
 * The Interface IGrinSpecies.
 *
 * @author Matija Obreza
 */
public interface IGrinSpecies {

	/**
	 * Gets the species name.
	 *
	 * @return the speciesName
	 */
	String getSpeciesName();

	/**
	 * Gets the genus id.
	 *
	 * @return the genus id
	 */
	Long getGenusId();

	/**
	 * Gets the species id.
	 *
	 * @return the species id
	 */
	Long getSpeciesId();

	/**
	 * Gets the checks if is specific hybrid.
	 *
	 * @return the checks if is specific hybrid
	 */
	String getIsSpecificHybrid();

	/**
	 * Checks if is current.
	 *
	 * @return true, if is current
	 */
	boolean isCurrent();

	/**
	 * Gets the species authority.
	 *
	 * @return the species authority
	 */
	String getSpeciesAuthority();

	/**
	 * Gets the subtaxa.
	 *
	 * @return the subtaxa
	 */
	String getSubtaxa();

	/**
	 * Gets the subtaxa authority.
	 *
	 * @return the subtaxa authority
	 */
	String getSubtaxaAuthority();

	/**
	 * Gets the subspecies name.
	 *
	 * @return the subspecies name
	 */
	String getSubspeciesName();

	/**
	 * Gets the variety name.
	 *
	 * @return the variety name
	 */
	String getVarietyName();

	/**
	 * Gets the name.
	 *
	 * @return the name
	 */
	String getName();

	/**
	 * Gets the current taxonomy species id.
	 *
	 * @return the current taxonomy species id
	 */
	Long getCurrentTaxonomySpeciesId();

	/**
	 * Check if this record matches MCPD SUBTAXA.
	 * 
	 * @param subtaxa MCPD SUBTAXA value
	 * @return true if it's a match
	 */
	default boolean subtaxaMatches(String subtaxa) {
		subtaxa = StringUtils.defaultIfBlank(subtaxa, null);
		String subtaxPattern = StringUtils.defaultIfBlank(getSubtaxa(), null);
		if (subtaxa == null) {
			return subtaxPattern == null;
		}

		if (StringUtils.equalsIgnoreCase(subtaxPattern, subtaxa)) {
			return true;
		}

		if (StringUtils.isBlank(subtaxPattern)) {
			return false;
		}

		// Convert to a regular expression
		subtaxPattern = subtaxPattern.replaceAll("^(\\w+)\\.\\s+(\\w+)", "($1\\\\.\\\\s*)$2\\$");
		Pattern p = Pattern.compile(subtaxPattern, Pattern.CASE_INSENSITIVE);
		Matcher m = p.matcher(subtaxa);
		boolean found = m.find();
//		if (subtaxa.contains("endivia") && getSpeciesName().equalsIgnoreCase("endivia")) {
//			System.err.println("Regexp of " + getSubtaxa() + " = " + mySubtaxaRx + "\tmatches=" + subtaxa + " = " + found);
//		}
		return found;
	}
}
