/*
 * Copyright 2018 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.taxonomy.gringlobal.model;

/**
 * GRIN name qualifier:
 * <ul>
 * <li>blank if name is accepted without qualification</li>
 * <li>"=" if always synonym</li>
 * <li>"=~" if mostly a synonym</li>
 * <li>"~" if sometimes a synonym</li>
 * </ul>
 * .
 */
public enum Qualifier {

	/** Name accepted without qualification. */
	ACCEPTED(""),
	/** Always synonym */
	ALWAYS_SYNONYM("="),
	/** Mostly synonym. */
	MOSTLY_SYNONYM("=~"),
	/** Sometimes synonym. */
	SOMETIMES_SYNONYM("~");

	private String value;

	/**
	 * Instantiates a new qualifier.
	 *
	 * @param value the value
	 */
	Qualifier(String value) {
		this.value = value;
	}

	/**
	 * Gets the String value.
	 *
	 * @return the value
	 */
	public String getValue() {
		return value;
	}

	/**
	 * Get Qualifier from string code
	 *
	 * @param value the value
	 * @return the qualifier
	 */
	public static Qualifier fromString(String value) {
		if ("~".equals(value)) {
			return SOMETIMES_SYNONYM;
		} else if ("=~".equals(value)) {
			return MOSTLY_SYNONYM;
		} else if ("=".equals(value)) {
			return Qualifier.ALWAYS_SYNONYM;
		} else if ("".equals(value) || null == value) {
			return ACCEPTED;
		} else {
			throw new RuntimeException("No qualifier with code " + value);
		}
	}
}
