/*
 * Copyright 2020 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.taxonomy.gringlobal.model;

import java.time.LocalDateTime;

import org.apache.commons.lang3.StringUtils;
import org.genesys.taxonomy.gringlobal.component.CabReader;

import com.opencsv.bean.CsvBindByName;
import com.opencsv.bean.CsvDate;

import lombok.Data;
import lombok.Getter;
import lombok.ToString;

/**
 * "TAXONOMY_SPECIES_ID", "CURRENT_TAXONOMY_SPECIES_ID", "NOMEN_NUMBER",
 * "IS_SPECIFIC_HYBRID", "SPECIES_NAME", "SPECIES_AUTHORITY",
 * "IS_SUBSPECIFIC_HYBRID", "SUBSPECIES_NAME", "SUBSPECIES_AUTHORITY",
 * "IS_VARIETAL_HYBRID", "VARIETY_NAME", "VARIETY_AUTHORITY",
 * "IS_SUBVARIETAL_HYBRID", "SUBVARIETY_NAME", "SUBVARIETY_AUTHORITY",
 * "IS_FORMA_HYBRID", "FORMA_RANK_TYPE", "FORMA_NAME", "FORMA_AUTHORITY",
 * "TAXONOMY_GENUS_ID", "PRIORITY1_SITE_ID", "PRIORITY2_SITE_ID",
 * "CURATOR1_COOPERATOR_ID", "CURATOR2_COOPERATOR_ID", "RESTRICTION_CODE",
 * "LIFE_FORM_CODE", "COMMON_FERTILIZATION_CODE", "IS_NAME_PENDING",
 * "SYNONYM_CODE", "VERIFIER_COOPERATOR_ID", "NAME_VERIFIED_DATE", "NAME",
 * "NAME_AUTHORITY", "PROTOLOGUE", "NOTE", "SITE_NOTE", "ALTERNATE_NAME",
 * "CREATED_DATE", "CREATED_BY", "MODIFIED_DATE", "MODIFIED_BY", "OWNED_DATE",
 * "OWNED_BY"
 */
@Data
public class SpeciesRow implements IGrinSpecies {

	private static final String PREFIX_SUBSPECIES = "subsp. ";
	private static final String PREFIX_NOTHOSUBSPECIES = "nothosubsp. ";
	private static final String PREFIX_VARIETY = "var. ";
	private static final String PREFIX_SUBVARIETY = "subvar. ";
	private static final String PREFIX_FORMA = "f.";

	@CsvBindByName(column = "taxonomy_species_id")
	private Long taxonomySpeciesId;

	@CsvBindByName(column = "taxonomy_genus_id")
	private Long taxonomyGenusId;

	@CsvBindByName(column = "name")
	private String name;

	@CsvBindByName(column = "name_authority")
	private String nameAuthority;

	/** MCPD SUBTAXA is not part of GRIN-Global taxonomy data */
	@Getter(lazy=true) private final String subtaxa = toSubtaxa();

	/** MCPD SUBTAUTHOR is not part of GRIN-Global taxonomy data */
	@Getter(lazy=true) private final String subtaxaAuthority = toSubtaxaAuthority();

	@CsvBindByName(column = "nomen_number")
	private Long nomenNumber;

	@CsvBindByName(column = "synonym_code")
	private String synonymCode;

	@CsvBindByName(column = "current_taxonomy_species_id")
	private Long currentTaxonomySpeciesId;

	@CsvBindByName(column = "is_specific_hybrid")
	private String isSpecificHybrid;

	@CsvBindByName(column = "species_name")
	private String speciesName;

	@CsvBindByName(column = "species_authority")
	private String speciesAuthority;

	@CsvBindByName(column = "is_subspecific_hybrid")
	private String isSubspecificHybrid;

	@CsvBindByName(column = "subspecies_name")
	private String subspeciesName;

	@CsvBindByName(column = "subspecies_authority")
	private String subspeciesAuthority;

	@CsvBindByName(column = "is_varietal_hybrid")
	private String isVarietalHybrid;

	@CsvBindByName(column = "variety_name")
	private String varietyName;

	@CsvBindByName(column = "variety_authority")
	private String varietyAuthority;

	@CsvBindByName(column = "is_subvarietal_hybrid")
	private String isSubvarietalHybrid;

	@CsvBindByName(column = "subvariety_name")
	private String subvarietyName;

	@CsvBindByName(column = "subvariety_authority")
	private String subvarietyAuthority;

	@CsvBindByName(column = "is_forma_hybrid")
	private String isFormaHybrid;

	@CsvBindByName(column = "forma_rank_type")
	private String formaRankType;

	@CsvBindByName(column = "forma_name")
	private String formaName;

	@CsvBindByName(column = "forma_authority")
	private String formaAuthority;

	@CsvBindByName(column = "priority1_site_id")
	@ToString.Exclude
	private Long priority1SiteId;

	@CsvBindByName(column = "priority2_site_id")
	@ToString.Exclude
	private Long priority2SiteId;

	@CsvBindByName(column = "curator1_cooperator_id")
	@ToString.Exclude
	private Long curator1CooperatorId;

	@CsvBindByName(column = "curator2_cooperator_id")
	@ToString.Exclude
	private Long curator2CooperatorId;

	@CsvBindByName(column = "restriction_code")
	private String restrictionCode;

	@CsvBindByName(column = "life_form_code")
	private String lifeFormCode;

	@CsvBindByName(column = "common_fertilization_code")
	@ToString.Exclude
	private String commonFertilizationCode;

	@CsvBindByName(column = "is_name_pending")
	private String isNamePending;

	@CsvBindByName(column = "verifier_cooperator_id")
	@ToString.Exclude
	private Long verifierCooperatorId;

	@CsvBindByName(column = "name_verified_date")
	@CsvDate(CabReader.CSV_DATE_FORMAT)
	@ToString.Exclude
	private LocalDateTime nameVerifiedDate;

	@CsvBindByName(column = "protologue")
	@ToString.Exclude
	private String protologue;

	@CsvBindByName(column = "protologue_virtual_path")
	@ToString.Exclude
	private String protologueVirtualPath;

	@CsvBindByName(column = "note")
	@ToString.Exclude
	private String note;

	@CsvBindByName(column = "site_note")
	@ToString.Exclude
	private String siteNote;

	@CsvBindByName(column = "alternate_name")
	private String alternateName;

	@CsvBindByName(column = "hybrid_parentage")
	private String hybridParentage;

	@CsvBindByName(column = "is_web_visible")
	private String isWebVisible;

	@CsvBindByName(column = "created_date")
	@CsvDate(CabReader.CSV_DATE_FORMAT)
	@ToString.Exclude
	private LocalDateTime createdDate;

	@CsvBindByName(column = "created_by")
	@ToString.Exclude
	private Long createdBy;

	@CsvBindByName(column = "modified_date")
	@CsvDate(CabReader.CSV_DATE_FORMAT)
	@ToString.Exclude
	private LocalDateTime modifiedDate;

	@CsvBindByName(column = "modified_by")
	@ToString.Exclude
	private Long modifiedBy;

	@CsvBindByName(column = "owned_date")
	@CsvDate(CabReader.CSV_DATE_FORMAT)
	@ToString.Exclude
	private LocalDateTime ownedDate;

	@CsvBindByName(column = "owned_by")
	@ToString.Exclude
	private Long ownedBy;

	@Override
	public Long getSpeciesId() {
		return this.taxonomySpeciesId;
	}

	@Override
	public Long getGenusId() {
		return this.taxonomyGenusId;
	}

	/**
	 * Is this record still current?.
	 *
	 * @return <code>true</code> when currentSpeciesId equals speciesId
	 */
	public boolean isCurrent() {
		return taxonomySpeciesId.equals(currentTaxonomySpeciesId);
	}

	/**
	 * Generates and returns the subspecies, variety information as the MCPD
	 * SUBTAXA. This should be like taking the {@link #name} field and removing
	 * Genus and species.
	 *
	 * @return the subtaxa
	 */
	private String toSubtaxa() {
		if (formaName != null) {
			String prefix = StringUtils.defaultIfBlank(formaRankType, PREFIX_FORMA) + " ";
			return prefix + formaName;
		}
		if (subvarietyName != null) {
			return PREFIX_SUBVARIETY + subvarietyName;
		}
		if (varietyName != null) {
			return PREFIX_VARIETY + varietyName;
		}
		if (subspeciesName != null) {
			if ("Y".equals(isSubspecificHybrid)) {
				return PREFIX_NOTHOSUBSPECIES + subspeciesName;
			}
			return PREFIX_SUBSPECIES + subspeciesName;
		}
		return null;
	}

	/**
	 * Get MCPD SUBTAUTHOR: subtaxon authority at the most detailed taxonomic level.
	 * 
	 * @return subtaxon authority at the most detailed taxonomic level.
	 */
	private String toSubtaxaAuthority() {
		if (formaName != null) {
			return formaAuthority;
		}
		if (subvarietyName != null) {
			return subvarietyAuthority;
		}
		if (varietyName != null) {
			return varietyAuthority;
		}
		if (subspeciesName != null) {
			return subspeciesAuthority;
		}
		return null;
	}

}
