/**
 * Copyright 2013 Bioversity International and
 *                the Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * The code tree likely contains a copy of the License,
 * ('LICENSE'), but you may also obtain a copy at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys2.rdf.model.dwc;

import java.io.IOException;
import java.util.Map;

import org.genesys2.rdf.model.ModelException;
import org.genesys2.rdf.model.rdf.ModelDocument;
import org.genesys2.rdf.model.rdf.Parser;
import org.genesys2.rdf.model.rdf.RDFParserException;
import org.genesys2.rdf.model.rdf.Vocabularies;

import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.rdf.model.ModelFactory;
import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.rdf.model.Resource;

// TODO: Auto-generated Javadoc
/**
 * Wrapper of the Darwin Core (RDF) ontology.
 *
 * @author Richard Bruskiewich
 */
public class DarwinCore extends ModelDocument {

	/**
	 * <p>
	 * The RDF model that holds the vocabulary terms
	 * </p>
	 * .
	 */
	private static Model m_model = ModelFactory.createDefaultModel();

	/**
	 * <p>
	 * The namespace of the vocabulary as a string
	 * </p>
	 * .
	 */
	public static final String NS = Vocabularies.DARWIN_CORE_TERMS.getURI();

	/**
	 * <p>
	 * The namespace of the vocabulary as a string
	 * </p>
	 * .
	 *
	 * @return the uri
	 * @see #NS
	 */
	public static String getURI() {
		return NS;
	}

	/**
	 * <p>
	 * The namespace of the vocabulary as a resource
	 * </p>
	 * .
	 */
	public static final Resource NAMESPACE = m_model.createResource(NS);

	/**
	 * <p>
	 * A Darwin Core (class type).
	 * </p>
	 */
	public static final Resource Taxon = m_model.createResource(NS + "Taxon");

	/**
	 * <p>
	 * Indicates the creator of a document.
	 * </p>
	 */
	public static final Property scientificName = m_model.createProperty(NS + "scientificName");

	/**
	 * <p>
	 * Indicates the creator of a document.
	 * </p>
	 */
	public static final Property vernacularName = m_model.createProperty(NS + "vernacularName");

	/**
	 * <p>
	 * Indicates the creator of a document.
	 * </p>
	 */
	public static final Property genus = m_model.createProperty(NS + "genus");

	/**
	 * Constructor to initialize Darwin Core ontology.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws ModelException the model exception
	 */
	public DarwinCore() throws IOException, ModelException {
		super(Vocabularies.DARWIN_CORE_TERMS.getURI());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.bioversityinternational.model.rdf.ModelDocument#propertyDetails(java
	 * .lang.String,java.lang.String)
	 */
	@Override
	public Map<String, String> propertyDetails(String termURI, String language) throws RDFParserException {

		Map<String, String> parts = this.parser().parseURI(termURI);

		ModelDocument md = lookup(parts.get(Parser.NAMESPACE));

		Map<String, String> details = super.propertyDetails(parts.get(Parser.URI), language);

		_load(details, md.parser().getPropertyValues(parts.get(Parser.URI), DCA.abcdEquivalence.getURI()),
				DCA.abcdEquivalence.term());
		_load(details, md.parser().getPropertyValues(parts.get(Parser.URI), DCA.organizedInClass.getURI()),
				DCA.organizedInClass.term());
		_load(details, md.parser().getPropertyValues(parts.get(Parser.URI), DCA.status.getURI()), DCA.status.term());

		return details;
	}

}
