/**
 * Copyright 2013 Bioversity International and
 *                the Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * The code tree likely contains a copy of the License,
 * ('LICENSE'), but you may also obtain a copy at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.genesys2.rdf.model.germplasm;

import org.genesys2.rdf.model.skos.SKOS;

import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.rdf.model.ModelFactory;
import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.rdf.model.Resource;
import com.hp.hpl.jena.vocabulary.RDFS;

// TODO: Auto-generated Javadoc
/**
 * The Class CCO.
 */
public class CCO {

	/**
	 * <p>
	 * The RDF model that holds the vocabulary terms
	 * </p>
	 * .
	 */
	private static Model m_model = ModelFactory.createDefaultModel();

	/**
	 * <p>
	 * The namespace of the vocabulary as a string
	 * </p>
	 * .
	 */
	public static final String NS = "http://data.bioversityinternational.org/cco#";

	/**
	 * <p>
	 * The namespace of the vocabulary as a string
	 * </p>
	 * .
	 *
	 * @return the uri
	 * @see #NS
	 */
	public static String getURI() {
		return NS;
	}

	/**
	 * <p>
	 * The namespace of the vocabulary as a resource
	 * </p>
	 * .
	 */
	public static final Resource NAMESPACE = m_model.createResource(NS);

	/**
	 * <p>
	 * A Species (class type) resource.
	 * </p>
	 */
	public static final Resource Species = m_model.createResource(NS + "Species");

	/**
	 * <p>
	 * Indicates that a species is included in a given crop grouping (this is a
	 * GENESYS thing...).
	 * </p>
	 */
	public static final Property speciesIncluded = m_model.createProperty(NS + "speciesIncluded");

	/**
	 * <p>
	 * The crop to which which a given trait descriptor is assigned.
	 * </p>
	 */
	public static final Property taxonomicScope = m_model.createProperty(NS + "taxonomicScope");

	/**
	 * <p>
	 * The trait property of a crop.
	 * </p>
	 */
	public static final Property trait = m_model.createProperty(NS + "trait");

	/**
	 * <p>
	 * The category of a trait property of a crop.
	 * </p>
	 */
	public static final Property category = m_model.createProperty(NS + "category");

	/**
	 * <p>
	 * The method of a trait property of a crop.
	 * </p>
	 */
	public static final Property method = m_model.createProperty(NS + "method");

	/**
	 * <p>
	 * The scale of the method of a trait property of a crop.
	 * </p>
	 */
	public static final Property scale = m_model.createProperty(NS + "scale");

	/**
	 * <p>
	 * The type of scale.
	 * </p>
	 */
	public static final Property scaleType = m_model.createProperty(NS + "scaleType");

	/**
	 * <p>
	 * The units of a scale.
	 * </p>
	 */
	public static final Property scaleUnit = m_model.createProperty(NS + "scaleUnit");

	/**
	 * <p>
	 * A range of values for a given property.
	 * </p>
	 */
	public static final Resource ValueRange = m_model.createResource(NS + "ValueRange");

	/**
	 * <p>
	 * A range of integer values for a given property.
	 * </p>
	 */
	public static final Resource IntegerValueRange = m_model.createResource(NS + "IntegerValueRange");

	/**
	 * <p>
	 * The minimum integer value of an Integer Value Range.
	 * </p>
	 */
	public static final Property integerMinValue = m_model.createProperty(NS + "integerMinValue");

	/**
	 * <p>
	 * The maximum integer value of an Integer Value Range.
	 * </p>
	 */
	public static final Property integerMaxValue = m_model.createProperty(NS + "integerMaxValue");

	/**
	 * The Enum Vocabulary.
	 */
	public enum Vocabulary {

		/** The trait. */
		TRAIT("Trait", m_model.createResource(NS + "PlantTraitVocabulary")),

		/** The category. */
		CATEGORY("TraitCategory",

		m_model.createResource(NS + "PlantTraitCategoryVocabulary")),

		/** The method. */
		METHOD("TraitMethod",

		m_model.createResource(NS + "PlantTraitMethodVocabulary")),

		/** The scale. */
		SCALE("TraitScale",

		m_model.createResource(NS + "PlantTraitScaleVocabulary"));

		/** The tag. */
		private final String tag;

		/** The resource. */
		private final Resource resource;

		/**
		 * Instantiates a new vocabulary.
		 *
		 * @param tag the tag
		 * @param resource the resource
		 */
		Vocabulary(String tag, Resource resource) {
			this.tag = tag;
			this.resource = resource;
		}

		/**
		 * Gets the tag.
		 *
		 * @return the tag
		 */
		public String getTag() {
			return this.tag;
		}

		/**
		 * Gets the resource.
		 *
		 * @return the resource
		 */
		public Resource getResource() {
			return this.resource;
		}

		/**
		 * Method to create an RDF Resource documenting a CCO vocabulary.
		 *
		 * @param model the model
		 * @param subject the subject
		 * @param vType the v type
		 * @param name the name
		 * @return the resource
		 */
		public static Resource set(Model model, Resource subject, Vocabulary vType, String name) {

			Resource vocabulary = model.createResource(NS + name + vType.getTag() + "Vocabulary");
			vocabulary.addProperty(RDFS.subClassOf, vType.getResource());
			subject.addProperty(SKOS.inScheme, vocabulary);

			return vocabulary;
		}
	}

	/**
	 * <p>
	 * The PlantTraitCategoryVocabulary (as a Resource).
	 * </p>
	 */
	public static final Resource PlantTraitCategoryVocabulary = m_model.createResource(NS
			+ "PlantTraitCategoryVocabulary");

	/**
	 * <p>
	 * A TraitCategory (as a Resource).
	 * </p>
	 */
	public static final Resource TraitCategory = m_model.createResource(NS + "TraitCategory");

	/**
	 * <p>
	 * The PlantTraitCategoryVocabulary (as a Resource).
	 * </p>
	 */
	public static final Resource PlantTraitMethodVocabulary = m_model.createResource(NS + "PlantTraitMethodVocabulary");

	/**
	 * <p>
	 * The PlantTraitCategoryVocabulary (as a Resource).
	 * </p>
	 */
	public static final Resource PlantTraitScaleVocabulary = m_model.createResource(NS + "PlantTraitScaleVocabulary");

}
