/**
 * Copyright 2013 Bioversity International and
 *                the Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * The code tree likely contains a copy of the License,
 * ('LICENSE'), but you may also obtain a copy at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/
package org.genesys2.rdf.model.germplasm;

import java.io.IOException;
import java.util.Map;

import org.genesys2.rdf.model.ModelException;
import org.genesys2.rdf.model.rdf.ModelDocument;
import org.genesys2.rdf.model.rdf.Parser;
import org.genesys2.rdf.model.rdf.RDFParserException;
import org.genesys2.rdf.model.skos.SKOS;

/**
 * The Class DescriptorOntology.
 */
abstract public class DescriptorOntology extends ModelDocument {

	/** The Constant NAME. */
	public static final String NAME = "Name";

	/** The Constant ALIAS. */
	public static final String ALIAS = "Alias";

	/** The Constant DEFINITION. */
	public static final String DEFINITION = "Definition";

	/**
	 * Instantiates a new descriptor ontology.
	 *
	 * @param modelURI the model uri
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws ModelException the model exception
	 */
	protected DescriptorOntology(String modelURI) throws IOException, ModelException {
		super(modelURI);
	}

	/**
	 * Instantiates a new descriptor ontology.
	 *
	 * @param modelURI the model uri
	 * @param language the language
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws ModelException the model exception
	 */
	protected DescriptorOntology(String modelURI, String language) throws IOException, ModelException {
		this(modelURI);
		this.setLanguage(language);
	}

	/**
	 * This is a variant of the ModelDocument details() method, which uses SKOS
	 * prefLabel instead of the rdfs:label property and the SKOS definition
	 * instead of the rdfs:comment and rdfs:description properties.
	 *
	 * @param details is a pre-existing target Map for loading
	 * @param descriptorURI the descriptor uri
	 * @param language ISO639-2 two or three character language code. Defaults
	 *        to English if not given.
	 * @return the map
	 * @throws RDFParserException the RDF parser exception
	 */
	public Map<String, String> definition(Map<String, String> details, String descriptorURI, String language)
			throws RDFParserException {

		Map<String, String> parts = this.parser().parseURI(descriptorURI);

		ModelDocument md = lookup(parts.get(Parser.NAMESPACE));

		_load(details, md.parser().getPropertyValues(parts.get(Parser.URI), SKOS.prefLabel.getURI(), language, false),
				NAME);
		_load(details, md.parser().getPropertyValues(parts.get(Parser.URI), SKOS.altLabel.getURI(), language, true),
				ALIAS);
		_load(details, md.parser().getPropertyValues(parts.get(Parser.URI), SKOS.definition.getURI(), language, false),
				DEFINITION);

		return details;
	}

	/**
	 * Method to return all the categories of descriptors in a given set.
	 *
	 * @return categories where the key is a given category URI and the value is
	 *         a pipe '|' concatenated list of descriptors.
	 * @throws RDFParserException the RDF parser exception
	 */
	public Map<String, String> categories() throws RDFParserException {
		return collections();
	}

	/**
	 * Loads descriptor category-specific details.
	 *
	 * @param categoryURI the category uri
	 * @param language the language
	 * @return map loaded with descriptor meta-data
	 * @throws RDFParserException the RDF parser exception
	 */
	public Map<String, String> categoryDetails(String categoryURI, String language) throws RDFParserException {

		Map<String, String> parts = this.parser().parseURI(categoryURI);

		return definition(classDetails((String) parts.get(Parser.URI), language), categoryURI, language);
	}

	/**
	 * Loads Map of descriptors from a given category. Modelled as members of a
	 * categoryURI collection.
	 *
	 * @param categoryURI the category uri
	 * @return map loaded with descriptor meta-data
	 * @throws RDFParserException the RDF parser exception
	 */
	public Map<String, String> descriptors(String categoryURI) throws RDFParserException {

		return members(categoryURI);
	}

	/**
	 * Loads descriptor descriptor specific details.
	 *
	 * @param descriptorURI - URI of the descriptor
	 * @param language - ISO
	 * @return map loaded with descriptor meta-data
	 * @throws RDFParserException the RDF parser exception
	 */
	public Map<String, String> descriptorDetails(String descriptorURI, String language) throws RDFParserException {

		Map<String, String> parts = this.parser().parseURI(descriptorURI);
		return definition(propertyDetails((String) parts.get(Parser.URI), language), descriptorURI, language);
	}
}
