/**
 * Copyright 2013 Bioversity International and
 *                the Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * The code tree likely contains a copy of the License,
 * ('LICENSE'), but you may also obtain a copy at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/
package org.genesys2.rdf.model.rdf;

import org.genesys2.rdf.model.dc.DCMI;
import org.genesys2.rdf.model.dc.DublinCore;
import org.genesys2.rdf.model.dwc.DarwinCore;
import org.genesys2.rdf.model.dwc.DarwinCoreAttributes;
import org.genesys2.rdf.model.foaf.FOAF;
import org.genesys2.rdf.model.germplasm.DarwinCoreGermplasmType;
import org.genesys2.rdf.model.germplasm.MCPDOntology;
import org.genesys2.rdf.model.w3c.WGS94GeoPositioning;

// TODO: Auto-generated Javadoc
/**
 * The Enum Vocabularies.
 *
 * @author Richard Bruskiewich
 * 
 *         Make sure that your base URI's are accurate: if a hash '#' is
 *         required, it should be there!
 */
public enum Vocabularies {

	// XML Schema is not RDF but it is recorded
	// here in case one comes across it in an RDF schema
	// Requires special treatment in the code...
	/** The xml schema. */
	XML_SCHEMA("http://www.w3.org/2001/XMLSchema#", null),

	/** The rdf schema. */
	RDF_SCHEMA("http://www.w3.org/2000/01/rdf-schema#", RDFSchema.class),

	/** The dublin core metadata initiative. */
	DUBLIN_CORE_METADATA_INITIATIVE("http://purl.org/dc/dcmitype/", DCMI.class),

	/** The dublin core terms. */
	DUBLIN_CORE_TERMS("http://purl.org/dc/terms/", DublinCore.class),

	/** The foaf. */
	FOAF("http://xmlns.com/foaf/0.1/", FOAF.class),

	/** The WG s84_ ge o_ positioning. */
	WGS84_GEO_POSITIONING("http://www.w3.org/2003/01/geo/wgs84_pos#", WGS94GeoPositioning.class),

	/** The darwin core terms. */
	DARWIN_CORE_TERMS("http://rs.tdwg.org/dwc/terms/", DarwinCore.class),

	/** The darwin core attributes. */
	DARWIN_CORE_ATTRIBUTES("http://rs.tdwg.org/dwc/terms/attributes/", DarwinCoreAttributes.class),

	/** The darwin core germplasm types. */
	DARWIN_CORE_GERMPLASM_TYPES("http://purl.org/germplasm/germplasmType#", DarwinCoreGermplasmType.class),

	/** The darwin core germplasm terms. */
	DARWIN_CORE_GERMPLASM_TERMS("http://purl.org/germplasm/germplasmTerm#", MCPDOntology.class);

	/** The uri. */
	private final String uri;

	/** The document class. */
	private final Class<ModelDocument> documentClass;

	/**
	 * Instantiates a new vocabularies.
	 *
	 * @param uri the uri
	 * @param documentClass the document class
	 */
	@SuppressWarnings("unchecked")
	Vocabularies(String uri, Class<? extends ModelDocument> documentClass) {
		this.uri = uri;
		this.documentClass = (Class<ModelDocument>) documentClass;
	}

	/**
	 * Gets the uri.
	 *
	 * @return the uri
	 */
	public String getURI() {
		return this.uri;
	}

	/**
	 * Gets the doc class.
	 *
	 * @return the doc class
	 */
	public Class<ModelDocument> getDocClass() {
		return this.documentClass;
	}

	/**
	 * Method to look up the Class model wrapping a given RDF target document.
	 *
	 * @param uri the uri
	 * @return the class
	 */
	public static Class<ModelDocument> lookup(String uri) {
		// Brute force: linear search, not efficient for a large catalog of
		// Ontology
		// TODO: should I generalize the Vocabulary catalog to use a Map?
		Class<ModelDocument> c = null;
		for (Vocabularies v : Vocabularies.values()) {
			if (v.getURI().equals(uri)) {
				c = v.getDocClass();
				break;
			}
		}
		return c; // may be null if not found?
	}

}
