/*
 * Copyright (c) 2021, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * The Universal Permissive License (UPL), Version 1.0
 *
 * Subject to the condition set forth below, permission is hereby granted to any
 * person obtaining a copy of this software, associated documentation and/or
 * data (collectively the "Software"), free of charge and under any and all
 * copyright rights in the Software, and any and all patent rights owned or
 * freely licensable by each licensor hereunder covering either (i) the
 * unmodified Software as contributed to or provided by such licensor, or (ii)
 * the Larger Works (as defined below), to deal in both
 *
 * (a) the Software, and
 *
 * (b) any piece of software and/or hardware listed in the lrgrwrks.txt file if
 * one is included with the Software each a "Larger Work" to which the Software
 * is contributed by such licensors),
 *
 * without restriction, including without limitation the rights to copy, create
 * derivative works of, display, perform, and distribute the Software and make,
 * use, sell, offer for sale, import, export, have made, and have sold the
 * Software and the Larger Work(s), and to sublicense the foregoing rights on
 * either these or other terms.
 *
 * This license is subject to the following condition:
 *
 * The above copyright notice and either this complete permission notice or at a
 * minimum a reference to the UPL must be included in all copies or substantial
 * portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package org.graalvm.buildtools.gradle.tasks.actions;

import groovy.json.JsonGenerator;
import groovy.json.JsonOutput;
import groovy.json.JsonSlurper;
import org.gradle.api.Action;
import org.gradle.api.GradleException;
import org.gradle.api.Task;
import org.gradle.api.file.Directory;
import org.gradle.api.provider.ListProperty;
import org.gradle.api.provider.Provider;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * This task is responsible for processing the JSON files generated by
 * the GraalVM agent, in particular to filter out entries which are
 * inherited from the Gradle environment itself.
 */
public class ProcessGeneratedGraalResourceFilesAction implements Action<Task> {
    private final Provider<Directory> inputDirectory;
    private final ListProperty<String> filterableEntries;

    public ProcessGeneratedGraalResourceFilesAction(Provider<Directory> inputDirectory, ListProperty<String> filterableEntries) {
        this.inputDirectory = inputDirectory;
        this.filterableEntries = filterableEntries;
    }

    @Override
    public void execute(Task task) {
        try {
            for (File resourceFile : inputDirectory.get().getAsFileTree()) {
                processFile(resourceFile);
            }
        } catch (IOException e) {
            throw new GradleException("An IO error occured when processing the agent generated files", e);
        }
    }

    protected void processFile(File file) throws IOException {
        if (file.getName().endsWith(".json")) {
            processJsonFile(file);
        }
    }

    protected void processJsonFile(File jsonFile) throws IOException {
        JsonSlurper json = new JsonSlurper();
        Object result = json.parse(jsonFile);
        Object filtered = filter(result);
        JsonGenerator generator = new JsonGenerator.Options()
                .build();
        String processed = JsonOutput.prettyPrint(generator.toJson(filtered));
        try (Writer writer = new OutputStreamWriter(new FileOutputStream(jsonFile), StandardCharsets.UTF_8)) {
            writer.write(processed);
        }
    }

    /**
     * Filters the parsed JSON file to remove entries which are configured
     * by the filterable entries parameter. This is a very rough algorithm
     * which would deserve specific implementation for each JSON format.
     * Instead it takes a "brute force" approach which may result in some
     * weird errors.
     */
    @SuppressWarnings("unchecked")
    private Object filter(Object in) {
        Class<?> clazz = in.getClass();
        if (shouldFilterString(in)) {
            return null;
        }
        if (List.class.isAssignableFrom(clazz)) {
            return filterList((List<Object>) in);
        }
        if (Map.class.isAssignableFrom(clazz)) {
            return filterMap((Map<Object, Object>) in);
        }
        return in;
    }

    private Map<Object, Object> filterMap(Map<Object, Object> map) {
        if (shouldFilterString(map.get("name"))) {
            return null;
        }
        Map<Object, Object> out = new HashMap<>(map.size());
        for (Map.Entry<Object, Object> entry : map.entrySet()) {
            Object key = entry.getKey();
            Object value = entry.getValue();
            if (shouldFilterString(value)) {
                continue;
            }
            out.put(key, filter(value));
        }
        return out;
    }

    private boolean shouldFilterString(Object value) {
        if (value instanceof CharSequence) {
            String string = value.toString();
            return filterableEntries.get().stream().anyMatch(string::startsWith);
        }
        return false;
    }

    private List<Object> filterList(List<Object> in) {
        List<Object> out = new ArrayList<>(in.size());
        for (Object element : in) {
            Object filtered = filter(element);
            if (filtered == null || (filtered instanceof Collection && ((Collection<?>) filtered).isEmpty())) {
                continue;
            }
            out.add(filtered);
        }
        return out;
    }
}
