/* Copyright (c) 2023 Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package com.oracle.truffle.espresso.vm.structs;

import java.nio.ByteBuffer;

import com.oracle.truffle.api.interop.TruffleObject;

import com.oracle.truffle.espresso.runtime.staticobject.StaticObject;
import com.oracle.truffle.espresso.jni.JniEnv;
import com.oracle.truffle.espresso.ffi.RawPointer;

@SuppressWarnings(value = "unused") public final class JvmtiLocalVariableEntry extends StructStorage<JvmtiLocalVariableEntry.JvmtiLocalVariableEntryWrapper> {

    final int start_location;

    final int length;

    final int name;

    final int signature;

    final int generic_signature;

    final int slot;

    JvmtiLocalVariableEntry(MemberOffsetGetter offsetGetter) {
        super(offsetGetter.getInfo("_jvmtiLocalVariableEntry"));
        start_location = (int) offsetGetter.getOffset("_jvmtiLocalVariableEntry", "start_location");
        length = (int) offsetGetter.getOffset("_jvmtiLocalVariableEntry", "length");
        name = (int) offsetGetter.getOffset("_jvmtiLocalVariableEntry", "name");
        signature = (int) offsetGetter.getOffset("_jvmtiLocalVariableEntry", "signature");
        generic_signature = (int) offsetGetter.getOffset("_jvmtiLocalVariableEntry", "generic_signature");
        slot = (int) offsetGetter.getOffset("_jvmtiLocalVariableEntry", "slot");
    }

    public final class JvmtiLocalVariableEntryWrapper extends StructWrapper {

        JvmtiLocalVariableEntryWrapper(JniEnv jni, TruffleObject pointer) {
            super(jni, pointer, structSize);
        }

        public long startLocation() {
            return getLong(start_location);
        }

        public void startLocation(long valueToPut) {
            putLong(start_location, valueToPut);
        }

        public int length() {
            return getInt(length);
        }

        public void length(int valueToPut) {
            putInt(length, valueToPut);
        }

        public TruffleObject name() {
            return getPointer(name);
        }

        public void name(TruffleObject valueToPut) {
            putPointer(name, valueToPut);
        }

        public TruffleObject signature() {
            return getPointer(signature);
        }

        public void signature(TruffleObject valueToPut) {
            putPointer(signature, valueToPut);
        }

        public TruffleObject genericSignature() {
            return getPointer(generic_signature);
        }

        public void genericSignature(TruffleObject valueToPut) {
            putPointer(generic_signature, valueToPut);
        }

        public int slot() {
            return getInt(slot);
        }

        public void slot(int valueToPut) {
            putInt(slot, valueToPut);
        }

    }

    @Override
    public JvmtiLocalVariableEntryWrapper wrap(JniEnv jni, TruffleObject pointer) {
        return new JvmtiLocalVariableEntryWrapper(jni, pointer);
    }

}

