/* Copyright (c) 2023 Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package com.oracle.truffle.espresso.vm.structs;

import java.nio.ByteBuffer;

import com.oracle.truffle.api.interop.TruffleObject;

import com.oracle.truffle.espresso.runtime.staticobject.StaticObject;
import com.oracle.truffle.espresso.jni.JniEnv;
import com.oracle.truffle.espresso.ffi.RawPointer;

@SuppressWarnings(value = "unused") public final class JvmtiMonitorUsage extends StructStorage<JvmtiMonitorUsage.JvmtiMonitorUsageWrapper> {

    final int owner;

    final int entry_count;

    final int waiter_count;

    final int waiters;

    final int notify_waiter_count;

    final int notify_waiters;

    JvmtiMonitorUsage(MemberOffsetGetter offsetGetter) {
        super(offsetGetter.getInfo("_jvmtiMonitorUsage"));
        owner = (int) offsetGetter.getOffset("_jvmtiMonitorUsage", "owner");
        entry_count = (int) offsetGetter.getOffset("_jvmtiMonitorUsage", "entry_count");
        waiter_count = (int) offsetGetter.getOffset("_jvmtiMonitorUsage", "waiter_count");
        waiters = (int) offsetGetter.getOffset("_jvmtiMonitorUsage", "waiters");
        notify_waiter_count = (int) offsetGetter.getOffset("_jvmtiMonitorUsage", "notify_waiter_count");
        notify_waiters = (int) offsetGetter.getOffset("_jvmtiMonitorUsage", "notify_waiters");
    }

    public final class JvmtiMonitorUsageWrapper extends StructWrapper {

        JvmtiMonitorUsageWrapper(JniEnv jni, TruffleObject pointer) {
            super(jni, pointer, structSize);
        }

        public StaticObject owner() {
            return getObject(owner);
        }

        public void owner(StaticObject valueToPut) {
            putObject(owner, valueToPut);
        }

        public int entryCount() {
            return getInt(entry_count);
        }

        public void entryCount(int valueToPut) {
            putInt(entry_count, valueToPut);
        }

        public int waiterCount() {
            return getInt(waiter_count);
        }

        public void waiterCount(int valueToPut) {
            putInt(waiter_count, valueToPut);
        }

        public TruffleObject waiters() {
            return getPointer(waiters);
        }

        public void waiters(TruffleObject valueToPut) {
            putPointer(waiters, valueToPut);
        }

        public int notifyWaiterCount() {
            return getInt(notify_waiter_count);
        }

        public void notifyWaiterCount(int valueToPut) {
            putInt(notify_waiter_count, valueToPut);
        }

        public TruffleObject notifyWaiters() {
            return getPointer(notify_waiters);
        }

        public void notifyWaiters(TruffleObject valueToPut) {
            putPointer(notify_waiters, valueToPut);
        }

    }

    @Override
    public JvmtiMonitorUsageWrapper wrap(JniEnv jni, TruffleObject pointer) {
        return new JvmtiMonitorUsageWrapper(jni, pointer);
    }

}

