/* Copyright (c) 2023 Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package com.oracle.truffle.espresso.vm.structs;

import java.nio.ByteBuffer;

import com.oracle.truffle.api.interop.TruffleObject;

import com.oracle.truffle.espresso.runtime.staticobject.StaticObject;
import com.oracle.truffle.espresso.jni.JniEnv;
import com.oracle.truffle.espresso.ffi.RawPointer;

@SuppressWarnings(value = "unused") public final class JvmtiThreadGroupInfo extends StructStorage<JvmtiThreadGroupInfo.JvmtiThreadGroupInfoWrapper> {

    final int parent;

    final int name;

    final int max_priority;

    final int is_daemon;

    JvmtiThreadGroupInfo(MemberOffsetGetter offsetGetter) {
        super(offsetGetter.getInfo("_jvmtiThreadGroupInfo"));
        parent = (int) offsetGetter.getOffset("_jvmtiThreadGroupInfo", "parent");
        name = (int) offsetGetter.getOffset("_jvmtiThreadGroupInfo", "name");
        max_priority = (int) offsetGetter.getOffset("_jvmtiThreadGroupInfo", "max_priority");
        is_daemon = (int) offsetGetter.getOffset("_jvmtiThreadGroupInfo", "is_daemon");
    }

    public final class JvmtiThreadGroupInfoWrapper extends StructWrapper {

        JvmtiThreadGroupInfoWrapper(JniEnv jni, TruffleObject pointer) {
            super(jni, pointer, structSize);
        }

        public StaticObject parent() {
            return getObject(parent);
        }

        public void parent(StaticObject valueToPut) {
            putObject(parent, valueToPut);
        }

        public TruffleObject name() {
            return getPointer(name);
        }

        public void name(TruffleObject valueToPut) {
            putPointer(name, valueToPut);
        }

        public int maxPriority() {
            return getInt(max_priority);
        }

        public void maxPriority(int valueToPut) {
            putInt(max_priority, valueToPut);
        }

        public boolean isDaemon() {
            return getBoolean(is_daemon);
        }

        public void isDaemon(boolean valueToPut) {
            putBoolean(is_daemon, valueToPut);
        }

    }

    @Override
    public JvmtiThreadGroupInfoWrapper wrap(JniEnv jni, TruffleObject pointer) {
        return new JvmtiThreadGroupInfoWrapper(jni, pointer);
    }

}

