/*
 * Copyright (c) 2018, 2023, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * The Universal Permissive License (UPL), Version 1.0
 *
 * Subject to the condition set forth below, permission is hereby granted to any
 * person obtaining a copy of this software, associated documentation and/or
 * data (collectively the "Software"), free of charge and under any and all
 * copyright rights in the Software, and any and all patent rights owned or
 * freely licensable by each licensor hereunder covering either (i) the
 * unmodified Software as contributed to or provided by such licensor, or (ii)
 * the Larger Works (as defined below), to deal in both
 *
 * (a) the Software, and
 *
 * (b) any piece of software and/or hardware listed in the lrgrwrks.txt file if
 * one is included with the Software each a "Larger Work" to which the Software
 * is contributed by such licensors),
 *
 * without restriction, including without limitation the rights to copy, create
 * derivative works of, display, perform, and distribute the Software and make,
 * use, sell, offer for sale, import, export, have made, and have sold the
 * Software and the Larger Work(s), and to sublicense the foregoing rights on
 * either these or other terms.
 *
 * This license is subject to the following condition:
 *
 * The above copyright notice and either this complete permission notice or at a
 * minimum a reference to the UPL must be included in all copies or substantial
 * portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package com.oracle.truffle.js.nodes.intl;

import com.oracle.truffle.api.CompilerDirectives.TruffleBoundary;
import com.oracle.truffle.api.dsl.Cached;
import com.oracle.truffle.api.dsl.Specialization;
import com.oracle.truffle.api.profiles.InlinedBranchProfile;
import com.oracle.truffle.js.nodes.JavaScriptBaseNode;
import com.oracle.truffle.js.nodes.cast.JSToNumberNode;
import com.oracle.truffle.js.runtime.Errors;
import com.oracle.truffle.js.runtime.JSException;
import com.oracle.truffle.js.runtime.JSRuntime;

public abstract class DefaultNumberOptionNode extends JavaScriptBaseNode {

    protected DefaultNumberOptionNode() {
    }

    public abstract int executeInt(Object value, int minimum, int maximum, int fallback);

    public static DefaultNumberOptionNode create() {
        return DefaultNumberOptionNodeGen.create();
    }

    @Specialization(guards = "!isUndefined(value)")
    public int getOption(Object value, int minimum, int maximum,
                    @SuppressWarnings("unused") int fallback,
                    @Cached JSToNumberNode toNumberNode,
                    @Cached InlinedBranchProfile errorBranch) {
        Number numValue = toNumberNode.executeNumber(value);
        double doubleValue = JSRuntime.doubleValue(numValue);
        if (Double.isNaN(doubleValue) || doubleValue < minimum || maximum < doubleValue) {
            errorBranch.enter(this);
            throw createRangeError(doubleValue, minimum, maximum);
        }
        return (int) doubleValue;
    }

    @TruffleBoundary
    private JSException createRangeError(double value, int minimum, int maximum) throws JSException {
        return Errors.createRangeErrorFormat("invalid value %f found where only values between %d and %d are allowed", this, value, minimum, maximum);
    }

    @Specialization(guards = "isUndefined(value)")
    @SuppressWarnings("unused")
    public int getOptionFromUndefined(Object value, int minimum, int maximum, int fallback) {
        return fallback;
    }

}
