/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.truffle.llvm.parser.coff;

import com.oracle.truffle.api.source.Source;
import com.oracle.truffle.llvm.parser.filereader.ObjectFileReader;
import com.oracle.truffle.llvm.runtime.except.LLVMParserException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import org.graalvm.polyglot.io.ByteSequence;

public final class CoffFile {
    private final Source source;
    private final ByteSequence bytes;
    private final ImageFileHeader header;
    private final ImageOptionHeader optionHeader;
    private final ImageSectionHeader[] sections;
    private final List<ImageSymbol> symbols;

    private CoffFile(Source source, ByteSequence bytes, ImageFileHeader header, ImageOptionHeader optionHeader) {
        this.source = source;
        this.bytes = bytes;
        this.header = header;
        this.optionHeader = optionHeader;
        this.sections = new ImageSectionHeader[header.numberOfSections];
        this.symbols = new ArrayList<ImageSymbol>();
    }

    public ImageSectionHeader getSectionByNumber(int number) {
        assert (number > 0);
        return this.sections[number - 1];
    }

    public ImageSectionHeader getSection(String sectionName) {
        for (ImageSectionHeader section : this.sections) {
            if (!section.getName().equals(sectionName)) continue;
            return section;
        }
        return null;
    }

    private void initializeSections(ObjectFileReader reader) {
        reader.setPosition(this.header.firstSection);
        for (int i = 0; i < this.header.numberOfSections; ++i) {
            this.sections[i] = this.createImageSectionHeader(reader);
        }
    }

    private void initializeSymbols(ObjectFileReader reader) {
        ImageSymbol symbol;
        reader.setPosition(this.header.symbolTablePosition);
        for (int i = 0; i < this.header.numberOfSymbols; i += 1 + (symbol.numberOfAuxSymbols > 0 ? symbol.numberOfAuxSymbols : (byte)0)) {
            symbol = ImageSymbol.read(this, reader);
            this.symbols.add(symbol);
        }
    }

    public Iterable<ImageSymbol> getSymbols() {
        return this.symbols;
    }

    ImageSectionHeader lookupOffset(int offset) {
        for (int i = 0; i < this.sections.length; ++i) {
            int virtualOffset = offset - this.sections[i].virtualAddress;
            if (virtualOffset < 0 || virtualOffset > this.sections[i].sizeOfRawData) continue;
            return this.sections[i];
        }
        throw new LLVMParserException(String.format("Invalid virtual address %d.", offset));
    }

    public ImageOptionHeader getOptionHeader() {
        return this.optionHeader;
    }

    public ObjectFileReader getSectionReader(ImageSectionHeader section) {
        ByteSequence sectionBytes = this.bytes.subSequence(section.pointerToRawData, section.pointerToRawData + section.sizeOfRawData);
        ObjectFileReader reader = new ObjectFileReader(sectionBytes, true);
        return reader;
    }

    public ObjectFileReader getReaderAtVirtualAddress(int virtualAddress) {
        ImageSectionHeader section = this.lookupOffset(virtualAddress);
        ObjectFileReader reader = this.getSectionReader(section);
        reader.setPosition(virtualAddress - section.virtualAddress);
        return reader;
    }

    public static CoffFile create(Source source, ByteSequence bytes) {
        return CoffFile.create(source, bytes, new ObjectFileReader(bytes, true));
    }

    static CoffFile create(Source source, ByteSequence bytes, ObjectFileReader reader) {
        ImageFileHeader header = ImageFileHeader.createImageFileHeader(reader);
        ImageOptionHeader optionHeader = header.sizeOfOptionalHeader > 0 ? ImageOptionHeader.create(reader) : null;
        CoffFile coffFile = new CoffFile(source, bytes, header, optionHeader);
        coffFile.initializeSections(reader);
        coffFile.initializeSymbols(reader);
        return coffFile;
    }

    private ImageSectionHeader createImageSectionHeader(ObjectFileReader reader) {
        int startOffset = reader.getPosition();
        ImageSectionHeader section = new ImageSectionHeader(startOffset);
        byte[] nameBytes = new byte[8];
        reader.get(nameBytes);
        section.virtualSize = reader.getInt();
        section.virtualAddress = reader.getInt();
        section.sizeOfRawData = reader.getInt();
        section.pointerToRawData = reader.getInt();
        section.pointerToRelocations = reader.getInt();
        section.pointerToLinenumbers = reader.getInt();
        section.numberOfRelocations = reader.getShort();
        section.numberOfLinenumbers = reader.getShort();
        section.characteristics = reader.getInt();
        int endOffset = reader.getPosition();
        section.name = this.parseName(nameBytes, reader);
        reader.setPosition(endOffset);
        return section;
    }

    private String parseName(byte[] nameBytes, ObjectFileReader reader) {
        String s = CoffFile.nameBytesToString(nameBytes);
        if (s.startsWith("/")) {
            int stringTableOffset = Integer.parseInt(s.substring(1));
            return this.readStringTableOffset(stringTableOffset, reader);
        }
        return s;
    }

    private String readStringTableOffset(int offset, ObjectFileReader reader) {
        int start = this.header.stringTablePosition + offset;
        reader.setPosition(start);
        byte b = reader.getByte();
        while (b != 0) {
            b = reader.getByte();
        }
        int end = reader.getPosition();
        return reader.getString(start, end - 1, StandardCharsets.UTF_8);
    }

    public String readStringAtVirtualOffset(int offset) {
        int virtualOffset;
        ImageSectionHeader section = this.lookupOffset(offset);
        int endOffset = virtualOffset = offset - section.virtualAddress + section.pointerToRawData;
        while (this.bytes.byteAt(endOffset) != 0) {
            ++endOffset;
        }
        return new String(this.bytes.subSequence(virtualOffset, endOffset).toByteArray(), StandardCharsets.UTF_8);
    }

    public static String nameBytesToString(byte[] nameBytes) {
        int length;
        for (length = 0; length < nameBytes.length && nameBytes[length] != 0; ++length) {
        }
        return new String(nameBytes, 0, length, StandardCharsets.UTF_8);
    }

    public Source getSource() {
        return this.source;
    }

    public String toString() {
        return String.format("<CoffFile: %s>", this.source.getPath());
    }

    public static final class ImageFileHeader {
        static final short IMAGE_FILE_MACHINE_AMD64 = -31132;
        static final int NUMBER_OF_SYMBOLS = 8;
        static final int SIZE_OF_OPTIONAL_HEADER_OFFSET = 16;
        static final int IMAGE_SIZEOF_FILE_HEADER = 20;
        static final int SIZE_OF_SYMBOL_TABLE_RECORD = 18;
        private final short numberOfSections;
        private final int firstSection;
        private final int symbolTablePosition;
        private final int numberOfSymbols;
        private final int stringTablePosition;
        private final short sizeOfOptionalHeader;

        private static ImageFileHeader createImageFileHeader(ObjectFileReader reader) {
            int headerStartOffset = reader.getPosition();
            short machine = reader.getShort();
            ImageFileHeader.checkIdent(machine);
            short numberOfSections = reader.getShort();
            reader.getInt();
            int symbolTablePosition = reader.getInt();
            int numberOfSymbols = reader.getInt();
            short sizeOfOptionalHeader = reader.getShort();
            reader.getShort();
            int firstSection = headerStartOffset + 20 + Short.toUnsignedInt(sizeOfOptionalHeader);
            int stringTablePosition = ImageFileHeader.getStringTablePosition(symbolTablePosition, numberOfSymbols);
            return new ImageFileHeader(numberOfSections, firstSection, symbolTablePosition, numberOfSymbols, stringTablePosition, sizeOfOptionalHeader);
        }

        private static int getStringTablePosition(int pointerToSymbolTable, int numberOfSymbols) {
            return pointerToSymbolTable + numberOfSymbols * 18;
        }

        private static void checkIdent(short magic) {
            if (magic != -31132) {
                throw new LLVMParserException("Invalid COFF file!");
            }
        }

        ImageFileHeader(short numberOfSections, int firstSection, int symbolTablePosition, int numberOfSymbols, int stringTablePosition, short sizeOfOptionalHeader) {
            this.numberOfSections = numberOfSections;
            this.firstSection = firstSection;
            this.symbolTablePosition = symbolTablePosition;
            this.numberOfSymbols = numberOfSymbols;
            this.stringTablePosition = stringTablePosition;
            this.sizeOfOptionalHeader = sizeOfOptionalHeader;
        }

        public String toString() {
            return "ImageFileHeader[numberOfSections=" + this.numberOfSections + "]";
        }
    }

    public static abstract class ImageOptionHeader {
        byte majorLinkerVersion;
        byte minorLinkerVersion;
        int sizeOfCode;
        int sizeOfInitializedData;
        int sizeOfUninitializedData;
        int addressOfEntryPoint;
        int baseOfCode;

        private static ImageOptionHeader create(ObjectFileReader reader) {
            short magic = reader.getShort();
            ImageOptionNT64Header header = switch (magic) {
                case 523 -> new ImageOptionNT64Header();
                default -> throw new LLVMParserException(String.format("Unsupported coff optional header magic number 0x%x.", magic));
            };
            header.readStandard(reader);
            ((ImageOptionHeader)header).readExtended(reader);
            return header;
        }

        protected void readStandard(ObjectFileReader reader) {
            this.majorLinkerVersion = reader.getByte();
            this.minorLinkerVersion = reader.getByte();
            this.sizeOfCode = reader.getInt();
            this.sizeOfInitializedData = reader.getInt();
            this.sizeOfUninitializedData = reader.getInt();
            this.addressOfEntryPoint = reader.getInt();
            this.baseOfCode = reader.getInt();
        }

        protected abstract void readExtended(ObjectFileReader var1);

        private ImageOptionHeader() {
        }
    }

    public final class ImageSectionHeader {
        private static final int IMAGE_SIZEOF_SHORT_NAME = 8;
        final int startOffset;
        int virtualSize;
        int virtualAddress;
        String name;
        int sizeOfRawData;
        int pointerToRawData;
        int pointerToRelocations;
        int pointerToLinenumbers;
        short numberOfRelocations;
        short numberOfLinenumbers;
        int characteristics;

        private ImageSectionHeader(int startOffset) {
            this.startOffset = startOffset;
        }

        public String getName() {
            return this.name;
        }

        public String toString() {
            return "ImageSectionHeader[" + this.name + ", virtualSize=" + this.virtualAddress + ", virtualAddress=" + this.virtualAddress + ", sizeOfRawData=" + this.sizeOfRawData + ", pointerToRawData=" + this.pointerToRawData + "]";
        }

        public ByteSequence getData() {
            return CoffFile.this.bytes.subSequence(this.pointerToRawData, this.pointerToRawData + this.sizeOfRawData);
        }
    }

    public static final class ImageSymbol {
        private static final int IMAGE_SYMBOL_SIZE = 18;
        private static final int AUXILIARY_SYMBOL_SIZE = 18;
        private static final int TYPE_FUNCTION_SYMBOL = 32;
        private static final int IMAGE_SYM_CLASS_EXTERNAL = 2;
        private static final int IMAGE_SYM_CLASS_EXTERNAL_DEF = 5;
        String name;
        int value;
        short sectionNumber;
        short type;
        byte storageClass;
        byte numberOfAuxSymbols;

        private static ImageSymbol read(CoffFile file, ObjectFileReader reader) {
            int start = reader.getPosition();
            ImageSymbol symbol = new ImageSymbol();
            int shortName = reader.getInt();
            int nameOffset = reader.getInt();
            symbol.value = reader.getInt();
            symbol.sectionNumber = reader.getShort();
            symbol.type = reader.getShort();
            symbol.storageClass = reader.getByte();
            symbol.numberOfAuxSymbols = reader.getByte();
            assert (reader.getPosition() - start == 18);
            if (shortName == 0) {
                symbol.name = file.readStringTableOffset(nameOffset, reader);
            } else {
                byte[] bytes = new byte[8];
                ByteBuffer.wrap(bytes).order(ByteOrder.nativeOrder()).putInt(shortName).putInt(nameOffset);
                symbol.name = CoffFile.nameBytesToString(bytes);
            }
            reader.setPosition(start + 18 + 18 * symbol.numberOfAuxSymbols);
            return symbol;
        }

        public boolean hasValidSectionNumber() {
            return this.sectionNumber > 0;
        }

        public boolean isFunction() {
            return this.type == 32;
        }

        public boolean isDefinedExternally() {
            return this.storageClass == 5;
        }

        public boolean isExternal() {
            return this.storageClass == 2;
        }
    }

    public static final class ImageOptionNT64Header
    extends ImageOptionHeader {
        protected static final short IMAGE_NT_OPTIONAL_HDR64_MAGIC = 523;
        long imageBase;
        int sectionAlignment;
        int fileAlignment;
        short majorOperatingSystemVersion;
        short minorOperatingSystemVersion;
        short majorImageVersion;
        short minorImageVersion;
        short majorSubsystemVersion;
        short minorSubsystemVersion;
        int win32VersionValue;
        int sizeOfImage;
        int sizeOfHeaders;
        int checksum;
        short subsystem;
        short dllCharacteristics;
        long sizeOfStackReserve;
        long sizeOfStackCommit;
        long sizeOfHeadReserve;
        long sizeOfHeadCommit;
        int loaderFlags;
        ImageDataDirectory[] directories;

        @Override
        protected void readExtended(ObjectFileReader reader) {
            this.imageBase = reader.getLong();
            this.sectionAlignment = reader.getInt();
            this.fileAlignment = reader.getInt();
            this.majorOperatingSystemVersion = reader.getShort();
            this.minorOperatingSystemVersion = reader.getShort();
            this.majorImageVersion = reader.getShort();
            this.minorImageVersion = reader.getShort();
            this.majorSubsystemVersion = reader.getShort();
            this.minorSubsystemVersion = reader.getShort();
            this.win32VersionValue = reader.getInt();
            this.sizeOfImage = reader.getInt();
            this.sizeOfHeaders = reader.getInt();
            this.checksum = reader.getInt();
            this.subsystem = reader.getShort();
            this.dllCharacteristics = reader.getShort();
            this.sizeOfStackReserve = reader.getLong();
            this.sizeOfStackCommit = reader.getLong();
            this.sizeOfHeadReserve = reader.getLong();
            this.sizeOfHeadCommit = reader.getLong();
            this.loaderFlags = reader.getInt();
            int numberOfRvaAndSizes = reader.getInt();
            this.directories = new ImageDataDirectory[numberOfRvaAndSizes];
            for (int i = 0; i < numberOfRvaAndSizes; ++i) {
                this.directories[i] = ImageDataDirectory.createImageDataDirectory(reader);
            }
        }

        private ImageOptionNT64Header() {
        }

        public ImageDataDirectory getDirectory(ImageDataIndex index) {
            return this.directories.length > index.getIndex() ? this.directories[index.getIndex()] : null;
        }

        public ImageDataDirectory getImportAddressTableDirectory() {
            return this.getDirectory(ImageDataIndex.ImportAddressTable);
        }

        public static enum ImageDataIndex {
            ExportTable(0),
            ImportTable(1),
            ImportAddressTable(12);

            private final int index;

            private ImageDataIndex(int index) {
                this.index = index;
            }

            public int getIndex() {
                return this.index;
            }
        }
    }

    public static final class ImageDataDirectory {
        private final int virtualAddress;
        private final int size;

        private static ImageDataDirectory createImageDataDirectory(ObjectFileReader reader) {
            int virtualAddress = reader.getInt();
            int size = reader.getInt();
            return new ImageDataDirectory(virtualAddress, size);
        }

        public ImageDataDirectory(int virtualAddress, int size) {
            this.virtualAddress = virtualAddress;
            this.size = size;
        }

        public int getVirtualAddress() {
            return this.virtualAddress;
        }

        public int getSize() {
            return this.size;
        }
    }
}

